#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import BasicCli
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.DhcpServerCliSubmodeLib import (
   DhcpServerClientClassMatchMacAddr,
   DhcpServerClientClassMatchVendorId,
   DhcpServerClientClassMatchAristaInfoOption,
   DhcpServerClientClassMatchInfoOptionHexOrStr,
   DhcpServerClientClassMatchStringArbitraryOption,
   DhcpServerClientClassMatchHexArbitraryOption,
   DhcpServerClientClassMatchIpAddrArbitraryOption,
   DhcpServerClientClassMatchDuid,
   DhcpServerClientClassMatchVendorClass,
   DhcpServerClientClassMatchAristaRemoteId,
   DhcpServerClientClassMatchRemoteIdHexOrStr,
   )
from CliMode.DhcpServer import (
   DhcpServerClientClassMatchBaseMode,
   DhcpServerSubnetClientClassMatchBaseMode,
   DhcpServerRangeClientClassMatchBaseMode,
   )
from EosDhcpServerLib import (
   featureArbitraryOptionMatch,
   )

# CliDynamicPlugin nested match mode
nestedMatchSubmodes = CliDynamicPlugin( "DhcpServerClientNestedMatchMode" )

class DhcpServerClientClassV4MatchMode( DhcpServerClientClassMatchBaseMode,
                                        BasicCli.ConfigModeBase ):
   name = "DHCP Server Client Class V4 Match"

   def __init__( self, parent, session, clientClassName, matchAny ):
      self.vrf = parent.vrf
      param = ( clientClassName, matchAny, self.vrf, 'ipv4' )
      DhcpServerClientClassMatchBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig
      self.clientClassConfig = parent.dhcpServerConfig.clientClassConfigIpv4[
            clientClassName ]

class DhcpServerClientClassV6MatchMode( DhcpServerClientClassMatchBaseMode,
                                        BasicCli.ConfigModeBase ):
   name = "DHCP Server Client Class V6 Match"

   def __init__( self, parent, session, clientClassName, matchAny ):
      self.vrf = parent.vrf
      param = ( clientClassName, matchAny, self.vrf, 'ipv6' )
      DhcpServerClientClassMatchBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig
      self.clientClassConfig = parent.dhcpServerConfig.clientClassConfigIpv6[
            clientClassName ]

# subnet's mode client class mode
class DhcpServerSubnetClientClassV4MatchMode(
      DhcpServerSubnetClientClassMatchBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server Client Class V4 Match"

   def __init__( self, parent, session, subnetPrefix, clientClassName, matchAny ):
      self.vrf = parent.vrf
      param = ( subnetPrefix, clientClassName, matchAny, self.vrf, 'ipv4' )
      DhcpServerSubnetClientClassMatchBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      subnetConfig = parent.dhcpServerConfig.subnetConfigIpv4[ subnetPrefix ]
      self.dhcpServerConfig = parent.dhcpServerConfig
      self.clientClassConfig = subnetConfig.clientClassConfig[ clientClassName ]

class DhcpServerSubnetClientClassV6MatchMode(
      DhcpServerSubnetClientClassMatchBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server Client Class V6 Match"

   def __init__( self, parent, session, subnetPrefix, clientClassName, matchAny ):
      self.vrf = parent.vrf
      param = ( subnetPrefix, clientClassName, matchAny, self.vrf, 'ipv6' )
      DhcpServerSubnetClientClassMatchBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      subnetConfig = parent.dhcpServerConfig.subnetConfigIpv6[ subnetPrefix ]
      self.dhcpServerConfig = parent.dhcpServerConfig
      self.clientClassConfig = subnetConfig.clientClassConfig[ clientClassName ]

class DhcpServerRangeClientClassV4MatchMode(
      DhcpServerRangeClientClassMatchBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server Range Client Class V4 Match"

   def __init__( self, parent, session, subnetConfig,
                 rangeConfig, clientClassName, matchAny ):
      self.vrf = parent.vrf
      param = ( subnetConfig, rangeConfig, clientClassName, matchAny,
                self.vrf, 'ipv4' )
      DhcpServerRangeClientClassMatchBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig
      self.clientClassConfig = rangeConfig.clientClassConfig[ clientClassName ]

class DhcpServerRangeClientClassV6MatchMode(
      DhcpServerRangeClientClassMatchBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Server Range Client Class V6 Match"

   def __init__( self, parent, session, subnetConfig,
                 rangeConfig, clientClassName, matchAny ):
      self.vrf = parent.vrf
      param = ( subnetConfig, rangeConfig, clientClassName, matchAny,
                self.vrf, 'ipv6' )
      DhcpServerRangeClientClassMatchBaseMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dhcpServerConfig = parent.dhcpServerConfig
      self.clientClassConfig = rangeConfig.clientClassConfig[ clientClassName ]

class EnterDhcpServerNestedMatchModeBase( CliCommand.CliCommandClass ):
   syntax = 'NESTED_MATCH'
   noOrDefaultSyntax = syntax

   @staticmethod
   def adapterBase( mode, args, ipv4, v4MatchMode, v6MatchMode ):
      if ipv4:
         args[ 'af' ] = 'ipv4'
         args[ 'nestedMatchMode' ] = v4MatchMode
      else:
         args[ 'af' ] = 'ipv6'
         args[ 'nestedMatchMode' ] = v6MatchMode

class NestedMatchExpr( CliCommand.CliExpression ):
   expression = 'match [ any ] NAME'
   data = {
      'match': 'Specify nested match criteria for the client class',
      'any': "Requires clients' packets to match any of the match criteria to be "
             "included in this client class",
      'NAME': CliMatcher.PatternMatcher( pattern=r'^(?!any$)[a-zA-Z0-9\._-]+',
                                         helpname='NAME',
                                         helpdesc='Name of the nested match criteria'
                                         )
      }

class EnterDhcpServerClientClassNestedMatchMode(
      EnterDhcpServerNestedMatchModeBase ):
   data = {
         'NESTED_MATCH': NestedMatchExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerClientClassV4MatchMode )
      EnterDhcpServerNestedMatchModeBase.adapterBase(
            mode, args, ipv4,
            nestedMatchSubmodes.DhcpServerClientClassV4NestedMatchMode,
            nestedMatchSubmodes.DhcpServerClientClassV6NestedMatchMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerClientClassNestedMatchMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerClientClassNestedMatchMode" )

class EnterDhcpServerSubnetClientClassNestedMatchMode(
      EnterDhcpServerNestedMatchModeBase ):
   data = {
         'NESTED_MATCH': NestedMatchExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerSubnetClientClassV4MatchMode )
      EnterDhcpServerNestedMatchModeBase.adapterBase(
            mode, args, ipv4,
            nestedMatchSubmodes.DhcpServerSubnetClientClassV4NestedMatchMode,
            nestedMatchSubmodes.DhcpServerSubnetClientClassV6NestedMatchMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerSubnetClientClassNestedMatchMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerSubnetClientClassNestedMatchMode" )

class EnterDhcpServerRangeClientClassNestedMatchMode(
      EnterDhcpServerNestedMatchModeBase ):
   data = {
         'NESTED_MATCH': NestedMatchExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ipv4 = isinstance( mode, DhcpServerRangeClientClassV4MatchMode )
      EnterDhcpServerNestedMatchModeBase.adapterBase(
            mode, args, ipv4,
            nestedMatchSubmodes.DhcpServerRangeClientClassV4NestedMatchMode,
            nestedMatchSubmodes.DhcpServerRangeClientClassV6NestedMatchMode )

   handler = "DhcpServerCliHandler.doEnterDhcpServerRangeClientClassNestedMatchMode"

   noOrDefaultHandler = ( "DhcpServerCliHandler."
                          "noEnterDhcpServerRangeClientClassNestedMatchMode" )


# Add Command
DhcpServerClientClassV4MatchMode.addCommandClass(
   EnterDhcpServerClientClassNestedMatchMode )
DhcpServerClientClassV6MatchMode.addCommandClass(
   EnterDhcpServerClientClassNestedMatchMode )

DhcpServerClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchMacAddr )
DhcpServerClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchVendorId )
DhcpServerClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchAristaInfoOption )
DhcpServerClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchInfoOptionHexOrStr )

DhcpServerSubnetClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchMacAddr )
DhcpServerSubnetClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchVendorId )
DhcpServerSubnetClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchAristaInfoOption )
DhcpServerSubnetClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchInfoOptionHexOrStr )

DhcpServerRangeClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchMacAddr )
DhcpServerRangeClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchVendorId )
DhcpServerRangeClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchAristaInfoOption )
DhcpServerRangeClientClassV4MatchMode.addCommandClass(
   DhcpServerClientClassMatchInfoOptionHexOrStr )

if featureArbitraryOptionMatch():
   DhcpServerClientClassV4MatchMode.addCommandClass(
         DhcpServerClientClassMatchStringArbitraryOption )
   DhcpServerClientClassV4MatchMode.addCommandClass(
         DhcpServerClientClassMatchHexArbitraryOption )
   DhcpServerClientClassV4MatchMode.addCommandClass(
         DhcpServerClientClassMatchIpAddrArbitraryOption )

# V6 match mode
DhcpServerClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchMacAddr )
DhcpServerClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchDuid )
DhcpServerClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchVendorClass )
DhcpServerClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchAristaRemoteId )
DhcpServerClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchRemoteIdHexOrStr )

DhcpServerSubnetClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchMacAddr )
DhcpServerSubnetClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchDuid )
DhcpServerSubnetClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchVendorClass )
DhcpServerSubnetClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchAristaRemoteId )
DhcpServerSubnetClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchRemoteIdHexOrStr )

DhcpServerRangeClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchMacAddr )
DhcpServerRangeClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchDuid )
DhcpServerRangeClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchVendorClass )
DhcpServerRangeClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchAristaRemoteId )
DhcpServerRangeClientClassV6MatchMode.addCommandClass(
   DhcpServerClientClassMatchRemoteIdHexOrStr )

# Subnet Match Mode
DhcpServerSubnetClientClassV4MatchMode.addCommandClass(
   EnterDhcpServerSubnetClientClassNestedMatchMode )
DhcpServerSubnetClientClassV6MatchMode.addCommandClass(
   EnterDhcpServerSubnetClientClassNestedMatchMode )

# Range Match Mode
DhcpServerRangeClientClassV4MatchMode.addCommandClass(
   EnterDhcpServerRangeClientClassNestedMatchMode )
DhcpServerRangeClientClassV6MatchMode.addCommandClass(
   EnterDhcpServerRangeClientClassNestedMatchMode )
