# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliDynamicSymbol
from CliPlugin.FlowTrackingCliLib import (
   ftrCapabilities,
   ftrTypeCpuQueue,
)
import SmashLazyMount
import Tac

DmaMonitorModels = CliDynamicSymbol.loadDynamicPlugin( "DmaMonitorModel" )

# See AID/8076 for use of SmashLazyMount
counters = None
countersType = 'DmaMonitor::CpuQueue::Counters'
Counters = Tac.Type( countersType )

def _cpuQueueName( cpuQueueId: int ) -> str:
   """Retrieve the cpu-queue name from the identifier.

      Args:
        cpuQueueId: Idenitifer to look up.

      Returns:
        Name of the queue, or and empty string if not found.
   """
   # The number of queues is small so we are just doing a linear lookup.
   ftrCpuQueueCapabilities = ftrCapabilities[ ftrTypeCpuQueue ]
   monitoredCpuQueueSupported = ftrCpuQueueCapabilities.monitoredCpuQueueSupported
   queueName = next(
         ( y[ 0 ] for
            y in monitoredCpuQueueSupported.items() if y[ 1 ] == cpuQueueId ),
         "" )
   return queueName

def showFlowTrackingCpuQueueCounters( mode, args ):
   countersModel = DmaMonitorModels.CountersModel()
   v = counters.cpuQueueCounters.values()
   s = sorted( v, key=lambda c: c.key )
   for cpuQueueCounter in s:
      cpuQueueCountersModel = DmaMonitorModels.CpuQueueCountersModel()
      cpuQueueCountersModel.cpuQueueId = cpuQueueCounter.key
      cpuQueueCountersModel.cpuQueueName = (
            _cpuQueueName( cpuQueueCounter.key ) )
      cpuQueueCountersModel.sampledPackets = cpuQueueCounter.sampledPackets
      cpuQueueCountersModel.congestionEvents = cpuQueueCounter.congestionEvents
      countersModel.cpuQueueCounters.append( cpuQueueCountersModel )
   return countersModel

def Plugin( entityManager ):
   global counters
   # Unmount the entity if it has not been accessed in 10 minutes
   # and there are no more references to it.
   countersPath = Counters.mountPath()
   counters = SmashLazyMount.mount( entityManager, countersPath,
                  countersType, SmashLazyMount.mountInfo( 'reader' ),
                  autoUnmount=True, unmountTimeout=600 )
