#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Enum, Model, Str
import CliPlugin.FruModel as FruModel
from CliPlugin.DpllCommon import parseClockName

def pickNth( o, idx ):
   return o[ idx ] if isinstance( o, tuple ) else o

states = [
   "disabled",
   "initializing",
   "failed",
   "active",
   ( "freeRunning", "free-running" )
]
statesJson = [ pickNth( s, 0 ) for s in states ]
jsonToCli = { pickNth( s, 0 ): pickNth( s, 1 ) for s in states }

class ClockModel( Model ):
   model = Str( help="PLL manufacturer and model" )
   description = Str( help="PLL description" )
   role = Enum( help="PLL function", values=( "timeSync", "phy", "servo",
                                              "timeSyncFreeRun", "system" ) )
   state = Enum( values=statesJson,
                 help="Current state of the PLL" )

class PlatformModel( Model ):
   clocks = Dict( keyType=str, valueType=ClockModel,
                  help="Clock name to clock information" )

   def render( self ):
      fmt = "%-20s %-30s %-25s %15s %15s"
      print( fmt % ( "Name", "Model", "Description", "State", "Role" ) )
      FruModel.printDashes( fmt )
      for clockName in sorted( self.clocks.keys(), key=parseClockName ):
         clockInfo = self.clocks[ clockName ]
         print( fmt % ( clockName, clockInfo.model, clockInfo.description,
                       jsonToCli[ clockInfo.state ], clockInfo.role ) )
