# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import Str, Model, Enum, Int, Submodel, Dict, Bool
from CliModel import GeneratorDict, GeneratorList
from ArnetModel import IpGenericPrefix
from CliPlugin.DynamicPrefixListHelper import getDplConfigParams
import Tac

BgpPeerState = Tac.Type( 'Routing::Bgp::BgpPeerState' )

DPL_STATE_MAP = {
      1 : 'inactive',
      2 : 'active',
      3 : 'notApplicableToRoutingInstance',
}

DPL_STATE_RENDER_MAP = {
      'active' : 'Active',
      'inactive' : 'Inactive',
      'notApplicableToRoutingInstance' : 'Not applicable to routing instance'
}

def printDynamicPrefixListLegendInformation():
   print( '* - Not all contributing routes are listed' )

class DynamicPrefixListContributingRoutes( Model ):
   prefix = IpGenericPrefix( help='Route Prefix' )

   def processData( self, data ):
      return data

   def render( self ):
      print( '       %s' % ( self.prefix ) )

class PeerSpecification( Model ):
   sessionState = Enum( values=BgpPeerState.attributes,
                        help='The state of BGP session' )
   originatePeerAddress = Bool( help='Originate peer-address' )

   def printModel( self ):
      print( '      Session State:', self.sessionState )
      print( '      Originate peer-address:', self.originatePeerAddress )

class BgpSession( Model ):
   peers = Dict( keyType=str, valueType=PeerSpecification,
                 help='Peer specifications' )

   def render( self ):
      for peer in self.peers:
         print( '    BGP session:', peer )
         self.peers[ peer ].printModel()

class DynamicPrefixListModel( Model ):
   _undefinedPolicy = Bool( help="No policy found for given policy name",
                            optional=True )
   matchRouteMap = Str( help='Match route map name', optional=True )
   matchRcfFunction = Str( help='Match RCF function name', optional=True )
   ipv4PrefixList = Str( help='IPv4 prefix list name', optional=True )
   ipv6PrefixList = Str( help='IPv6 prefix list name', optional=True )
   state = Enum( values=list( DPL_STATE_MAP.values() ),
                 help='State of the dynamic prefix list' )
   numContributingRoutes = Int( help='Number of contributing routes', optional=True )
   contributingRoutes = GeneratorList( 
                           valueType=DynamicPrefixListContributingRoutes,
                           help='Capped list of contributing routes (a sample of '\
                                'contributing routes), current cap is 10.',
                           optional=True )
   bgpSession = Submodel(
         valueType=BgpSession,
         help='BGP session activated dynamic prefix list information',
         optional=True )

   def getKey( self, data ):
      assert 'name' in data
      return data[ 'name' ]

   def processData( self, data ):
      dplState = data.pop( 'dpl_state', None )
      if dplState is not None:
         self.state = DPL_STATE_MAP[ dplState ]
      if 'ipv6PrefixList' in data:
         # Remove '.ipv6.' from prefix list name
         self.ipv6PrefixList = data.pop( 'ipv6PrefixList' )[ 6: ]

      # if state == notApplicableToRoutingInstance, that means the
      # rib instance does not have information about the dpl. So
      # populate the available fields from the config directly
      dplName = data.get( 'name', None )
      if self.state == 'notApplicableToRoutingInstance':
         matchMap, matchRcf, ipv4PrefixList, ipv6PrefixList = \
               getDplConfigParams( dplName )
         if matchMap:
            self.matchRouteMap = matchMap
         if matchRcf:
            self.matchRcfFunction = matchRcf
         if ipv4PrefixList:
            self.ipv4PrefixList = ipv4PrefixList
         if ipv6PrefixList:
            self.ipv6PrefixList = ipv6PrefixList

      return data

   def renderEntry( self, dynamicPrefixListName ):
      print( '  Dynamic prefix list: %s' % ( dynamicPrefixListName ) )
      if not self.bgpSession:
         if self.matchRouteMap:
            print( '    Match route map: %s' % ( self.matchRouteMap ) )
         if self.matchRcfFunction:
            print( '    Match RCF function: %s' % ( self.matchRcfFunction ) )
         if self.ipv4PrefixList:
            print( '    IPv4 prefix list: %s' % ( self.ipv4PrefixList ) )
         if self.ipv6PrefixList:
            print( '    IPv6 prefix list: %s' % ( self.ipv6PrefixList ) )
      else:
         self.bgpSession.render()
      
      stateStr = DPL_STATE_RENDER_MAP[ self.state ]
      if self._undefinedPolicy:
         stateStr = 'No RCF function or route map configured for matching'

      print( '    State: %s' % ( stateStr ) )
      if self.state == 'active':
         notAllListed = '*' if self.numContributingRoutes > 10 else ''
         # pylint: disable-next=bad-string-format-type
         print( '    %sContributing routes: %d' % ( notAllListed,
                                                    self.numContributingRoutes ) )
         for route in self.contributingRoutes:
            route.render()

class DynamicPrefixLists( Model ):
   _vrf = Str( help="Vrf Name" )
   dynamicPrefixLists = GeneratorDict( valueType=DynamicPrefixListModel,
      help='Table of dynamic prefix lists by name' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      print( 'VRF: %s' % ( self._vrf ) )
      for dynamicPrefixListName, dynamicPrefixList in self.dynamicPrefixLists:
         dynamicPrefixList.renderEntry( dynamicPrefixListName )

