# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IntfId
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.LauncherDaemonCli import DaemonConfigMode
from CliPlugin.EosSdkRpcCli import (
   controlConfig,
   serverConfigDir,
   MgmtEosSdkRpcConfigMode,
   EosSdkRpcTransportConfigMode,
)
import ConfigMount
from EosSdkRpcAgentLib import RpcConfigConstants
from IpLibConsts import DEFAULT_VRF
import LauncherLib

EosSdkRpcCliLib = CliDynamicPlugin( "EosSdkRpcCliLib" )

agentConfigCliDir = None
daemonConfigDir = None

def MgmtApiEosSdkRpcCmd_handler( mode, args ):
   childMode = mode.childMode( MgmtEosSdkRpcConfigMode )
   mode.session_.gotoChildMode( childMode )

def MgmtApiEosSdkRpcCmd_noOrDefaultHandler( mode, args ):
   """Resets EosSdkRpc configuration to default."""
   for name in serverConfigDir:
      TransportModeCmd_noHandler( mode, { 'TRANSPORT_NAME': name } )
   # shutdown EosSdkRpcControlAgent
   controlConfig.enabled = False

def TransportModeCmd_handler( mode, args ):
   """Handler for 'transport grpc SERVER_NAME'"""
   name = args[ 'TRANSPORT_NAME' ]
   childMode = mode.childMode( EosSdkRpcTransportConfigMode, name=name )
   mode.session_.gotoChildMode( childMode )

def TransportModeCmd_noHandler( mode, args ):
   """Handler for 'no transport grpc SERVER_NAME'"""
   name = args[ 'TRANSPORT_NAME' ]

   if serverConfigDir.get( name ):
      if agentConfigCliDir.agent.get( name ):
         del agentConfigCliDir.agent[ name ]
      serverConfigDir.deleteEntity( name )
      daemonConfigDir.deleteEntity( name )
   # No EosSdkRpcAgents left to monitor, shutdown control agent
   if not serverConfigDir.values():
      controlConfig.enabled = False

def AccountingRequests_handler( mode, args ):
   mode.serverConfig.accounting = True

def AccountingRequests_noOrDefaultHandler( mode, args ):
   mode.serverConfig.accounting = False

def DisableCmd_handler( mode, args ):
   """Handler for 'disabled'"""
   mode.serverConfig.enabled = False
   DaemonConfigMode.shutdownIs( mode, True )

def checkIfEndpointConfigured( mode ):
   return ( mode.serverConfig.localInterface.intfId !=
            RpcConfigConstants.intfDefault or
            mode.serverConfig.localhostLoopback or
            mode.serverConfig.unixEndpointAddress )

def DisableCmd_noOrDefaultHandler( mode, args ):
   """Handler for 'no disabled'"""
   if not checkIfEndpointConfigured( mode ):
      mode.addError( "No endpoint configured." )
      return
   if not ( mode.serverConfig.servicesEnabled or
            mode.serverConfig.enableAllServices ):
      mode.addError( "No services configured." )
      return
   mode.serverConfig.enabled = True
   DaemonConfigMode.shutdownIs( mode, False )

def LocalIntfPortCmd_handler( mode, args ):
   """Handler for 'local interface INTF [ port PORT ]'"""
   portArg = args.get( 'PORT', RpcConfigConstants.portDefault )
   intf = args[ 'INTERFACE' ]
   tacIntf = IntfId( intf )
   mode.serverConfig.localInterface.intfId = tacIntf
   mode.serverConfig.localInterface.port = portArg

def LocalIntfPortCmd_noOrDefaultHandler( mode, args ):
   """Handler for '( no | default ) local interface INTF [ port PORT ]'"""
   port = args.get( "PORT" )
   intfArg = args.get( 'INTERFACE' )
   defaultIntf = RpcConfigConstants.intfDefault
   defaultPort = RpcConfigConstants.portDefault
   if mode.serverConfig.localInterface.intfId == IntfId( intfArg ).stringValue:
      if mode.serverConfig.localInterface.port == port:
         mode.serverConfig.localInterface.port = defaultPort
      mode.serverConfig.localInterface.intfId = defaultIntf

def LocalhostLoopbackCmd_handler( mode, args ):
   """Handler for 'localhost loopback [ port PORT ] [ vrf VRF ]'"""
   portArg = args.get( 'PORT', RpcConfigConstants.portDefault )
   vrfArg = args.get( 'VRF', DEFAULT_VRF )
   mode.serverConfig.localhostLoopback = ()
   mode.serverConfig.localhostLoopback.port = portArg
   mode.serverConfig.localhostLoopback.vrfName = vrfArg

def LocalhostLoopbackCmd_noOrDefaultHandler( mode, args ):
   """Handler for '( no | default ) localhost loopback [ port PORT ] [ vrf VRF ]'"""
   mode.serverConfig.localhostLoopback = None

def LocalhostUnixSocketCmd_handler( mode, args ):
   """Handler for 'localhost unix-socket FILE'"""
   socketPath = args.get( 'FILE' )
   mode.serverConfig.unixEndpointAddress = ( socketPath.url, )

def LocalhostUnixSocketCmd_noOrDefaultHandler( mode, args ):
   """Handler for '( no | default ) localhost unix-socket'"""
   mode.serverConfig.unixEndpointAddress = None

def EnableServicesCmd_handler( mode, args ):
   """Handler for 'service ( all | { SERVICES } )'"""
   enableAll = 'all' in args
   enableServices = args.get( 'SERVICES' )
   if enableAll:
      mode.serverConfig.enableAllServices = True
   elif enableServices:
      for service in enableServices:
         mode.serverConfig.servicesEnabled.add( service )

def EnableServicesCmd_noHandler( mode, args ):
   """Handler for 'no service { SERVICES }'"""
   disableAll = 'all' in args
   disableServices = args.get( 'SERVICES' )
   if disableAll:
      mode.serverConfig.enableAllServices = False
   elif disableServices:
      for service in disableServices:
         del mode.serverConfig.servicesEnabled[ service ]
   else:
      mode.serverConfig.enableAllServices = False
      mode.serverConfig.servicesEnabled.clear()

def EnableServicesCmd_defaultHandler( mode, args ):
   """Handler for 'default service [ all ]'"""
   mode.serverConfig.enableAllServices = False
   mode.serverConfig.servicesEnabled.clear()

def IpAccessGroupAclNameCmd_handler( mode, args ):
   """Handler for 'ip access-group ACLNAME [ in ]'"""
   if ( mode.serverConfig.serviceAcl
        or not EosSdkRpcCliLib.existsOtherTransportSameVrfWithServiceAcl(
           mode, serverConfigDir ) ):
      mode.serverConfig.serviceAcl = args[ 'ACLNAME' ]
      EosSdkRpcCliLib.setServiceAcl( mode, 'eosSdkRpc', mode.serverConfig )

def IpAccessGroupAclNameCmd_noOrDefaultHandler( mode, args ):
   """Handler for '( no | default ) ip access-group ACLNAME [ in ]'"""
   EosSdkRpcCliLib.noServiceAcl( mode, 'eosSdkRpc', mode.serverConfig )
   mode.serverConfig.serviceAcl = ''

def VrfCmd_handler( mode, args ):
   """Handler for 'vrf [ VRFNAME ]'"""
   vrfName = args.get( 'VRFNAME', DEFAULT_VRF )
   if vrfName == mode.serverConfig.vrfName:
      return
   if mode.serverConfig.serviceAcl:
      if EosSdkRpcCliLib.existsOtherTransportSameVrfWithServiceAcl(
            mode, serverConfigDir, vrfName ):
         return
      EosSdkRpcCliLib.noServiceAcl( mode, 'eosSdkRpc', mode.serverConfig )
   mode.serverConfig.vrfName = vrfName
   if mode.serverConfig.serviceAcl:
      EosSdkRpcCliLib.setServiceAcl( mode, 'eosSdkRpc', mode.serverConfig )

def SslProfileProfileNameCmd_handler( mode, args ):
   """Handler for 'ssl profile PROFILENAME'"""
   profileName = args[ 'PROFILENAME' ]
   mode.serverConfig.sslProfile = profileName

def SslProfileProfileNameCmd_noOrDefaultHandler( mode, args ):
   """Handler for '( no | default ) ssl profile PROFILENAME'"""
   mode.serverConfig.sslProfile = ''

def UserAuthCmd_handler( mode, args ):
   """Handler for 'metadata username authentication channel secure'"""
   mode.serverConfig.usernameAuthSecure = True

def UserAuthCmd_noOrDefaultHandler( mode, args ):
   """
   Handler for '( no | default ) metadata username authentication channel secure'
   """
   mode.serverConfig.usernameAuthSecure = False

def Plugin( entityManager ):
   global agentConfigCliDir, daemonConfigDir

   agentConfigCliDir = ConfigMount.mount( entityManager,
                                          LauncherLib.agentConfigCliDirPath,
                                          "Launcher::AgentConfigDir",
                                          "wi" )
   daemonConfigDir = ConfigMount.mount( entityManager,
                                        'daemon/agent/config',
                                        'Tac::Dir', 'wi' )
