# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from ArnetModel import IpGenericAddress
from CliModel import (
      Bool,
      Dict,
      Enum,
      Int,
      Model,
      Str,
      Submodel )
from IntfModels import Interface
from Toggles import EosSdkRpcToggleLib
from TypeFuture import TacLazyType

UserAuthType = TacLazyType( "EosSdkRpc::Server::UsernameAuthentication" )
ServerConfigErrors = TacLazyType( "EosSdkRpc::Server::ConfigError" )
ConfigErrorApi = TacLazyType( "EosSdkRpc::Server::ConfigErrorApi" )
ServerSecurity = TacLazyType( "EosSdkRpc::Server::ServerSecurity" )

def metadataAuthEnumToStr( value ):
   if value == UserAuthType.disabled:
      return 'disabled'
   elif value == UserAuthType.secureChannelOnly:
      return "secure channel"
   assert False, "unknown value"

def printLineItem( label, content ):
   print( f'{label}: {content}' )

def printEnabledService( serviceName, isEnabled ):
   if isEnabled:
      print( f'   {serviceName}' )

def printConfigError( error ):
   print( f'   {error}' )

class EnabledServices( Model ):
   aclService = Bool( help="AclService is enabled", optional=True )
   agentService = Bool( help="AgentService is enabled", optional=True )
   bgpPathService = Bool( help="BgpPathService is enabled", optional=True )
   bgpService = Bool( help="BgpService is enabled", optional=True )
   if EosSdkRpcToggleLib.toggleEosSdkRpcClassMapEnabled():
      classMapService = Bool( help="ClassMapService is enabled", optional=True )
   eapiService = Bool( help="EapiService is enabled", optional=True )
   ethLagIntfService = Bool( help="EthLagIntfService is enabled", optional=True )
   ethPhyIntfService = Bool( help="EthPhyIntfService is enabled", optional=True )
   ethPhyIntfCountersService = Bool( help="EthPhyIntfCountersService is enabeld",
                                     optional=True )
   intfService = Bool( help="IntfService is enabled", optional=True )
   intfCounterService = Bool( help="IntfCounterService is enabled", optional=True )
   ipRouteService = Bool( help="IpRouteService is enabled", optional=True )
   ipIntfService = Bool( help="IpIntfService is enabled", optional=True )
   macsecService = Bool( help="MacsecService is enabled", optional=True )
   mplsRouteService = Bool( help="MplsRouteService is enabled", optional=True )
   mplsVrfLabelService = Bool( help="MplsvrfLabelService is enabled", optional=True )
   nexthopGroupService = Bool( help="NexthopGroupService is enabled", optional=True )
   policyMapService = Bool( help="PolicyMapService is enabled", optional=True )
   policyMapCountersService = Bool( help="PolicyMapServiceCounterService is enabled",
                                    optional=True )

   def render( self ):
      print( "Enabled services:" )
      printEnabledService( "AclService", self.aclService )
      printEnabledService( "AgentService", self.agentService )
      printEnabledService( "BgpPathService", self.bgpPathService )
      printEnabledService( "BgpService", self.bgpService )
      if EosSdkRpcToggleLib.toggleEosSdkRpcClassMapEnabled():
         printEnabledService( "ClassMapService", self.classMapService )
      printEnabledService( "EapiService", self.eapiService )
      printEnabledService( "EthLagIntfService", self.ethLagIntfService )
      printEnabledService( "EthPhyIntfService", self.ethPhyIntfService )
      printEnabledService( "EthPhyIntfCountersService",
                           self.ethPhyIntfCountersService )
      printEnabledService( "IntfService", self.intfService )
      printEnabledService( "IntfCounterService", self.intfCounterService )
      printEnabledService( "IpRouteService", self.ipRouteService )
      printEnabledService( "IpIntfService", self.ipIntfService )
      printEnabledService( "MacsecService", self.macsecService )
      printEnabledService( "MplsRouteService", self.mplsRouteService )
      printEnabledService( "MplsVrfLabelService", self.mplsRouteService )
      printEnabledService( "NexthopGroupService", self.nexthopGroupService )
      printEnabledService( "PolicyMapService", self.policyMapService )
      printEnabledService( "PolicyMapCountersService",
                           self.policyMapCountersService )

class IpEndpoint( Model ):
   ipAddress = IpGenericAddress( help="IP address for gRPC to listen on" )
   port = Int( help="The port this transport is listening on" )
   isListening = Bool( help="This endpoint is listening for incoming connections" )

class LocalhostUnixSocketEndpoint( Model ):
   socketPath = Str( help="Unix domain socket path to listen on" )
   isListening = Bool( help="This endpoint is listening for incoming connections" )

class LocalInterfaceEndpoint( IpEndpoint ):
   intfId = Interface( help="Listening interface ID" )

class ListeningEndpoints( Model ):
   localInterface = Submodel( valueType=LocalInterfaceEndpoint,
         help="Listening local interface endpoint details",
         optional=True )
   localhostLoopback = Submodel( valueType=IpEndpoint,
         help="Listening local host loopback endpoint details",
         optional=True )
   localhostUnixSocket = Submodel( valueType=LocalhostUnixSocketEndpoint,
         help="Local host loopback endpoint details",
         optional=True )
   vrfName = Str( help="The VRF this transport is listening in" )

class ConfigurationErrors( Model ):
   localIntfAddressInUse = Bool( help="Local interface IP address and port "
                                    "are already in use by another process",
                                 optional=True )
   loopbackAddressInUse = Bool( help="Local interface IP address and port "
                                    "is already in use by another process",
                                optional=True )
   invalidSslProfileState = Bool( help="Invalid SSL profile configured for this "
                                       "transport",
                                  optional=True )
   noSslCertOrKeyFile = Bool( help="Could not locate the certificate or key for the "
                                   "configured SSL profile",
                              optional=True )
   noSslProfileFound = Bool( help="No SSL profile found for the configured SSL "
                                  "profile name",
                             optional=True )
   noTrustedCertsFound = Bool( help="No trusted certificate found for the "
                                    "configured SSL profile",
                               optional=True )
   unavailableActiveIp = Bool( help="Local interface is not configured with an "
                                    "available IP address",
                               optional=True )
   loopbackVrfSkipped = Bool( help="Listener for localhost loopback is not enabled, "
                                   "due to conflicting VRF configuration",
                              optional=True )
   unixSocketInUse = Bool( help="UNIX Domain Socket is already in use by "
                                "another process",
                           optional=True )
   noValidAddresses = Bool( help="No valid addresses are configured to listen on",
                            optional=True )

class EosSdkRpcTransportStatus( Model ):
   enabled = Bool( help="Transport is enabled" )
   running = Bool( help="Transport is running" )
   pid = Int( help='The PID of the process, if the agent is running',
              optional=True )
   endpoints = Submodel( valueType=ListeningEndpoints,
         help="The endpoints this transport is listening on",
         optional=True )
   allServicesEnabled = Bool( help="All EosSdkRpc services are enabled" )
   enabledServices = Submodel( valueType=EnabledServices,
                               help="Explicitly enabled EosSdkRpc services",
                               optional=True )
   sslProfile = Str( help="SSL profile name", optional=True )
   serverSecurity = Enum( values=ServerSecurity.attributes,
                          help="Type of security channel in place" )
   metadataUserAuth = Enum( values=UserAuthType.attributes,
                            help="Certificate username authentication" )
   configErrors = Submodel( valueType=ConfigurationErrors,
                       help="Configuration error which occurred before "
                            "starting the gRPC server",
                       optional=True )

   def renderConfigErrors( self ):
      errors = self.configErrors
      print( "Configuration errors:" )
      if errors.localIntfAddressInUse:
         printConfigError( "Local interface IP address and port are already in use "
               "by another process" )
      if errors.loopbackAddressInUse:
         printConfigError( "Local loopback IP address and port are already in use "
               "by another process" )
      if errors.invalidSslProfileState:
         printConfigError( f"Invalid SSL profile configured for this transport: "
               f"{self.sslProfile}" )
      if errors.noSslCertOrKeyFile:
         printConfigError( f"No certificate or key found for the configured "
               f"SSL profile: {self.sslProfile}" )
      if errors.noSslProfileFound:
         printConfigError( f"No matching SSL profile found for {self.sslProfile}" )
      if errors.noTrustedCertsFound:
         printConfigError( f"No trusted certificate found for the configured "
               f"SSL profile: {self.sslProfile}" )
      if errors.unavailableActiveIp:
         intfId = ( "unknown" if not ( self.endpoints and
                  self.endpoints.localInterface ) else
               self.endpoints.localInterface.intfId.stringValue )
         printConfigError( f"IP address for {intfId} local interface not found" )
      if errors.loopbackVrfSkipped:
         printConfigError( "Only one VRF is supported for all configured "
               "endpoints, listener for localhost loopback is not enabled" )
      if errors.unixSocketInUse:
         printConfigError( "Unix Domain Socket is already in use by "
               "another process" )
      if errors.noValidAddresses:
         printConfigError( "No valid addresses are configured to listen on" )

   def render( self ):
      printLineItem( "Enabled", "yes" if self.enabled else "no" )
      if self.pid:
         printLineItem( "Process ID", self.pid )
      printLineItem( "Server", "running" if self.running else "not running" )
      if self.endpoints:
         printHeader = True

         def printListeningOnHeader( printHeader ):
            if printHeader:
               print( "Listening on:" )
               return False
            return printHeader
         if ( self.endpoints.localInterface and
               self.endpoints.localInterface.isListening ):
            printHeader = printListeningOnHeader( printHeader )
            localIntf = self.endpoints.localInterface
            print( "   Local interface:" )
            print( f"      {localIntf.intfId.stringValue}, "
                   f"IP address is {localIntf.ipAddress}" )
            print( f"      port: {localIntf.port}, VRF: {self.endpoints.vrfName}" )
         if ( self.endpoints.localhostLoopback and
               self.endpoints.localhostLoopback.isListening ):
            printHeader = printListeningOnHeader( printHeader )
            localLoopback = self.endpoints.localhostLoopback
            print( "   Localhost loopback:" )
            print( f"      port: {localLoopback.port}, "
                   f"VRF: {self.endpoints.vrfName}" )
         if ( self.endpoints.localhostUnixSocket and
               self.endpoints.localhostUnixSocket.isListening ):
            printHeader = printListeningOnHeader( printHeader )
            print( "   Localhost unix-socket:" )
            print( f"      file: {self.endpoints.localhostUnixSocket.socketPath}" )
      if self.allServicesEnabled:
         printLineItem( "Enabled services", "All" )
      elif self.enabledServices:
         self.enabledServices.render()
      serverSecurityStr = None
      if self.serverSecurity in [ 'mtls', 'tls' ]:
         serverSecurityStr = "mutual TLS" if self.serverSecurity == 'mtls' else "TLS"
         serverSecurityStr += " configured"
      elif self.serverSecurity in [ 'insecure' ]:
         serverSecurityStr = "insecure transport"
      sslProfileLine = 'none'
      if self.sslProfile:
         sslProfileLine = self.sslProfile
         if serverSecurityStr:
            sslProfileLine += f", {serverSecurityStr}"
      printLineItem( "SSL profile", sslProfileLine )
      printLineItem( "Metadata username authentication",
            metadataAuthEnumToStr( self.metadataUserAuth ) )
      if self.configErrors:
         self.renderConfigErrors()

class EosSdkRpcAggregateStatus( Model ):
   enabled = Bool( help="Aggregate transports enabled" )
   transports = Dict( valueType=EosSdkRpcTransportStatus,
                      help="Transports indexed by name" )

   def render( self ):
      if not self.transports:
         printLineItem( "Enabled", "no transports enabled" )
         return
      for i, ( name, transport ) in enumerate( sorted(
                                   self.transports.items() ) ):
         printLineItem( "Transport", name )
         transport.render()
         # Print a blank line only in between transports.
         if i != len( self.transports ) - 1:
            print()
