#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sqlite3
from CliPlugin import EventMonCli
from CliDynamicPlugin.EventMonStatusModel import EventMonStatus, TableStatus
from EventMonCliTables import allEventsDict

def getSanitizedTableSize( tableName ):
   maxTableSize = EventMonCli.config.table[ tableName ].maxTableSize.val

   if tableName in EventMonCli.status.tableFull and \
      EventMonCli.status.tableFull[ tableName ]:
      return maxTableSize

   currTableSize = getTableSize( tableName )
   if currTableSize is None:
      currTableSize = 0
   
   # Use min since currTableSize can drift above maxTableSize by a margin.
   return min( maxTableSize, currTableSize )
   
def getTableSize( tableName ):
   size = 0
   db = sqlite3.connect( EventMonCli.config.dbLocation )
   try:
      with db:
         sql = f'select sum("pgsize") from "dbstat" where name="{tableName}";'
         cursor = db.execute( sql )
         size = cursor.fetchone()[ 0 ]
   except sqlite3.Error:
      return 0
   finally:
      db.close()
   return size

def getEventCount( tableName ):
   if tableName in EventMonCli.status.rowCount:
      return EventMonCli.status.rowCount[ tableName ]
   return 0

def showEventMonStatus( mode, args ):
   events = {}

   if not EventMonCli.config.agentEnabled:
      mode.addWarning( "event-monitor not running. "
                       "It can be started by running 'event-monitor' " 
                       "command in config mode" )
      return EventMonStatus( events=events )
   tableName = args.get( 'TABLE' )

   totalCurrTableSize = 0
   totalMaxTableSize = 0
   # When table name is 'all' or is None display status for all tables.
   if tableName is None or tableName == 'all':
      for table in EventMonCli.config.table:
         if table not in allEventsDict:
            continue
         tableConfig = EventMonCli.config.table[table]

         currTableSize = getSanitizedTableSize( table )

         maxTableSize = EventMonCli.config.table[ table ].maxTableSize.val

         totalCurrTableSize += currTableSize
         totalMaxTableSize += maxTableSize

         events[ table ] = TableStatus(
            enabled = tableConfig.enabled,
            eventCount = getEventCount( table ),
            tableSize = currTableSize,
            maxTableSize = maxTableSize )
   
   elif tableName not in EventMonCli.config.table:
      mode.addError( "Specified event name does not exist in EventMon config." )
      return EventMonStatus( events=events )
   else:
      table = tableName
      tableConfig = EventMonCli.config.table[table]

      maxTableSize = EventMonCli.config.table[ table ].maxTableSize.val
      currTableSize = getSanitizedTableSize( tableName )

      events[ table ] = TableStatus(
         enabled = tableConfig.enabled,
         eventCount = getEventCount( table ),
         tableSize = currTableSize,
         maxTableSize = maxTableSize )

   if tableName is None or tableName == 'all':
      return EventMonStatus( events=events, 
         totalCurSize=totalCurrTableSize, totalMaxSize=totalMaxTableSize )

   return EventMonStatus( events=events )
