#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Int, Model, Bool
from TableOutput import (
      createTable,
      Format
)

class TableStatus( Model ):
   enabled = Bool( help = "Event monitoring enabled/disabled for this event" )
   eventCount = Int( help = "Number of events stored in its table" )
   tableSize = Int( help = "Current size of the event's table" )
   maxTableSize = Int( help = "Maximum size of the event's table" )

class EventMonStatus( Model ):
   events = Dict( keyType=str, valueType=TableStatus,
      help="Map an event name to its table status" )

   totalCurSize = Int( help="Sum of current table sizes.", optional=True )
   totalMaxSize = Int( help="Sum of max table sizes.", optional=True )

   def enabledString( self, enabledFlag ):
      return "enabled" if enabledFlag else "disabled"

   def tableSizeToString( self, tableSize ):
      kilobyte = 1000
      megabyte = kilobyte * 1000
      gigabyte = megabyte * 1000
      if tableSize >= gigabyte:
         return f"{ round( tableSize / gigabyte, 2 ) }G"
      elif tableSize > megabyte:
         return f"{ round( tableSize / megabyte, 2 ) }M"
      elif tableSize > kilobyte:
         return f"{ round( tableSize / kilobyte, 2 ) }K"
      else:
         return f"{ tableSize }B"

   def render( self ):
      textColumn = Format( justify="left" )
      textColumn.noPadLeftIs( True )
      textColumn.padLimitIs( True )
      floatColumn = Format( justify="right",dotAlign=True)
      floatColumn.noPadLeftIs( True )
      floatColumn.padLimitIs( True )
      intColumn = Format( justify="right")
      intColumn.padLimitIs( True )
      output = createTable(
         ( 'Event Name', 'Enabled', 'Event Count', 'Current Size', 'Maximum Size' ) )
      output.formatColumns( textColumn, textColumn, intColumn, floatColumn, 
         floatColumn )

      for eventName, tableStatus in sorted( self.events.items() ):
         output.newRow( eventName, self.enabledString( tableStatus.enabled ),
            tableStatus.eventCount,
            self.tableSizeToString( tableStatus.tableSize ),
            self.tableSizeToString( tableStatus.maxTableSize ) )
      
      print(output.output())

      if self.totalCurSize is not None:
         # Add the summary if all tables are being printed out
         print( 'Total current table size:',
            self.tableSizeToString( self.totalCurSize ))
         print( 'Total maximum table size:',
            self.tableSizeToString( self.totalMaxSize ))