# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import CliDynamicPlugin
from CliModel import Model, DeferredModel, Submodel, Str, Int, Float, List, Enum, \
GeneratorDict, Bool, Dict
from ArnetModel import IpGenericAddress, Ip4Address, IpGenericPrefix, MacAddress
from CliPlugin.BgpCliModels import (
   _degradeToIpGenAddress,
   BgpRoutePeerEntry,
   EncapLabel,
   PmsiTunnel,
   REASON_NOT_BEST_LIST,
)
from IntfModels import Interface
from prettytable import PLAIN_COLUMNS, PrettyTable
from TableOutput import Format, createTable
from Toggles import (
   EvpnToggleLib,
)
import time

BgpVpnModels = CliDynamicPlugin( "BgpVpnModels" )
BgpVpnRouteDetailEntry = BgpVpnModels.BgpVpnRouteDetailEntry
BgpVpnRouteTypeEntry = BgpVpnModels.BgpVpnRouteTypeEntry

class MulticastFlags( DeferredModel ):
   filterMode = Enum( optional=True, values=( 'include', 'exclude' ),
                      help='Group membership filter mode' )

class EvpnRouteASPathEntry( DeferredModel ):
   asPath = Str( optional=True, help='AS path string (if absent,  \
                 then the route was originated locally)' )
   asPathType = Enum( values=( 'Internal', 'External', 'Confed-External', 'Local', 
                               'Invalid' ), 
                      help='AS path type: \
                            Internal - originated by I-BGP \
                            External - originated by E-BGP \
                            Confed-External - originated by a E-BGP confederation \
                            Local - originated locally \
                            Invalid - AS path is invalid' )

class TunnelEncap( DeferredModel ):
   asn = Int( help='Autonomous System Number' )
   endpoint = IpGenericAddress( help='Tunnel egress endpoint' )

class EvpnRouteDetailEntry( BgpVpnRouteDetailEntry ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         if 'peerEntry' in dictRepr:
            dictRepr[ 'peerEntry' ] = \
                  BgpRoutePeerEntry().degrade( dictRepr[ 'peerEntry' ], 1 )
      return dictRepr

   aggregator = Str( optional=True, help='Aggregator of the route' )
   rxPathId = Int( optional=True, help='Received path ID of this route' )
   txPathId = Int( optional=True, help='Advertised path ID for this route' )
   seqNumber = Int( optional=True, help='Route sequence number' )
   pendingTimeToAdv = Float( optional=True, help='Timestamp of route advertisement' )
   # the redistibution protocol strings here need to be kept in sync
   # with the strings returned by
   # gated/gated-ctk/src/bgp/bgp_dget_route.c:redist_proto_str()
   # note these are usually transformed by maybeCamelize()
   redistributionProtocol = Enum( values=( 'Connected', 'Static', 'Ospf3', 'Ospf',
                                           'Rip', 'Is-Is', 'unknown' ),
                                  optional=True,
                                  help='Protocol from which route got \
                                     redistributed into BGP' )
   labelStack = List( optional=True, valueType=int, 
                      help='MPLS label stack information')

   labelVxlan = Submodel( optional=True, valueType=EncapLabel,
                          help='Local or remote VXLAN label' )
   labelMpls = Submodel( optional=True, valueType=EncapLabel,
                         help='Local or remote MPLS label' )
   labelSrv6 = Submodel( optional=True, valueType=EncapLabel,
                         help='Local or remote SRv6 SID (label)' )
   label = Submodel( optional=True, valueType=EncapLabel,
                     help='Local or remote label' )
   l3LabelVxlan = Submodel( optional=True, valueType=EncapLabel,
                            help='Local or remote L3 VXLAN label' )
   l3LabelMpls = Submodel( optional=True, valueType=EncapLabel,
                           help='Local or remote L3 MPLS label' )
   l3Label = Submodel( optional=True, valueType=EncapLabel,
                       help='Local or remote L3 label' )
   esi = Str( optional=True, help="Ethernet segment identifier" )
   gatewayIpAddr = IpGenericAddress( optional=True, help="Route gateway IP address" )
   domainPath = List( optional=True, valueType=str, help="Domain path attribute" )
   pmsiTunnel = Submodel( optional=True, valueType=PmsiTunnel,
                          help="P-Multicast Serivce Interface Tunnel Attribute" )
   multicastFlags = Submodel( optional=True, valueType=MulticastFlags,
                              help="Multicast flags for route-type smet, "
                                   "join-sync, and leave-sync" )
   tunnelEncap = GeneratorDict( optional=True, valueType=TunnelEncap,
         help="Tunnel encapsulation attribute" )

class EvpnRoutePath( DeferredModel ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2 :
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         if 'nextHop' in dictRepr:
            dictRepr[ 'nextHop' ] = _degradeToIpGenAddress( dictRepr[ 'nextHop' ] )
         if 'routeDetail' in dictRepr:
            dictRepr[ 'routeDetail' ] = \
                  EvpnRouteDetailEntry().degrade( dictRepr[ 'routeDetail' ], 1 )
      return dictRepr

   nextHop = Str( optional=True, help='Route next hop address' )
   asPathEntry = Submodel( valueType=EvpnRouteASPathEntry, 
                           help='AS path information' )
   importedEvpnPathRd = Str( optional=True,
                             help='Imported EVPN path RouteDistinguisher' )
   med = Int( optional=True, help='Multi Exit Discriminator for the route' )
   localPreference = Int( optional=True, help='I-BGP Local preference indicator' )
   routeType = Submodel( valueType=BgpVpnRouteTypeEntry, help='Route type' )
   weight = Int( optional=True, help='Weight for the route' )
   tag = Int( optional=True, help='Tag for the route' )
   timestamp = Int( optional=True,
                    help="UTC seconds since epoch when the route was received.\
                          Only returned with 'show bgp evpn detail'" )
   routeDetail = Submodel( valueType=EvpnRouteDetailEntry, optional=True, 
                           help='Route details' )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )

class EvpnRouteKeyDetail( DeferredModel ):
   rd = Str( optional=True, help='Route distinguisher' )
   nlriType = Enum( values=( 'auto-discovery', 'mac-ip', 'imet', 'ethernet-segment',
                             'ip-prefix', 'smet', 'spmsi', 'join-sync',
                             'leave-sync', 'mac-address', 'mac-arp' ),
                             help='NLRI type' )
   eti = Int( optional=True,
         help='For auto-discovery, mac-ip, imet, and ip-prefix routes, the ethernet \
              tag identifier' )
   eti2 = Int( optional=True,
         help='For dual-encap mac-ip and imet routes, the second ethernet tag \
              identifier.' )
   esi = Str( optional=True,
         help='For auto-discovery and ethernet-segment routes, the ethernet segment \
              identifier' )
   mac = Str( optional=True, help='For mac-ip routes, the ethernet address' )
   ipGenAddr = IpGenericAddress( optional=True,
         help='For mac-ip routes, the IP address corresponding to the ethernet \
               address. \
               For imet and ethernet-segment routes, the originating router IP \
               address.' )
   ipGenAddr2 = IpGenericAddress( optional=True,
         help='For dual-encap imet routes, the second originating router IP \
               address.' )
   ipGenPrefix = IpGenericPrefix( optional=True,
         help='For ip-prefix routes, the IPv4 or IPv6 address prefix' )
   source = IpGenericAddress( optional=True, help='Multicast source address' )
   group = IpGenericAddress( optional=True, help='Multicast group address' )
   syncnum = Int( optional=True, help='Leave group synchronization number' )
   domain = Enum( optional=True, values=( 'local', 'remote' ),
                  help='BGP route domain' )

class EvpnRouteEntry( DeferredModel ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         evpnRoutePaths = [ EvpnRoutePath().degrade( rtPath, 1 ) \
                           for rtPath in dictRepr[ 'evpnRoutePaths' ] ]
         dictRepr[ 'evpnRoutePaths' ] = evpnRoutePaths
      return dictRepr

   routeKeyDetail = Submodel( valueType=EvpnRouteKeyDetail, help='NLRI details' )
   totalPaths = Int( optional=True, help='Total number of paths for this route' )
   routePriority = Enum( values=( 'low', 'medium', 'high' ),
                         help="Route processing priority",
                         optional=True )
   evpnRoutePaths = List( valueType=EvpnRoutePath,
                          help='List of BGP EVPN route ECMP paths' )

class EvpnRoutes( DeferredModel ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         dictRepr[ 'evpnRoutes' ] = {
               key : EvpnRouteEntry().degrade( rtEntry, 1 ) for key, rtEntry in
               dictRepr[ 'evpnRoutes' ].items() }
      return dictRepr

   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Int( help='Autonomous System Number' )
   lastProcessedSeqNum = Int( optional=True, 
         help='Last route sequence number acknowledged' )
   currentSeqNum = Int( optional=True, 
         help='Current route sequence number' )
   _detail = Bool( optional=True, help='Detailed output is requested' )
   _advRoutes = Bool( optional=True, help='Advertised routes output is requested' )
   evpnRoutes = GeneratorDict( keyType=str, valueType=EvpnRouteEntry, 
         help='Dictionary of BGP EVPN route entries indexed by the route key' )

class EvpnPathL2TableEntry( DeferredModel ):
   nextHop = Str( help='Route next hop address' )
   vlan = Int( help='VLAN that the path was imported into' )
   labelVxlan = Submodel( optional=True, valueType=EncapLabel,
                          help='Local or remote VXLAN label' )
   labelMpls = Submodel( optional=True, valueType=EncapLabel,
                         help='Local or remote MPLS label' )
   seqNum = Int( help='EVPN MAC mobility extended community',
                 optional=True )

class EvpnRouteL2TableEntry( DeferredModel ):
   routeKeyDetail = Submodel( valueType=EvpnRouteKeyDetail, help='NLRI details' )
   totalPaths = Int( optional=True, help='Total number of paths for this route' )
   routePriority = Enum( values=( 'low', 'medium', 'high' ),
                         help="Route processing priority",
                         optional=True )
   evpnRoutePaths = List( valueType=EvpnPathL2TableEntry,
                          help='List of BGP EVPN route ECMP paths' )

class EvpnRoutesL2( DeferredModel ):
   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Int( help='Autonomous System Number' )
   evpnRoutes = GeneratorDict( keyType=str, valueType=EvpnRouteL2TableEntry,
         help='Dictionary of BGP EVPN route entries indexed by the route key' )

class EvpnPathRouteTypeCount( DeferredModel ):
   autoDiscovery = Int( optional=True,
         help="Number of Ethernet auto-discovery (A-D) route (type 1) paths" )
   macIp = Int( optional=True,
         help="Number of MAC/IP advertisement route (type 2) paths" )
   imet = Int( optional=True,
         help="Number of inclusive multicast Ethernet tag route (type 3) paths" )
   ethernetSegment = Int( optional=True,
         help="Number of Ethernet segment route (type 4) paths" )
   ipPrefixIpv4 = Int( optional=True,
         help="Number of IPv4 prefix route (type 5) paths" )
   ipPrefixIpv6 = Int( optional=True,
         help="Number of IPv6 prefix route (type 5) paths" )

class EsiPeer( DeferredModel ):
   ip = IpGenericAddress( help='IP address of peer' )
   esiLabel = Int( help='MPLS ESI label advertised by peer',
                   optional=True )

class EthernetSegmentPseudowire( DeferredModel ):
   primary = IpGenericAddress( help="IP address of the primary peer", optional=True )
   backup = IpGenericAddress( help="IP address of the backup peer", optional=True )

class EthernetSegment( DeferredModel ):
   intf = Interface( help='Interface name' )
   redundancyMode = Enum( values=( 'singleActive', 'allActive' ),
                          help='Redundancy mode' )
   state = Enum( values=( 'up', 'down' ), help='Ethernet segment state' )
   esiLabel = Int( help='MPLS ESI label used for split-horizon filtering',
                   optional=True )
   sharedEsiLabel = Bool( help="MPLS ESI label shared with other PEs",
                          optional=True )
   if EvpnToggleLib.toggleEvpnType1EsiEnabled():
      importRTs = List( valueType=str, help='ES import route target(s)' )
   importRT = Str( help=( 'ES import route target'
                          if EvpnToggleLib.toggleEvpnType1EsiEnabled() else
                          'ES import route target (now obsolete as this field is '
                          'included in importRTs)' ) )
   dFElectionState = Enum( values=( 'pending', 'started', 'finished' ),
                           help='DF election state', optional=True )
   dFElectionAlgorithm = Enum( values=( 'modulus', 'preference', 'hrw' ),
                               help='DF election algorithm', optional=True )
   forwardingPeers = List( valueType=EsiPeer, help='All forwarding peers' )
   dFPeer = Submodel( valueType=EsiPeer, help='Designated forwarder' )
   nonDFPeers = List( valueType=EsiPeer, help='Non-designated forwarders' )
   pseudowires = Dict( valueType=EthernetSegmentPseudowire,
                       help="Mapping of pseudowire name to detailed information",
                       optional=True )

class RemoteEthernetSegment( DeferredModel ):
   activeTEPs = List( valueType=IpGenericAddress,
                      help="IP addresses of the remote peers "
                      "that has the Ethernet segment" )

class BgpEvpnVpwsPseudowire( DeferredModel ):
   status = Enum( values=( "pending", "up", "noLocalId", "noRemoteId",
                           "localIdConflict", "remoteIdConflict", "noRequest",
                           "noLocalLabel" ),
                  help="Pseudowire status" )
   vpwsLabel = Int( help="Advertised pseudowire label", optional=True )
   vpwsIdLocal = Int( help="Advertised virtual private wire service (VPWS) "
                      "instance identifier", optional=True )
   vpwsIdRemote = Int( help="Peer's virtual private wire service (VPWS) "
                       "instance identifier", optional=True )
   # Pseudowire::Fxc::VidNormalization: singleVid -> single, doubleVid -> double
   # for noNormalization, this attribute is suppressed.
   vlanTagNormalization = Enum( values=( "single", "double" ),
                                help="Flexible cross connect VLAN tag normalization",
                                optional=True )
   # If the color is not configured, this attribute is suppressed.
   color = Int( help="Color", optional=True )

class BgpEvpnVpwsDetail( DeferredModel ):
   l2Mtu = Int( help="L2 MTU value used for validation, if enabled", optional=True )
   controlWord = Bool( help="Control word required when sending EVPN packets to "
                       "this device" )
   # Pseudowire::Fxc::Mode: default -> none, vlanSignaledFxc -> all
   # for notFxc, this attribute is suppressed.
   fxcNormalizedVlanTagSignaling = Enum(
      values=( "none", "all" ),
      help="Flexible cross connect normalized VLAN tag signaling",
      optional=True )
   pseudowires = Dict( valueType=BgpEvpnVpwsPseudowire,
                       help="Mapping of pseudowire name to detailed information" )

class InstanceAfiSafiConfigs( DeferredModel ):
   """
   Possible route targets Imports/Exports 
   """
   routeTargetImports = List( valueType=str, optional=True,
                              help="Route targets to import." )
   routeTargetExports = List( valueType=str, optional=True,
                              help="Route targets to export." )

class BgpEvpnInstance( DeferredModel ):
   __revision__ = 2

   rd = Str( help='EVPN instance route distinguisher' )
   remoteRd = Str( help='EVPN instance remote route distinguisher',
                   optional=True )
   importRts = List( valueType=int, help="Route targets to import",
                     optional=True )
   exportRts = List( valueType=int, help="Route targets to export",
                     optional=True )
   importRemoteRts = List( valueType=int, help="Remote route targets to import",
                           optional=True )
   exportRemoteRts = List( valueType=int, help="Remote route targets to export",
                           optional=True )
   serviceIntf = Enum( values=( 'vlanBased', 'vlanAwareBundle' ),
                       help='EVPN instance service interface',
                       optional=True )
   localVxlanIp = IpGenericAddress( help='Local VXLAN IP', optional=True )
   localMplsIp = IpGenericAddress( help='Local MPLS IP', optional=True )
   vxlanEnabled = Bool( default=False, help='VXLAN Enabled', optional=True )
   mplsEnabled = Bool( default=False, help='MPLS Enabled', optional=True )
   labelAllocMode = Enum( values=( 'perVlan', 'perInstance', 'perPseudowire' ),
                          help='Label allocation mode', optional=True )
   macLabel = Int( help='MPLS label used in mac-ip routes', optional=True )
   imetLabel = Int( help='MPLS label used in imet routes', optional=True )
   adLabel = Int( help='MPLS label used in auto-discovery routes',
                  optional=True )
   evpnVpwsDetail = Submodel( valueType=BgpEvpnVpwsDetail,
         help="EVPN virtual private wire service (VPWS) detailed information",
         optional=True )
   ethernetSegments = Dict( valueType=EthernetSegment,
                            help='Local ethernet segments indexed by ESI',
                            optional=True )
   remoteEthernetSegments = Dict( valueType=RemoteEthernetSegment,
                                  help='Remote ethernet segments',
                                  optional=True )
   afiSafiConfig = Dict( valueType=InstanceAfiSafiConfigs,
                         help='Route targets for an address family type',
                         optional=True )
   exportRtRcf = Str( help='RCF function to apply on route target export',
                      optional=True )

class BgpEvpnInstances( DeferredModel ):
   __revision__ = 2

   bgpEvpnInstances = Dict( valueType=BgpEvpnInstance,
                            help='BGP EVPN instances indexed by instance name' )

class BgpEvpnHostFlapEntry( Model ):
   vlan = Int( help="VLAN ID of blacklisted MAC Address" )
   macAddr = MacAddress( help="Blacklisted MAC Address due to duplication" )
   time = Float( help="Time when the MAC Address was added to the blacklist" )

class BgpEvpnHostFlapEntries( Model ):
   duplicationBlacklistedMacs = List( valueType=BgpEvpnHostFlapEntry,
         help='List of routes that contain MAC Addresses '\
         'that are blacklisted due to duplication' )

   def render( self ):
      table = PrettyTable( [ 'VLAN', 'MAC Address', 'time' ] )
      table.set_style( PLAIN_COLUMNS )
      for entry in self.duplicationBlacklistedMacs:
         table.add_row( [ entry.vlan, entry.macAddr,
                          time.ctime( entry.time ) ] )
      print( table )

class EvpnSanityItem( Model ):
   name = Str( help="Name of item" )
   status = Enum( help='Status of verification of item',
                  values=( 'fail', 'ok', 'warn' ),
                  default='fail' )
   reasonNotOk = Str( help='Reason why this check was not successful',
                      optional=True )
   _id = Str( help='Unique identifier of this item needed for Sysdb storage' )

   def idIs( self, ID ):
      self._id = ID

   def id( self ):
      return self._id

class EvpnSanityCategory( Model ):
   items = List( help='Items to verify in this category',
                 valueType=EvpnSanityItem )
   description = Str( help='Description of category' )
   _order = Int( help="Order for output", default=0 )

   def orderIs( self, order ):
      self._order = order

   def order( self ):
      return self._order

   def getCategoryStatus( self ):
      allOK = all( item.status == 'ok' for item in self.items )
      anyFAIL = any( item.status == 'fail' for item in self.items )
      status = 'OK' if allOK else 'FAIL' if anyFAIL else 'WARN'
      return status

# Takes in a category tuple ( categoryName, EvpnSanityCategory ) and returns
# the '_order' attribute of EvpnSanityCategory
def categorySortFunc( categoryTuple ):
   return categoryTuple[ 1 ][ '_order' ]

class EvpnSanityModel( Model ):
   categories = Dict( help='EVPN sanity categories to verify, keyed by their name',
                      keyType=str, valueType=EvpnSanityCategory )

   def _setFormatColumns( self, formatList ):
      for columnFormat in formatList:
         columnFormat.noPadLeftIs( True )
         columnFormat.padLimitIs( True )

   def _allChecksPass( self ):
      for c in self.categories.values():
         if c.getCategoryStatus() != 'OK':
            return False
      return True

   def _createTableCategoryOnly( self ):
      categoryFormat = Format( justify='left', minWidth=8,
                               isHeading=False, border=False )
      statusFormat = Format( justify='left', minWidth=6, maxWidth=9,
                             isHeading=False, border=False )
      self._setFormatColumns( [ categoryFormat, statusFormat ] )
      headingStrings = [ 'Category', 'Status' ]
      table = createTable( headingStrings )
      table.formatColumns( categoryFormat, statusFormat )
      return table

   def _createTable( self ):
      categoryFormat = Format( justify='left', minWidth=8,
                               isHeading=False, border=False )
      checkFormat = Format( justify='left', minWidth=5, maxWidth=20,
                             isHeading=False, border=False, wrap=True )
      statusFormat = Format( justify='left', minWidth=6, maxWidth=9,
                             isHeading=False, border=False )
      detailFormat = Format( justify='left', minWidth=6, maxWidth=70,
                             isHeading=False, border=False, wrap=True )
      self._setFormatColumns( [ categoryFormat, checkFormat,
                                statusFormat, detailFormat ] )
      headingStrings = [ 'Category', 'Check', 'Status', 'Detail' ]
      table = createTable( headingStrings )
      table.formatColumns( categoryFormat, checkFormat,
                           statusFormat, detailFormat )
      return table

   def render( self ):
      if self._allChecksPass():
         return

      table = self._createTableCategoryOnly()
      for categoryName, category in sorted( self.categories.items(),
                                            key=categorySortFunc ):
         categoryStatus = category.getCategoryStatus()
         if categoryStatus == 'OK':
            continue
         table.newRow( categoryName, categoryStatus )
      print( table.output() )

class EvpnSanityModelBrief( EvpnSanityModel ):
   def render( self ):
      if self._allChecksPass():
         return

      table = self._createTable()
      for categoryName, category in sorted( self.categories.items(),
                                            key=categorySortFunc ):
         for item in category.items:
            if item.status != 'ok':
               table.newRow( categoryName, item.name,
                             item.status.upper(), item.reasonNotOk )
      print( table.output() )

class EvpnSanityModelDetail( EvpnSanityModel ):
   def render( self ):
      table = self._createTable()
      for categoryName, category in sorted( self.categories.items(),
                                            key=categorySortFunc ):

         for item in category.items:
            table.newRow( categoryName, item.name,
                          item.status.upper(), item.reasonNotOk or '' )
      print( table.output() )
