# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliMode.Firewall
import CliParser
from CliPlugin import EthIntfCli
from CliPlugin import PhysicalIntfRule
from CliPlugin.FirewallCli import (
      applicationNameMatcher, firewallPolicyNameMatcher, segmentNameMatcher,
      vrfSupportedGuard )
from CliPlugin.FirewallCliLib import (
      clearFirewallCountersHook, clearFirewallSessionHook )
from CliPlugin.LagIntfCli import EthLagIntf, LagSubIntf
from CliPlugin.RouteMapCli import rtMapCommListPreListRe
from CliPlugin.VirtualIntfRule import IntfMatcher, VirtualIntfMatcher
from CliPlugin.VlanCli import vlanSetMatcher
from CliPlugin.VrfCli import DEFAULT_VRF, VrfExprFactory
from CliPlugin.IpAddrMatcher import IpAddrMatcher
import ConfigMount
from IntfRangePlugin.LagIntf import PortChannelNum
import LazyMount
import Tracing
import Tac
from TypeFuture import TacLazyType

t0 = Tracing.Handle( 'FirewallCli' ).trace0

gv = CliGlobal.CliGlobal(
   firewallConfig=None,
   firewallCounters=None,
   firewallCountersSnapshot=None,
   hwCapability=None,
   l3ConfigDir=None,
   matchListCapability=None,
   matchListConfig=None,
)

FwPolicy = TacLazyType( 'Firewall::Policy' )
FwVlanRange = TacLazyType( "Firewall::VlanRange" )
FwRuleDefinition = TacLazyType( "Firewall::RuleDefinition" )
ForwardingTypeMatch = TacLazyType( "Firewall::ForwardingTypeMatch" )

# -----------------------------------------------------------------------------------
# Context
# A context is created for FirewallPolicyConfigMode and stores(buffers) the command
# till the user exists the mode or aborts the changes.
#
# If the policy exists already, the context contains an editable copy of the
# contents; else it contains a new (editable) copy
# -----------------------------------------------------------------------------------
class Context:
   def __init__( self, policyName, segSecPolicy ):
      self.policyName = policyName
      self.policy = segSecPolicy
      self.tmpPolicy = None
      self.mode_ = None
      self.copyPolicy()

   def copyPolicy( self ):
      self.tmpPolicy = FwPolicy( self.policyName )
      pol = self.policy.get( self.policyName )
      if pol is None:
         return
      if pol.readonly:
         self.tmpPolicy.readonly = True # won't be modifying, don't need to copy
         return
      for app in pol.serviceToSeqnum:
         self.tmpPolicy.serviceToSeqnum[ app ] = pol.serviceToSeqnum[ app ]

      for rule in pol.rule.values():
         self.tmpPolicy.rule.addMember( rule )

   @property
   def readonly( self ):
      return self.tmpPolicy.readonly

   def modeIs( self, mode ):
      self.mode_ = mode

   def commit( self ):
      pol = self.policy.newMember( self.policyName )
      if pol.readonly:
         return # We shouldn't have reached this point but just do nothing if we have

      for app in self.tmpPolicy.serviceToSeqnum:
         pol.serviceToSeqnum[ app ] = self.tmpPolicy.serviceToSeqnum[ app ]

      # delete case
      for app in pol.serviceToSeqnum:
         if app not in self.tmpPolicy.serviceToSeqnum:
            del pol.serviceToSeqnum[ app ]

      for rule in self.tmpPolicy.rule.values():
         pol.rule.addMember( rule )

      # delete case
      for seqnum in pol.rule:
         if seqnum not in self.tmpPolicy.rule:
            del pol.rule[ seqnum ]

   def abort( self ):
      self.policyName = None
      self.tmpPolicy = None

# --------------------------------------------------------------------------
# region Nodes and matchers
# --------------------------------------------------------------------------
def segSecConfigurablePolicySupportedGuard( mode, token ):
   if gv.hwCapability.segSecConfigurablePolicySupported:
      return None
   return CliParser.guardNotThisPlatform

segmentSecurityConfigurablePolicyNode = CliCommand.guardedKeyword(
      'policy',
      helpdesc='Configure Segment Security policy',
      guard=segSecConfigurablePolicySupportedGuard )

def segSecTrafficInspectionConfigurableSupportedGuard( mode, token ):
   if gv.hwCapability.forwardingTypeSupported:
      return None
   return CliParser.guardNotThisPlatform

segmentSecurityTrafficInspectPolicyNode = CliCommand.guardedKeyword(
      'forwarding-type',
      helpdesc='Configure forwarding types that Segment Security applies to',
      guard=segSecTrafficInspectionConfigurableSupportedGuard )

def segSecBridgedSupportedGuard( mode, token ):
   if gv.hwCapability.forwardingTypeSupported.bridged:
      return None
   return CliParser.guardNotThisPlatform

segmentSecurityBridgedSupportedNode = CliCommand.guardedKeyword(
      'bridged',
      helpdesc='Inspect bridged traffic',
      guard=segSecBridgedSupportedGuard )

def segSecRoutedSupportedGuard( mode, token ):
   if gv.hwCapability.forwardingTypeSupported.routed:
      return None
   return CliParser.guardNotThisPlatform

segmentSecurityRoutedSupportedNode = CliCommand.guardedKeyword(
      'routed',
      helpdesc='Inspect routed traffic',
      guard=segSecRoutedSupportedGuard )

def getAllEditablePoliciesName( mode ):
   return [ policyName for policyName, policy in gv.firewallConfig.policy.items()
                       if not policy.readonly ]

firewallEditablePolicyNameMatcher = CliMatcher.DynamicNameMatcher(
      getAllEditablePoliciesName, helpdesc='Policy name' )

def getSegmentNamesInVrf( mode ):
   if hasattr( mode, 'vrfName_' ):
      vrfName = mode.vrfName_
   elif hasattr( mode.parent_.parent_, 'vrfName_' ):
      vrfName = mode.parent_.parent_.vrfName_
   else:
      return None

   return gv.firewallConfig.vrf[ vrfName ].segment

vrfSegmentNameMatcher = CliMatcher.DynamicNameMatcher( getSegmentNamesInVrf,
      pattern='[a-zA-Z0-9_-]+', helpdesc='Segment name', helpname='WORD' )

def intfVlanSupportedGuard( mode, token ):
   if gv.hwCapability.intfVlanSupported:
      return None
   return CliParser.guardNotThisPlatform

segmentNode = CliCommand.Node( matcher=segmentNameMatcher,
                               guard=intfVlanSupportedGuard )

# max rule sequence number 10-bit. We use only 10-bits out of 32-bit, as this
# will be combined with 10 bit service rule sequence number to generate a 20 bit
# ruleId.
# Platforms can use the remaining 12 bits to augment any PD specific things
MAX_SEQ = 0x3FF # 10-bit integer
sequenceNumMatcher = CliMatcher.IntegerMatcher(
      1, MAX_SEQ, helpdesc='Index in the sequence' )

def redirectActionSupportedGuard( mode, token ):
   if gv.hwCapability.segSecNexthopRedirectSupported:
      return None
   return CliParser.guardNotThisPlatform

redirectActionNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'redirect',
         helpdesc='Redirect an application' ),
      guard=redirectActionSupportedGuard )

def statelessActionSupportedGuard( mode, token ):
   if gv.hwCapability.statelessActionSupported:
      return None
   return CliParser.guardNotThisPlatform

statelessActionNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'stateless',
         helpdesc='Perform action irrespective of state' ),
      guard=statelessActionSupportedGuard )

def logActionSupportedGuard( mode, token ):
   if gv.hwCapability.logActionSupported:
      return None
   return CliParser.guardNotThisPlatform

logActionNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'log',
         helpdesc='Enable logging' ),
      guard=logActionSupportedGuard )

def defaultPolicyDropGuard( mode, token ):
   if gv.hwCapability.defaultPolicyDropSupported:
      return None
   return CliParser.guardNotThisPlatform

policyNodeForDefaultPolicyDrop = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'policy',
         helpdesc='Configure policy default drop behaviour' ),
      guard=defaultPolicyDropGuard )

def forwardingSegmentsSupportedGuard( mode, token ):
   if gv.hwCapability.forwardingSegmentsPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

forwardingSegmentsNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'forwarding-segments',
         helpdesc='All forwarding segments' ),
      guard=forwardingSegmentsSupportedGuard )

def subIntfSegmentSupportedGuard( mode, token ):
   if gv.hwCapability.subIntfSegmentSupported:
      return None
   return CliParser.guardNotThisPlatform

def portChannelSegmentSupportedGuard( mode, token ):
   if gv.hwCapability.portChannelSegmentSupported:
      return None
   return CliParser.guardNotThisPlatform

def segSecMatchInterfaceSupportedGuard( mode, token ):
   if gv.hwCapability.segmentMatchInterfaceSupported:
      return None
   return CliParser.guardNotThisPlatform

intfMatcher = IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= PhysicalIntfRule.PhysicalIntfMatcher(
      'Ethernet', subIntf=True,
      value=lambda mode, intf: EthIntfCli.EthPhyIntf( intf, mode ),
      guard=subIntfSegmentSupportedGuard )

intfMatcher |= VirtualIntfMatcher( 'Port-Channel',
                                   PortChannelNum.min,
                                   PortChannelNum.max,
                                   helpdesc="Link Aggregation Group (LAG)",
                                   value=lambda mode, intf: EthLagIntf( intf, mode ),
                                   guard=portChannelSegmentSupportedGuard )

SubIntfRangeDetails = Tac.Type( "Interface::SubIntfIdConstants" )
subLowerBound = SubIntfRangeDetails.subIntfExtendedIdMin
subUpperBound = SubIntfRangeDetails.subIntfExtendedIdMax

intfMatcher |= VirtualIntfMatcher( 'Port-Channel',
                                   PortChannelNum.min,
                                   PortChannelNum.max,
                                   helpdesc="Link Aggregation Group (LAG) Sub Intf",
                                   subIntf=True,
                                   value=lambda mode,
                                   intf: LagSubIntf( intf, mode ),
                                   subLbound=subLowerBound,
                                   subUbound=subUpperBound,
                                   guard=subIntfSegmentSupportedGuard )

segmentSecurityInterfaceNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'interface',
         helpdesc='Name of the interface' ),
      guard=segSecMatchInterfaceSupportedGuard )

matchNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'match',
         helpdesc='Specify interfaces or subnets' ) )

def getIpv4PrefixListNames( mode ):
   return gv.matchListConfig.matchIpv4PrefixList

def getIpv6PrefixListNames( mode ):
   return gv.matchListConfig.matchIpv6PrefixList

ipv4PrefixListNameMatcher = CliMatcher.DynamicNameMatcher( getIpv4PrefixListNames,
      pattern=rtMapCommListPreListRe, helpdesc='Ipv4 Prefix list' )

ipv6PrefixListNameMatcher = CliMatcher.DynamicNameMatcher( getIpv6PrefixListNames,
      pattern=rtMapCommListPreListRe, helpdesc='Ipv6 Prefix list' )

def segSecMatchIpv6PrefixSupportedGuard( mode, token ):
   if gv.matchListCapability.ipv6PrefixSupported:
      return None
   return CliParser.guardNotThisPlatform

segmentSecurityIpv6PrefixNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'prefix-ipv6',
         helpdesc='Match Ipv6 match list' ),
      guard=segSecMatchIpv6PrefixSupportedGuard )

coveredNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'covered',
         helpdesc='Specify subnets covered using a prefix list' ) )

prefixListNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'prefix-list',
         helpdesc='prefix list to use for this segment' ) )

def fallbackPolicySupportedGuard( mode, token ):
   if gv.hwCapability.fallbackPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

fallbackPolicyNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'fallback',
         helpdesc='Default policy per segment' ),
      guard=fallbackPolicySupportedGuard )
# --------------------------------------------------------------------------
# endregion Nodes and matchers
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region config mode definition
# --------------------------------------------------------------------------
class FirewallConfigMode( CliMode.Firewall.FirewallConfigMode,
                          BasicCli.ConfigModeBase ):
   name = "Global Segment Security configuration"

   def __init__( self, parent, session ):
      self.session_ = session
      CliMode.Firewall.FirewallConfigMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getClassName( self, segmentName ):
      return '__segment_%s' % ( segmentName )

   def getSegment( self, segmentName ):
      return gv.firewallConfig.segmentDir.segment[ segmentName ]

   def getVrfName( self ):
      # no VRF
      return None

   def getConfigChildModeType( self ):
      return ClassMapFirewallConfigMode

   def getPolicyChildModeType( self ):
      return SegmentPolicyMode

class FirewallPolicyConfigMode( CliMode.Firewall.FirewallPolicyConfigMode,
                                BasicCli.ConfigModeBase ):
   context = None
   name = "Segment Security Policy definition"

   def __init__( self, parent, session, context ):
      t0( "PolicyConfigMode init" )
      self.context = context
      self.aborted = False

      CliMode.Firewall.FirewallPolicyConfigMode.__init__( self,
                                                          context.policyName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( "onExit called for PolicyConfigMode" )
      if not self.aborted:
         self.context.commit()

   def abort( self ):
      t0( "abort called for PolicyConfigMode" )
      self.context.abort()
      self.aborted = True
      self.session_.gotoParentMode()

class FirewallVrfConfigMode( CliMode.Firewall.FirewallVrfConfigMode,
                             BasicCli.ConfigModeBase ):
   name = "Segment Security definition"

   def __init__( self, parent, session, vrfName ):
      self.session_ = session
      self.vrfName = vrfName
      gv.firewallConfig.vrf.newMember( vrfName )
      CliMode.Firewall.FirewallVrfConfigMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getClassName( self, segmentName ):
      return f'__VRF_{self.vrfName_}_segment_{segmentName}'

   def getSegment( self, segmentName ):
      return gv.firewallConfig.vrf[ self.vrfName_ ].segmentDir.segment[ segmentName ]

   def getVrfName( self ):
      return self.vrfName

   def getConfigChildModeType( self ):
      return ClassMapFirewallVrfConfigMode

   def getPolicyChildModeType( self ):
      return VrfSegmentPolicyMode

class FirewallVrfSegmentConfigMode( CliMode.Firewall.FirewallVrfSegmentConfigMode,
                                    BasicCli.ConfigModeBase ):
   name = "Segment definition"

   def __init__( self, parent, session, segmentName ):
      self.session_ = session
      gv.firewallConfig.vrf[ parent.vrfName ].segmentDir.segment.newMember(
            segmentName )
      CliMode.Firewall.FirewallVrfSegmentConfigMode.__init__( self,
                                                              ( parent.vrfName_,
                                                                segmentName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class FirewallSegmentConfigMode( CliMode.Firewall.FirewallSegmentConfigMode,
                                 BasicCli.ConfigModeBase ):
   name = "Segment definition"

   def __init__( self, parent, session, segmentName ):
      self.session_ = session
      if gv.firewallConfig.segmentDir is None:
         gv.firewallConfig.segmentDir = ( "l2SegDir", )
      gv.firewallConfig.segmentDir.segment.newMember( segmentName )
      CliMode.Firewall.FirewallSegmentConfigMode.__init__( self, segmentName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ClassMapFirewallVrfConfigMode( CliMode.Firewall.ClassMapFirewallVrfConfigMode,
                                     BasicCli.ConfigModeBase ):
   name = "Configure Segment Security Definition"

   def __init__( self, parent, session, vrfName, segmentName ):
      self.session_ = session
      self.cmapName = parent.parent_.getClassName( segmentName )
      gv.firewallConfig.classMap.newMember( self.cmapName )
      param = ( vrfName, segmentName )
      CliMode.Firewall.ClassMapFirewallVrfConfigMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ClassMapFirewallConfigMode( CliMode.Firewall.ClassMapFirewallConfigMode,
                                  BasicCli.ConfigModeBase ):
   name = "Configure Segment Security Definition"

   def __init__( self, parent, session, vrfName, segmentName ):
      self.session_ = session
      self.cmapName = parent.parent_.getClassName( segmentName )
      gv.firewallConfig.classMap.newMember( self.cmapName )
      param = ( vrfName, segmentName )
      CliMode.Firewall.ClassMapFirewallConfigMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class VrfSegmentPolicyMode( CliMode.Firewall.VrfSegmentPolicyMode,
                            BasicCli.ConfigModeBase ):
   name = "Segment policies"

   def __init__( self, parent, session, vrfName, segmentName ):
      self.session_ = session
      self.cmapName = parent.parent_.getClassName( segmentName )
      param = ( vrfName, segmentName )
      CliMode.Firewall.VrfSegmentPolicyMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class SegmentPolicyMode( CliMode.Firewall.SegmentPolicyMode,
                         BasicCli.ConfigModeBase ):
   name = "Segment policies"

   def __init__( self, parent, session, vrfName, segmentName ):
      self.session_ = session
      self.cmapName = parent.parent_.getClassName( segmentName )
      param = ( vrfName, segmentName )
      CliMode.Firewall.SegmentPolicyMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------
# endregion config mode definition
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region utility functions
# --------------------------------------------------------------------------
def intfVlanDataConfigured( mode, classMap ):
   if not classMap:
      return False
   maybeClassMap = gv.firewallConfig.classMap.get( classMap )
   return maybeClassMap and maybeClassMap.intfVlanData

def forwardingSegmentsAllowed( mode, classMap ):
   return mode.session_.startupConfig() or (
         gv.hwCapability.forwardingSegmentsPolicySupported and
         ( not intfVlanDataConfigured( mode, classMap ) ) )

def forwardingSegmentsConfigured( mode ):
   if mode.session_.startupConfig():
      return False

   if ( gv.hwCapability.forwardingSegmentsPolicySupported and
      isinstance( mode.parent_.parent_, FirewallVrfConfigMode ) ):

      seg = mode.parent_.parent_.getSegment( mode.parent_.segment_ )
      if 'forwarding-segments' in seg.policy:
         return True

   return False

def singleMatchInterfaceSupported( mode ):
   if mode.session_.startupConfig():
      return False

   if gv.hwCapability.singleMatchInterfaceSupported and \
      isinstance( mode.parent_.parent_, FirewallVrfConfigMode ):
      return True

   return False

def onlyDefaultVrfSupported( mode ):
   if mode.session_.startupConfig():
      return False

   if gv.hwCapability.onlyDefaultVrfSupported:
      return True

   return False

def segmentAbsent( configDir, segmentName ):
   if not configDir.segmentDir:
      return True
   return segmentName not in configDir.segmentDir.segment

def getSegmentCount( segDir ):
   if segDir:
      return len( segDir.segment )
   else:
      return 0

def intfInOtherClassMap( intf, thisCmapName ):
   for cmapName, cmapConfig in gv.firewallConfig.classMap.items():
      if cmapName == thisCmapName:
         continue
      if intf.name in cmapConfig.intfVlanData:
         return True
   return False

def vlanSetToRanges( vlanSet ):
   # vlanSet will be a set of ints
   vlanRanges = list()
   if not vlanSet:
      return vlanRanges

   vlanList = sorted( vlanSet )

   curRangeStart = None
   previous = None
   for vlan in vlanList:
      if curRangeStart is None:
         curRangeStart = vlan
      else:
         if vlan != previous + 1:
            # Reached range boundary
            vlanRanges.append( FwVlanRange( curRangeStart, previous ) )
            curRangeStart = vlan
      previous = vlan
   vlanRanges.append( FwVlanRange( curRangeStart, previous ) )
   return vlanRanges

def vlanRangesToSet( intfVlanData ):
   vlanSet = set()
   for vlanRange in intfVlanData.vlanRange:
      vlanSet.update( range( vlanRange.vlanBegin, vlanRange.vlanEnd + 1 ) )
   return vlanSet

def intfVlanInOtherClassMap( intf, vlanSet, thisCmapName ):
   for cmapName, cmapConfig in gv.firewallConfig.classMap.items():
      if cmapName == thisCmapName:
         continue
      intfVlanData = cmapConfig.intfVlanData.get( intf.name )
      if intfVlanData is None:
         continue
      if not intfVlanData.vlanRange:
         # NULL vlanRange means that all vlans are part of the config
         return True
      else:
         otherVlanSet = vlanRangesToSet( intfVlanData )
         if vlanSet.intersection( otherVlanSet ):
            # Overlapping vlan range
            return True
   return False

# --------------------------------------------------------------------------
# endregion utility functions
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'router segment-security' to enter config-router-seg-sec mode
# --------------------------------------------------------------------------
def routerSegmentSecurityHandler( mode, args ):
   childMode = mode.childMode( FirewallConfigMode )
   mode.session_.gotoChildMode( childMode )

def routerSegmentSecurityNoOrDefaultHandler( mode, args ):
   def clearPolicies():
      for policy in list( gv.firewallConfig.policy ):
         if not gv.firewallConfig.policy[ policy ].readonly:
            del gv.firewallConfig.policy[ policy ]
   if gv.firewallConfig.enabled:
      gv.firewallConfig.enabled = False
   clearPolicies()
   gv.firewallConfig.vrf.clear()
   gv.firewallConfig.classMap.clear()
   gv.firewallConfig.segmentDir = None
# --------------------------------------------------------------------------
# endregion 'router segment-security'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'policy <policy-name>' to enter router-seg-sec-policy mode
# --------------------------------------------------------------------------
class PolicyModeCmd( CliCommand.CliCommandClass ):
   syntax = 'policy POLICY_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'policy': segmentSecurityConfigurablePolicyNode,
         'POLICY_NAME': firewallEditablePolicyNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      t0( "%s" % policyName )
      context = Context( policyName, gv.firewallConfig.policy )
      if context.readonly:
         mode.addWarning( "Cannot edit readonly policy " + policyName )
         return
      childMode = mode.childMode( FirewallPolicyConfigMode, context=context )
      childMode.context.modeIs( childMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      t0( "Delete policy %s" % policyName )
      policy = gv.firewallConfig.policy.get( policyName )
      if policy is None:
         return
      if policy.readonly:
         mode.addWarning( "Cannot delete readonly policy " + policyName )
         return
      del gv.firewallConfig.policy[ policyName ]
      # If the delete is done from within the policy config mode, check
      # that it isn't this instance; if it is, abort all changes here
      if isinstance( mode.session_.modeOfLastPrompt(), FirewallPolicyConfigMode ):
         if policyName == mode.session_.modeOfLastPrompt().context.policyName:
            mode.session_.modeOfLastPrompt().context.abort()
            mode.session_.modeOfLastPrompt().aborted = True

FirewallConfigMode.addCommandClass( PolicyModeCmd )
# --------------------------------------------------------------------------
# endregion 'policy <policy-name>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region abort ( from FirewallPolicyConfigMode )
# --------------------------------------------------------------------------
class PolicyAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = { 'abort': 'Exit without committing changes' }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

FirewallPolicyConfigMode.addCommandClass( PolicyAbortCmd )
# --------------------------------------------------------------------------
# endregion abort ( from FirewallPolicyConfigMode )
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'vrf VRF' to enter router-seg-sec-vrf mode
# --------------------------------------------------------------------------
class RouterSegmentSecurityVrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
         'VRF': VrfExprFactory( helpdesc='Enter VRF sub-mode',
                                guard=vrfSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF' ]
      if onlyDefaultVrfSupported( mode ) and vrfName != DEFAULT_VRF:
         mode.addError( 'Only default vrf is supported on this hardware platform' )
         return
      childMode = mode.childMode( FirewallVrfConfigMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRF' ]
      del gv.firewallConfig.vrf[ vrfName ]
      classNamePrefix = '__VRF_%s_segment' % vrfName
      for cmap in gv.firewallConfig.classMap:
         if cmap[ : len( classNamePrefix ) ] == classNamePrefix:
            del gv.firewallConfig.classMap[ cmap ]

FirewallConfigMode.addCommandClass( RouterSegmentSecurityVrfModeCmd )
# --------------------------------------------------------------------------
# endregion 'vrf VRF'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'segment <segment-name>' to enter router-seg-sec-vrf-segment mode
# --------------------------------------------------------------------------
class RouterSegmentSecurityVrfSegmentModeCmd( CliCommand.CliCommandClass ):
   syntax = 'segment SEGMENT_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'segment': 'Configure Segment',
         'SEGMENT_NAME': vrfSegmentNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      segmentName = args[ 'SEGMENT_NAME' ]
      segAbsent = segmentAbsent( gv.firewallConfig.vrf[ mode.vrfName_ ],
                                 segmentName )
      segLimit = gv.hwCapability.maxSegmentsSupported
      segmentCount = 0
      if gv.hwCapability.maxSegmentsLimitsPerVrf:
         segDir = gv.firewallConfig.vrf[ mode.vrfName_ ].segmentDir
         segmentCount = getSegmentCount( segDir )
      else:
         for vrf in gv.firewallConfig.vrf:
            sDir = gv.firewallConfig.vrf[ vrf ].segmentDir
            segmentCount += getSegmentCount( sDir )
      if segAbsent and segmentCount >= segLimit and \
            not mode.session_.startupConfig():
         mode.addError( 'Number of segments cannot exceed the '
               'limit of %d' % segLimit )
         return

      if gv.hwCapability.forwardingSegmentsPolicySupported and \
            segmentName == 'forwarding-segments':
         mode.addError( 'forwarding-segments is an internal keyword and '
               'is not allowed to be used as segment name' )
         return
      childMode = mode.childMode( FirewallVrfSegmentConfigMode,
                                  segmentName=segmentName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segmentName = args[ 'SEGMENT_NAME' ]
      del gv.firewallConfig.vrf[ mode.vrfName_ ].segmentDir.segment[ segmentName ]
      className = mode.getClassName( segmentName )
      del gv.firewallConfig.classMap[ className ]

FirewallVrfConfigMode.addCommandClass( RouterSegmentSecurityVrfSegmentModeCmd )
# --------------------------------------------------------------------------
# endregion 'segment <segment-name>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'segment <segment-name>' to enter router-seg-sec-segment mode
# --------------------------------------------------------------------------
class RouterSegmentSecuritySegmentModeCmd( CliCommand.CliCommandClass ):
   syntax = 'segment SEGMENT_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'segment': 'Configure Segment',
         'SEGMENT_NAME': segmentNode,
   }

   @staticmethod
   def handler( mode, args ):
      segmentName = args[ 'SEGMENT_NAME' ]
      segAbsent = segmentAbsent( gv.firewallConfig, segmentName )
      segLimit = gv.hwCapability.maxSegmentsSupported
      segDir = gv.firewallConfig.segmentDir
      if segAbsent and getSegmentCount( segDir ) >= segLimit and \
            not mode.session_.startupConfig():
         mode.addError( 'Number of segments cannot exceed the '
               'limit of %d' % segLimit )
         return

      childMode = mode.childMode( FirewallSegmentConfigMode,
                                  segmentName=segmentName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segmentName = args[ 'SEGMENT_NAME' ]
      if gv.firewallConfig.segmentDir:
         del gv.firewallConfig.segmentDir.segment[ segmentName ]
         # If all L2 segments are removed, remove the dir
         if not gv.firewallConfig.segmentDir.segment:
            gv.firewallConfig.segmentDir = None
      className = mode.getClassName( segmentName )
      del gv.firewallConfig.classMap[ className ]

FirewallConfigMode.addCommandClass( RouterSegmentSecuritySegmentModeCmd )
# --------------------------------------------------------------------------
# endregion 'segment <segment-name>' to enter router-seg-sec-segment mode
# --------------------------------------------------------------------------

# ---------------------------------------------------------------------------------
# region 'definition' to enter 'config-router-seg-sec-vrf-segment-def' mode
#    or
# 'definition' to enter 'config-router-seg-sec-segment-def' mode
# ---------------------------------------------------------------------------------
class SegmentDefinitionModeCmd( CliCommand.CliCommandClass ):
   syntax = 'definition'
   noOrDefaultSyntax = syntax

   data = {
         'definition': 'Configure segment definition',
   }

   @staticmethod
   def handler( mode, args ):
      className = mode.parent_.getClassName( mode.segment_ )
      segment = mode.parent_.getSegment( mode.segment_ )
      vrfName = mode.parent_.getVrfName()
      childModeType = mode.parent_.getConfigChildModeType()
      gv.firewallConfig.classMap.newMember( className )
      segment.classMap = className
      childMode = mode.childMode( childModeType,
                                  vrfName=vrfName,
                                  segmentName=mode.segment_ )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      className = mode.parent_.getClassName( mode.segment_ )
      segment = mode.parent_.getSegment( mode.segment_ )
      segment.classMap = ''
      segment.ipv4PrefixList = ''
      del gv.firewallConfig.classMap[ className ]

FirewallVrfSegmentConfigMode.addCommandClass( SegmentDefinitionModeCmd )
FirewallSegmentConfigMode.addCommandClass( SegmentDefinitionModeCmd )
# ---------------------------------------------------------------------------------
# endregion 'definition' to enter 'config-router-seg-sec-vrf-segment-def' mode
#    or
# 'definition' to enter 'config-router-seg-sec-segment-def' mode
# ---------------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'policies' to enter router-seg-sec-vrf-segment-policies mode
#   or
# 'policies' to enter router-seg-sec-segment-policies mode
# --------------------------------------------------------------------------
class SegmentPoliciesModeCmd( CliCommand.CliCommandClass ):
   syntax = 'policies'
   noOrDefaultSyntax = syntax

   data = {
         'policies': 'Segment policies',
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = mode.parent_.getVrfName()
      childModeType = mode.parent_.getPolicyChildModeType()
      childMode = mode.childMode( childModeType,
                                  vrfName=vrfName,
                                  segmentName=mode.segment_ )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segment = mode.parent_.getSegment( mode.segment_ )
      segment.policy.clear()

FirewallVrfSegmentConfigMode.addCommandClass( SegmentPoliciesModeCmd )
FirewallSegmentConfigMode.addCommandClass( SegmentPoliciesModeCmd )
# --------------------------------------------------------------------------
# endregion 'policies' to enter router-seg-sec-vrf-segment-policies mode
#   or
# 'policies' to enter router-seg-sec-segment-policies mode
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region commands in 'policy <policy-name>' sub-mode
# [ no | <sequence-no> ] application <app-name>
#    action <forward|drop|<redirect next-hop IPADDR>> [stateless] [ log ]
# no <sequence-no>
# --------------------------------------------------------------------------
class ApplicationActionCmd( CliCommand.CliCommandClass ):
   syntax = '[ SEQUENCE_NUM ] application APPLICATION_NAME' \
            ' ( action ( forward \
                       | drop \
                       | ( redirect next-hop IPADDR ) ) \
              [ stateless ] ) [ log ]'
   noOrDefaultSyntax = '( SEQUENCE_NUM | ' \
        '( [ SEQUENCE_NUM ]application APPLICATION_NAME ) )' \
            ' [ ( action ( forward \
                         | drop \
                         | ( redirect next-hop IPADDR ) ) \
                [ stateless ] ) [ log ] ]'
   data = {
      'SEQUENCE_NUM': sequenceNumMatcher,
      'application': 'Application name',
      'APPLICATION_NAME': applicationNameMatcher,
      'action': 'Specify an action',
      'forward': 'Allow an application',
      'redirect': redirectActionNode,
      'next-hop': 'Redirect to Nexthop',
      'IPADDR': IpAddrMatcher( helpdesc='Nexthop Address' ),
      'stateless': statelessActionNode,
      'drop': 'Drop an application',
      'log': logActionNode,
   }
   @staticmethod
   def handler( mode, args ):
      serviceName = args[ 'APPLICATION_NAME' ]
      seqNum = args.get( 'SEQUENCE_NUM' )
      policy = mode.context.tmpPolicy

      if seqNum == 'default':
         return
      assert not ( isinstance( seqNum, bool ) and seqNum is True )
      if seqNum is None:
         seqNum = policy.lastSeqNum + 10
      if seqNum > MAX_SEQ or seqNum < 1:
         mode.addError( "Sequence number out of range [1..%s]" % str( MAX_SEQ ) )
         return

      if seqNum in policy.rule and \
            policy.rule[ seqNum ].serviceName != serviceName:
         mode.addError( "Sequence %d is already associated with application %s" %
               ( seqNum, policy.rule[ seqNum ].serviceName ) )
         return

      stateless = 'stateless' in args
      if ( ( not stateless ) and
           ( not gv.hwCapability.statefulActionSupported ) and
           mode.session_.guardsEnabled() ):
         mode.addError( "Stateful action not supported on this platform" )
         return

      # cleanup old sequence number associated with serviceName from
      if serviceName in policy.serviceToSeqnum:
         oldSeqnum = policy.serviceToSeqnum[ serviceName ]
         if oldSeqnum in policy.rule:
            del policy.rule[ oldSeqnum ]
      if seqNum in policy.rule:
         del policy.rule[ seqNum ]

      policy.serviceToSeqnum[ serviceName ] = seqNum
      # action redirect supports only on stateless mode
      redirect = 'redirect' in args
      if redirect:
         action = 'statelessRedirect'
         nexthop = Arnet.IpGenAddr( args[ 'IPADDR' ] )
      else:
         nexthop = Arnet.IpGenAddr()
         action = 'deny' if 'drop' in args else 'permit'
         if 'stateless' in args:
            action = 'stateless' + action.title()
      logValue = 'log' in args
      rule = FwRuleDefinition( seqNum, serviceName=serviceName,
         action=action, log=logValue, nexthop=nexthop )
      policy.rule.addMember( rule )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      seqNum = args.get( 'SEQUENCE_NUM' )
      policy = mode.context.tmpPolicy
      if not seqNum:
         serviceName = args[ 'APPLICATION_NAME' ]
         del policy.serviceToSeqnum[ serviceName ]
         for seq in list( policy.rule ):
            if policy.rule[ seq ].serviceName == serviceName:
               del policy.rule[ seq ]
               break
         return
      if seqNum not in policy.rule:
         return
      serviceName = policy.rule[ seqNum ].serviceName
      del policy.serviceToSeqnum[ serviceName ]
      del policy.rule[ seqNum ]

FirewallPolicyConfigMode.addCommandClass( ApplicationActionCmd )
# --------------------------------------------------------------------------
# endregion commands in 'policy <policy-name>' sub-mode
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region '[ no | default ] shutdown'
# --------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax

   data = {
      'shutdown': 'Shutdown Segment Security'
   }

   @staticmethod
   def handler( mode, args ):
      gv.firewallConfig.enabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.firewallConfig.enabled = True

FirewallConfigMode.addCommandClass( ShutdownCmd )
# --------------------------------------------------------------------------
# endregion '[ no | default ] shutdown'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region '[ no | default ] segment policy policy-drop-all default'
# --------------------------------------------------------------------------
class SegmentPolicyDefaultDrop( CliCommand.CliCommandClass ):
   syntax = 'segment policy policy-drop-all default'
   noOrDefaultSyntax = syntax

   data = {
      'segment': 'Configure Segment',
      'policy': policyNodeForDefaultPolicyDrop,
      'policy-drop-all': 'Configure policy drop',
      'default': 'Default policy drop behaviour',
   }

   @staticmethod
   def handler( mode, args ):
      gv.firewallConfig.defaultPolicyDrop = 'enable'

   @staticmethod
   def noHandler( mode, args ):
      gv.firewallConfig.defaultPolicyDrop = 'disable'

   defaultHandler = handler

FirewallConfigMode.addCommandClass( SegmentPolicyDefaultDrop )
# --------------------------------------------------------------------------
# endregion '[ no | default ] segment policy policy-drop-all default'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region '[ no ] from <segment-name> policy <policy-name>'
# --------------------------------------------------------------------------
class SegmentPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'from SEGMENT_NAME policy POLICY_NAME'
   noOrDefaultSyntax = 'from SEGMENT_NAME [ policy POLICY_NAME ]'

   data = {
         'from': 'Associate a policy from Segment',
         'SEGMENT_NAME': vrfSegmentNameMatcher,
         'policy': 'Policy name',
         'POLICY_NAME': firewallPolicyNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      fromSegmentName = args[ 'SEGMENT_NAME' ]
      segmentName = mode.parent_.segment_
      if segmentName == fromSegmentName and (
            not gv.hwCapability.intraSegmentPolicySupported ):
         mode.addError( "Intra segment policy is not allowed." )
         return

      segment = mode.parent_.parent_.getSegment( segmentName )
      if not mode.session_.startupConfig() and \
            forwardingSegmentsAllowed( mode, segment.classMap ):
         mode.addError( "Only forwarding-segments is allowed "
                         "as from segment in the absence of match interface" )
         return
      policyName = args[ 'POLICY_NAME' ]
      segment.policy[ fromSegmentName ] = policyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segment = mode.parent_.parent_.getSegment( mode.parent_.segment_ )
      fromSegmentName = args[ 'SEGMENT_NAME' ]
      del segment.policy[ fromSegmentName ]

VrfSegmentPolicyMode.addCommandClass( SegmentPolicyCmd )
SegmentPolicyMode.addCommandClass( SegmentPolicyCmd )
# --------------------------------------------------------------------------
# endregion '[ no ] from <segment-name> policy <policy-name>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region '[ no | default ] fallback policy <policy-name>'
# --------------------------------------------------------------------------
class SegmentFallbackPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'fallback policy POLICY_NAME'
   noOrDefaultSyntax = 'fallback ...'

   data = {
         'fallback': fallbackPolicyNode,
         'policy': 'Policy name',
         'POLICY_NAME': firewallPolicyNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      segmentName = mode.parent_.segment_
      segment = mode.parent_.parent_.getSegment( segmentName )
      policyName = args.get( 'POLICY_NAME' )
      segment.fallbackPolicy = policyName

   noOrDefaultHandler = handler

VrfSegmentPolicyMode.addCommandClass( SegmentFallbackPolicyCmd )
SegmentPolicyMode.addCommandClass( SegmentFallbackPolicyCmd )

# --------------------------------------------------------------------------
# endregion '[ no | default ] fallback policy <policy-name>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region '[ no ] from forwarding-segments policy <policy-name>'
# --------------------------------------------------------------------------
class SegmentPolicyFwdSegmentsCmd( CliCommand.CliCommandClass ):
   syntax = 'from forwarding-segments policy POLICY_NAME'
   noOrDefaultSyntax = 'from forwarding-segments [ policy POLICY_NAME ]'

   data = {
         'from': 'Associate a policy from Segment',
         'forwarding-segments': forwardingSegmentsNode,
         'policy': 'Policy name',
         'POLICY_NAME': firewallPolicyNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      fromSegmentName = 'forwarding-segments'
      segmentName = mode.parent_.segment_
      segment = mode.parent_.parent_.getSegment( segmentName )
      if not forwardingSegmentsAllowed( mode, segment.classMap ):
         mode.addError( "forwarding-segments is allowed only "
                         "in the absence of match interface" )
         return
      policyName = args[ 'POLICY_NAME' ]
      segment.policy[ fromSegmentName ] = policyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segment = mode.parent_.parent_.getSegment( mode.parent_.segment_ )
      fromSegmentName = 'forwarding-segments'
      del segment.policy[ fromSegmentName ]

VrfSegmentPolicyMode.addCommandClass( SegmentPolicyFwdSegmentsCmd )
# --------------------------------------------------------------------------
# endregion '[ no ] from forwarding-segments policy <policy-name>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'match interface'
# segment <segment-name>
#    definition
#       match interface <intf>+
# --------------------------------------------------------------------------
class MatchInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'match interface { INTF_LIST }'
   noOrDefaultSyntax = syntax
   data = {
         'match': matchNode,
         'interface': segmentSecurityInterfaceNode,
         'INTF_LIST': intfMatcher
   }

   @staticmethod
   def handler( mode, args ):
      intfs = args[ 'INTF_LIST' ]

      # Validate command
      for intf in intfs:
         if intfInOtherClassMap( intf, mode.cmapName ):
            mode.addError( 'Interface is part of another segment' )
            return

      if forwardingSegmentsConfigured( mode ):
         mode.addError( 'Segment policy contains forwarding-segments '
                        'which is mutually exclusive to match interface.' )
         return

      if singleMatchInterfaceSupported( mode ):
         if len( intfs ) > 1:
            mode.addError( 'Only one interface can be supplied '
                           'on this hardware platform.' )
            return

         if intfs[ 0 ].name not in \
            gv.firewallConfig.classMap[ mode.cmapName ].intfVlanData:
            gv.firewallConfig.classMap[ mode.cmapName ].intfVlanData.clear()

      oldIntfColl = gv.firewallConfig.classMap[ mode.cmapName ].intfVlanData
      for intf in intfs:
         if intf.name in oldIntfColl:
            if not oldIntfColl[ intf.name ].vlanRange:
               continue
            # Remove any stale vlan config
            oldIntfColl[ intf.name ].vlanRange.clear()
         else:
            gv.firewallConfig.classMap[ mode.cmapName ].intfVlanData.newMember(
               intf.name )
         if isinstance( mode.parent_.parent_, FirewallVrfConfigMode ):
            vrfName = mode.parent_.parent_.vrfName
            if vrfName == 'default':
               continue
            l3c = gv.l3ConfigDir.intfConfig.get( intf.name )
            if not l3c or l3c.vrf != vrfName:
               mode.addWarning(
                     f"Interface {intf.name} not in VRF {vrfName}" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfs = args[ 'INTF_LIST' ]
      for intf in intfs:
         del gv.firewallConfig.classMap[ mode.cmapName ].intfVlanData[ intf.name ]

ClassMapFirewallVrfConfigMode.addCommandClass( MatchInterfaceCmd )
ClassMapFirewallConfigMode.addCommandClass( MatchInterfaceCmd )
# --------------------------------------------------------------------------
# endregion 'match interface'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'match interface <intf> vlan [ ( add | remove ) ] <vlan_set>'
# segment <segment-name>
#    definition
#       match interface <intf> vlan [ ( add | remove ) ] <vlan_set>
# --------------------------------------------------------------------------
class MatchInterfaceVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'match interface INTF_NAME vlan [ (add | remove ) ] VLAN_SET'
   noOrDefaultSyntax = syntax
   data = {
      'match': matchNode,
      'interface': segmentSecurityInterfaceNode,
      'INTF_NAME': intfMatcher,
      'vlan': 'Specify VLAN',
      'add': 'Add VLAN',
      'remove': 'Remove VLAN',
      'VLAN_SET': vlanSetMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF_NAME' ]
      updateSet = args[ 'VLAN_SET' ]

      if 'remove' not in args and \
         intfVlanInOtherClassMap( intf, updateSet, mode.cmapName ):
         mode.addError( 'Interface VLAN range is part of another segment' )
         return

      classMap = gv.firewallConfig.classMap[ mode.cmapName ]
      intfVlanDataConfig = None
      curSet = set()
      if intf.name in classMap.intfVlanData:
         intfVlanDataConfig = classMap.intfVlanData[ intf.name ]
         curSet = vlanRangesToSet( intfVlanDataConfig )

      newSet = set()
      if 'add' in args:
         newSet = curSet.union( updateSet )
      elif 'remove' in args:
         newSet = curSet.difference( updateSet )
      else:
         newSet = updateSet

      if newSet:
         newVlanRanges = vlanSetToRanges( newSet )
         if not intfVlanDataConfig:
            intfVlanDataConfig = classMap.newIntfVlanData( intf.name )
         else:
            for vlanRange in intfVlanDataConfig.vlanRange:
               if vlanRange not in newVlanRanges:
                  # Remove stale range entry
                  intfVlanDataConfig.vlanRange.remove( vlanRange )

         for vlanRange in newVlanRanges:
            # New range entry, add
            intfVlanDataConfig.vlanRange.add( vlanRange )
      # If all vlans have been removed, remove the interface itself from config
      # Only remove the interface, if the original vlan set was not empty.
      # Running vlan remove on an interface originally configured without vlans
      # shouldn't remove the interface from config
      elif intfVlanDataConfig and curSet:
         del classMap.intfVlanData[ intf.name ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      classMap = gv.firewallConfig.classMap[ mode.cmapName ]
      intf = args[ 'INTF_NAME' ]
      if intf.name in classMap.intfVlanData:
         del classMap.intfVlanData[ intf.name ]

ClassMapFirewallConfigMode.addCommandClass( MatchInterfaceVlanCmd )
# --------------------------------------------------------------------------
# endregion 'match interface <intf> vlan [ ( add | remove ) ] <vlan_set>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'match prefix-ipv4 <match-list-prefix-ipv4-name>'
# segment <segment-name>
#    definition
#       match prefix-ipv4 <match-list-prefix-ipv4-name>
# --------------------------------------------------------------------------
class MatchIpv4PrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'match prefix-ipv4 IPV4_PREFIX_LIST'
   noOrDefaultSyntax = syntax

   data = {
         'match': matchNode,
         'prefix-ipv4': 'Match Ipv4 match list',
         'IPV4_PREFIX_LIST': ipv4PrefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      listName = args[ 'IPV4_PREFIX_LIST' ]
      fwConfig = gv.firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      segment.usePrefixListForIpv4 = False
      segment.ipv4PrefixList = listName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      listName = args[ 'IPV4_PREFIX_LIST' ]
      fwConfig = gv.firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      if segment.ipv4PrefixList == listName and \
            not segment.usePrefixListForIpv4:
         segment.ipv4PrefixList = ''

ClassMapFirewallVrfConfigMode.addCommandClass( MatchIpv4PrefixCmd )
# --------------------------------------------------------------------------
# endregion 'match prefix-ipv4 <match-list-prefix-ipv4-name>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'match prefix-ipv6 <match-list prefix-ipv6 name>'
# segment <segment-name>
#    definition
#       match prefix-ipv6 <match-list prefix-ipv6 name>
# --------------------------------------------------------------------------
class MatchIpv6PrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'match prefix-ipv6 IPV6_PREFIX_LIST'
   noOrDefaultSyntax = syntax

   data = {
         'match': matchNode,
         'prefix-ipv6': segmentSecurityIpv6PrefixNode,
         'IPV6_PREFIX_LIST': ipv6PrefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      listName = args[ 'IPV6_PREFIX_LIST' ]
      fwConfig = gv.firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      segment.usePrefixListForIpv6 = False
      segment.ipv6PrefixList = listName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      listName = args[ 'IPV6_PREFIX_LIST' ]
      fwConfig = gv.firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      if segment.ipv6PrefixList == listName and \
            not segment.usePrefixListForIpv6:
         segment.ipv6PrefixList = ''

ClassMapFirewallVrfConfigMode.addCommandClass( MatchIpv6PrefixCmd )
# --------------------------------------------------------------------------
# endregion 'match prefix-ipv6 <match-list prefix-ipv6 name>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'match covered prefix-list ipv4 <prefix-list>'
# segment <segment-name>
#    definition
#       match covered prefix-list ipv4 <prefix-list>
# --------------------------------------------------------------------------
class MatchCoveredIpv4PrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'match covered prefix-list ipv4 IPV4_PREFIX_LIST'
   noOrDefaultSyntax = syntax

   data = {
         'match': matchNode,
         'covered': coveredNode,
         'prefix-list': prefixListNode,
         'ipv4': 'Match Ipv4 prefix list',
         'IPV4_PREFIX_LIST': ipv4PrefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      listName = args[ 'IPV4_PREFIX_LIST' ]
      fwConfig = gv.firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      segment.usePrefixListForIpv4 = True
      segment.ipv4PrefixList = listName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      listName = args[ 'IPV4_PREFIX_LIST' ]
      fwConfig = gv.firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      if segment.ipv4PrefixList == listName and segment.usePrefixListForIpv4:
         segment.usePrefixListForIpv4 = False
         segment.ipv4PrefixList = ''

ClassMapFirewallVrfConfigMode.addCommandClass( MatchCoveredIpv4PrefixCmd )
# --------------------------------------------------------------------------
# endregion 'match covered prefix-list ipv4 <prefix-list>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'match covered prefix-list ipv6 <prefix-list>'
# --------------------------------------------------------------------------
class MatchCoveredIpv6PrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'match covered prefix-list ipv6 IPV6_PREFIX_LIST'
   noOrDefaultSyntax = syntax

   data = {
         'match': matchNode,
         'covered': coveredNode,
         'prefix-list': prefixListNode,
         'ipv6': 'Match Ipv6 prefix list',
         'IPV6_PREFIX_LIST': ipv6PrefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      listName = args[ 'IPV6_PREFIX_LIST' ]
      fwConfig = gv.firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      segment.usePrefixListForIpv6 = True
      segment.ipv6PrefixList = listName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      listName = args[ 'IPV6_PREFIX_LIST' ]
      fwConfig = gv.firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      if segment.ipv6PrefixList == listName and segment.usePrefixListForIpv6:
         segment.usePrefixListForIpv6 = False
         segment.ipv6PrefixList = ''

ClassMapFirewallVrfConfigMode.addCommandClass( MatchCoveredIpv6PrefixCmd )
# --------------------------------------------------------------------------
# endregion 'match covered prefix-list ipv6 <prefix-list>'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'forwarding-type [ routed | bridged ]'
# --------------------------------------------------------------------------
class ForwardingTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'forwarding-type ( routed | bridged )'
   noOrDefaultSyntax = 'forwarding-type [ routed | bridged ]'

   data = {
      'forwarding-type': segmentSecurityTrafficInspectPolicyNode,
      'routed': segmentSecurityRoutedSupportedNode,
      'bridged': segmentSecurityBridgedSupportedNode
   }

   @staticmethod
   def handler( mode, args ):
      forwardTypeMatch = ForwardingTypeMatch()
      if "routed" in args:
         forwardTypeMatch.routed = True
      elif "bridged" in args:
         forwardTypeMatch.bridged = True
      gv.firewallConfig.forwardType = forwardTypeMatch

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      forwardTypeMatch = ForwardingTypeMatch()
      gv.firewallConfig.forwardType = forwardTypeMatch

FirewallConfigMode.addCommandClass( ForwardingTypeCmd )
# --------------------------------------------------------------------------
# endregion 'forwarding-type [ routed | bridged ]'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'clear segment-security sessions [ VRF ]'
# --------------------------------------------------------------------------
def clearSegmentSecuritySessionsHandler( mode, args ):
   vrfName = args.get( 'VRF' )
   for hook in clearFirewallSessionHook.extensions():
      hook( mode, vrfName )
# --------------------------------------------------------------------------
# endregion 'clear segment-security sessions [ VRF ]'
# --------------------------------------------------------------------------

# --------------------------------------------------------------------------
# region 'clear segment-security counters [ VRF ]'
# --------------------------------------------------------------------------
def clearSegmentSecurityCountersHandler( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName is None and not gv.hwCapability.intfVlanSupported:
      vrfName = DEFAULT_VRF
   for hook in clearFirewallCountersHook.extensions():
      hook( mode, vrfName )
   if clearFirewallCountersHook.extensions():
      return
   # default snapshot implementation if no hooks
   if vrfName is None:
      vrfCounters = gv.firewallCounters.l2CountersDir
      if vrfCounters is None:
         gv.firewallCountersSnapshot.l2CountersDir = None
      else:
         gv.firewallCountersSnapshot.l2CountersDirIs1().takeSnapshot( vrfCounters )
   else:
      vrfCounters = gv.firewallCounters.vrf.get( vrfName )
      if vrfCounters is None:
         del gv.firewallCountersSnapshot.vrf[ vrfName ]
      else:
         gv.firewallCountersSnapshot.vrfIs1( vrfName ).takeSnapshot( vrfCounters )
# --------------------------------------------------------------------------
# endregion 'clear segment-security counters [ VRF ]'
# --------------------------------------------------------------------------

def Plugin( entityManager ):
   gv.firewallConfig = ConfigMount.mount(
         entityManager, 'firewall/config/cli', 'Firewall::Config', 'w' )
   gv.firewallCounters = LazyMount.mount(
         entityManager, 'firewall/counters', 'Firewall::Counters', 'r' )
   gv.firewallCountersSnapshot = LazyMount.mount(
         entityManager, 'firewall/snapshot/counters', 'Firewall::Counters', 'w' )
   gv.hwCapability = LazyMount.mount(
         entityManager, 'firewall/hw/capability', 'Firewall::HwCapability', 'r' )
   gv.l3ConfigDir = LazyMount.mount(
         entityManager, 'l3/intf/config', 'L3::Intf::ConfigDir', 'r' )
   gv.matchListCapability = LazyMount.mount(
         entityManager, 'matchlist/hw/capability', 'MatchList::HwCapability', 'r' )
   gv.matchListConfig = LazyMount.mount(
         entityManager, 'matchlist/config/cli', 'MatchList::Config', 'r' )
