# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Bool, Dict, Enum, Int, List, Model, Str
from IntfModels import Interface
from TableOutput import createTable
from TableOutput import Format
from ArnetModel import IpGenericAddress, IpGenericPrefix
from operator import attrgetter
from textwrap import wrap
import Arnet
from CliPlugin import IntfCli
import copy

actionMap = {
      'permit': 'forward',
      'statelessPermit': 'forward stateless',
      'deny': 'drop',
      'statelessDeny': 'drop stateless',
}

_dashMeaning = "-: Packets not matching other policy (fallback)"
_naMeaning = "n/a: No relevant policy present"

class FwPolicyInfo( Model ):
   class Policy( Model ):
      class PolicyDef( Model ):
         serviceName = Str( "Application name" )
         action = Enum( values=( 'permit', 'statelessPermit',
                                 'deny', 'statelessDeny', 'statelessRedirect' ),
                        help="Action to be applied on application" )
         nexthop = IpGenericAddress( help="Nexthop IP address", optional=True )
         log = Bool( help="Whether logging enabled" )
      policyDefs = Dict( keyType=int, valueType=PolicyDef,
                         help="A mapping of sequence no to policy rule" )
      readonly = Bool( help="Whether policy is readonly" )
   policies = Dict( keyType=str, valueType=Policy,
                    help="Segment Security Policy" )

   def render( self ):
      for policy in sorted( self.policies ):
         if self.policies[ policy ].readonly:
            print( "policy: %s [readonly]" % policy )
         else:
            print( "policy: %s" % policy )
         policyDefs = self.policies[ policy ].policyDefs
         for seqno in sorted( policyDefs ):
            ruleObj = policyDefs[ seqno ]
            logStr = ""
            if ruleObj.log:
               logStr = " log"
            if ruleObj.action == 'statelessRedirect':
               print( f"   {seqno} application {ruleObj.serviceName} action"
                  f" redirect next-hop {ruleObj.nexthop} stateless{logStr}" )
            else:
               print( f"   {seqno} application {ruleObj.serviceName} action"
                  f" {actionMap[ ruleObj.action ]}{logStr}" )

ipProtocolNumberToName = {
           0: 'ip',
           1: 'icmp',
           2: 'igmp',
           6: 'tcp',
           17: 'udp',
           41: 'ipv6',
           46: 'rsvp',
           47: 'gre',
           50: 'esp',
           51: 'ah',
           58: 'icmpv6',
           89: 'ospf',
           103: 'pim',
           112: 'vrrp',
           124: 'isis',
           132: 'sctp',
}

class FwServiceInfo( Model ):
   __revision__ = 3
   class Service( Model ):
      class ServiceDef( Model ):
         class PortRange( Model ):
            start = Int( help='Port range start' )
            end = Int( help='Port range end' )

            def formatStr( self ):
               if self.start != self.end:
                  return '%d-%d' % ( self.start, self.end )
               return str( self.start )
         allProtocols = Bool( help="Definitions supports all IP protocols" )
         ipProtocol = Int( help="IP protocol number", optional=True )
         dstPort = List( valueType=PortRange, help='destination port' )
         srcPort = List( valueType=PortRange, help='source port' )

         def protoName( self ):
            if self.allProtocols:
               protoName = 'all'
            else:
               protoName = ipProtocolNumberToName.get( self.ipProtocol )
            return protoName

         def renderServiceDef( self ):
            protoName = self.protoName()
            def renderPortRangeList( name, portRangeList ):
               portRanges = []
               for portRange in portRangeList:
                  portRanges.append( portRange.formatStr() )
               portRangesStr = ', '.join( portRanges )
               if portRangesStr and portRangesStr != '0-65535':
                  print( f'      {name}: {portRangesStr}' )
            protoStr = protoName if protoName else '%d' % ( self.ipProtocol )
            print( '   protocol: %s' % ( protoStr ) )
            if protoName in [ 'tcp', 'udp', 'all' ]:
               renderPortRangeList( 'source-port', self.srcPort )
               renderPortRangeList( 'destination-port', self.dstPort )

      serviceDef = List( valueType=ServiceDef,
                         help="Application definition" )
      srcPrefix = List( valueType=IpGenericPrefix, help='Source IP Prefix' )
      dstPrefix = List( valueType=IpGenericPrefix, help='Destination IP Prefix' )

   services = Dict( keyType=str, valueType=Service,
                    help="Segment Security applications" )

   def degrade( self, dictRepr, revision ):
      # Revision 2 and earlier had serviceDef as a dict indexed by
      # protocol names, instead of a list

      # Revision 1 had only dstPort in serviceDef, and it was named as port,
      # and was a list of ports and not a list of port ranges
      # Revision 1 didn't have prefixes in service either

      # Make a copy of dictRepr to iterate over while modifying dictRepr
      services = copy.deepcopy( dictRepr[ 'services' ] )

      def degradeServiceDef( serviceDefDictRepr, serviceDef ):

         def portRangesToPortList( portRangeList ):
            portSet = set()
            for portRange in portRangeList:
               thisPortRange = list( range( portRange[ 'start' ],
                                      portRange[ 'end' ] + 1 ) )
               portSet.update( thisPortRange )
            return list( portSet )

         protoName = None
         if serviceDef[ 'allProtocols' ]:
            protoName = 'all'
         elif 'ipProtocol' in serviceDef:
            protoNum = serviceDef[ 'ipProtocol' ]
            protoName = ipProtocolNumberToName.get( protoNum )

         # Earlier CLI models only supported named protocols
         if protoName:
            serviceDefDictRepr[ protoName ] = {}
            serviceDefDictReprEntry = serviceDefDictRepr[ protoName ] = {}
            serviceDefDictReprEntry[ 'proto' ] = protoName
            if revision == 1:
               if 'dstPort' in serviceDef:
                  serviceDefDictReprEntry[ 'port' ] = \
                        portRangesToPortList( serviceDef[ 'dstPort' ] )
            else:
               for attr in [ 'dstPort', 'srcPort' ]:
                  if attr in serviceDef:
                     serviceDefDictReprEntry[ attr ] = \
                           copy.deepcopy( serviceDef[ attr ] )

      for serviceName, service in services.items():
         serviceDictRepr = dictRepr[ 'services' ][ serviceName ]
         if revision == 1:
            serviceDictRepr.pop( 'dstPrefix', None )
            serviceDictRepr.pop( 'srcPrefix', None )
         serviceDictRepr[ 'serviceDef' ] = {}
         serviceDefDictRepr = serviceDictRepr[ 'serviceDef' ]
         for serviceDef in service[ 'serviceDef' ]:
            degradeServiceDef( serviceDefDictRepr, serviceDef )
      return dictRepr

   def render( self ):
      for serviceName in sorted( self.services ):
         print( "application: %s" % serviceName )
         service = self.services[ serviceName ]

         if service.serviceDef:
            # Display named protocols first, then numbered ones
            # Sort by name and number respectively
            for protoObj in sorted( service.serviceDef,
                                    key=lambda s: ( s.protoName() is None,
                                                    s.protoName(),
                                                    s.ipProtocol ) ):
               protoObj.renderServiceDef()

         if service.srcPrefix:
            srcPrefixList = [
                  prefix.stringValue for prefix in sorted( service.srcPrefix ) ]
            srcPrefixStr = ', '.join( srcPrefixList )
            print( '   source-prefix: %s' % srcPrefixStr )
         if service.dstPrefix:
            dstPrefixList = [
                  prefix.stringValue for prefix in sorted( service.dstPrefix ) ]
            dstPrefixStr = ', '.join( dstPrefixList )
            print( '   destination-prefix: %s' % dstPrefixStr )

class VlanRangeModel( Model ):
   startVlanId = Int( help="Start of VLAN range" )
   endVlanId = Int( help="End of VLAN range" )

class InterfaceVlanRange( Model ):
   interface = Interface( help="Interface" )
   vlanRanges = List( valueType=VlanRangeModel,
                       help="List of VLAN ranges associated with this "
                            "interface for this segment" )

class Segment( Model ):
   __revision__ = 2
   classMap = Str( help="Class Map of type segment-security"
                   " associated with this segment" )
   interfaces = List( valueType=Interface,
                      help="Interfaces which are part of this segment" )
   interfaceVlans = List( valueType=InterfaceVlanRange,
                          help="Interface plus VLAN ranges which "
                               "are part of this segment",
                          sinceRevision=2,
                          optional=True )
   fromSegments = Dict( keyType=str, valueType=str,
                     help="A mapping from source segment name to policy" )
   ipv4PrefixListName = Str( help="Name of IPv4 prefix list" )
   ipv6PrefixListName = Str( help="Name of IPv6 prefix list" )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         if 'interfaceVlans' in dictRepr:
            del dictRepr[ 'interfaceVlans' ]
      return dictRepr

class FirewallInfo( Model ):
   __revision__ = 2

   class Vrf( Model ):
      __revision__ = 2
      enabled = Bool( help="Indicates whether segment-security is enabled" )
      ipv6PrefixSupported = Bool(
            help="IPv6 prefix supoort is enabled" )
      vrfExist = Bool( help="Whether VRF exist" )

      def degrade( self, dictRepr, revision ):
         if revision == 1:
            segments = copy.deepcopy( dictRepr[ 'segments' ] )
            for segment in segments.values():
               segment[ 'fromAreas' ] = copy.deepcopy( segment[ 'fromSegments' ] )
               del segment[ 'fromSegments' ]
            dictRepr[ 'areas' ] = segments
            del dictRepr[ 'segments' ]
         return dictRepr

      segments = Dict( keyType=str, valueType=Segment,
                       help="A mapping of segment name to segment information " )

   vrfs = Dict( keyType=str, valueType=Vrf,
                help="A mapping from VRF to segment security" )

   segments = Dict( keyType=str, valueType=Segment,
                    help="A mapping of Segment name to segment information" )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         del dictRepr[ 'segments' ]
      return dictRepr

   def _createInterfacesString( self, interfaces, interfaceVlans ):
      interfaceSet = [ ]
      currInterfaceString = ''
      for intf in Arnet.sortIntf( interfaces ):
         shortInftName = IntfCli.Intf.getShortname( intf )
         if len( currInterfaceString + shortInftName ) > 16:
            interfaceSet.append( currInterfaceString )
            currInterfaceString = ''
         currInterfaceString += shortInftName + ' '

      # Don't print an empty line if we only have interfaceVlans
      if interfaces or not interfaceVlans:
         interfaceSet.append( currInterfaceString )

      def vlanIntfKey( vlan ):
         return Arnet.intfNameKey( vlan.interface.stringValue )

      currInterfaceString = ''
      for iv in sorted( interfaceVlans, key=vlanIntfKey ):
         shortIntfName = IntfCli.Intf.getShortname( iv.interface )
         vlanRanges = []
         for vlanRange in sorted( iv.vlanRanges, key=attrgetter( 'startVlanId' ) ):
            if vlanRange.startVlanId == vlanRange.endVlanId:
               vlanRanges.append( str( vlanRange.startVlanId ) )
            else:
               vlanRanges.append( '%d-%d' % ( vlanRange.startVlanId,
                                              vlanRange.endVlanId ) )
         vlanRangesString = ', '.join( vlanRanges )
         intfVlanRangesString = shortIntfName + " VLAN: " + vlanRangesString
         wrapped = wrap( intfVlanRangesString, 24,
                         break_long_words=False )
         interfaceSet.extend( wrapped )
      return interfaceSet

   def renderVrf( self, headings, vrfName ):
      assert vrfName
      vrfModel = self.vrfs[ vrfName ]
      table = createTable( headings )
      f1 = Format( justify='left' )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )
      table.formatColumns( *( [ f1 ] * len( headings ) ) )

      segmentsList = sorted( vrfModel.segments )
      segmentsInfo = vrfModel.segments

      def addNewRow( segName, intf, v4List, v6List, fromSeg, policy ):
         if vrfModel.ipv6PrefixSupported:
            table.newRow( segName, intf, v4List, v6List, fromSeg, policy )
         else:
            table.newRow( segName, intf, v4List, fromSeg, policy )

      for segmentName in segmentsList:
         segment = segmentsInfo[ segmentName ]
         intfSet = self._createInterfacesString( segment.interfaces,
                                                 segment.interfaceVlans )
         noOfIntfLine = len( intfSet )
         if not segment.fromSegments:
            currentIntfLine = 0
            while currentIntfLine < noOfIntfLine:
               if currentIntfLine == 0:
                  # print first intf line
                  addNewRow( segmentName, intfSet[ currentIntfLine ],
                             segment.ipv4PrefixListName,
                             segment.ipv6PrefixListName,
                             '', '' )
               else:
                  table.newRow( segmentName, intfSet[ currentIntfLine ],
                               '', '', '', '' )
               currentIntfLine = currentIntfLine + 1
         else:
            currentIntfLine = -1
            firstLine = True
            for fromSegment in _sortedWithDashLast( segment.fromSegments ):
               currentIntfLine = currentIntfLine + 1
               if firstLine:
                  addNewRow( segmentName, intfSet[ 0 ],
                             segment.ipv4PrefixListName,
                             segment.ipv6PrefixListName,
                             fromSegment,
                             segment.fromSegments[ fromSegment ] )
                  firstLine = False
               else:
                  intf = ''
                  if currentIntfLine < noOfIntfLine:
                     intf = intfSet[ currentIntfLine ]
                     addNewRow( segmentName, intf, '', '', fromSegment,
                                segment.fromSegments[ fromSegment ] )
                  else:
                     addNewRow( segmentName, '', '', '', fromSegment,
                                segment.fromSegments[ fromSegment ] )
            currentIntfLine = currentIntfLine + 1
            while currentIntfLine < noOfIntfLine:
               # if all fromSegment -> policyMap is exausted but intfSet is
               # still not completely printed
               table.newRow( segmentName, intfSet[ currentIntfLine ], '', '', '' )
               currentIntfLine = currentIntfLine + 1
      print( table.output() )
      print()

   def renderNonVrf( self, headings ):
      table = createTable( headings )
      f1 = Format( justify='left' )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )
      table.formatColumns( f1, f1, f1, f1, f1 )

      segmentsList = sorted( self.segments )
      segmentsInfo = self.segments

      for segmentName in segmentsList:
         segment = segmentsInfo[ segmentName ]
         intfSet = self._createInterfacesString( segment.interfaces,
                                                 segment.interfaceVlans )
         noOfIntfLine = len( intfSet )
         if not segment.fromSegments:
            currentIntfLine = 0
            while currentIntfLine < noOfIntfLine:
               if currentIntfLine == 0:
                  # print first intf line
                  table.newRow( segmentName, intfSet[ currentIntfLine ], '', '' )
               else:
                  table.newRow( segmentName, intfSet[ currentIntfLine ], '', '', '' )
               currentIntfLine = currentIntfLine + 1
         else:
            currentIntfLine = -1
            firstLine = True
            for fromSegment in _sortedWithDashLast( segment.fromSegments ):
               currentIntfLine = currentIntfLine + 1
               if firstLine:
                  table.newRow( segmentName, intfSet[ 0 ],
                                fromSegment,
                                segment.fromSegments[ fromSegment ] )
                  firstLine = False
               else:
                  intf = ''
                  if currentIntfLine < noOfIntfLine:
                     intf = intfSet[ currentIntfLine ]
                     table.newRow( segmentName, intf, fromSegment,
                                   segment.fromSegments[ fromSegment ] )
                  else:
                     table.newRow( segmentName, '', fromSegment,
                                   segment.fromSegments[ fromSegment ] )
            currentIntfLine = currentIntfLine + 1
            while currentIntfLine < noOfIntfLine:
               # if all fromSegment -> policyMap is exausted but intfSet is
               # still not completely printed
               table.newRow( segmentName, intfSet[ currentIntfLine ], '', '' )
               currentIntfLine = currentIntfLine + 1
      print( table.output() )
      print()

   def render( self ):
      for vrfName in self.vrfs:
         vrfModel = self.vrfs[ vrfName ]
         if not vrfModel.enabled:
            print( "Segment Security is not enabled." )
            return

         if not vrfModel.vrfExist:
            print( "Segment Security not configured in VRF %s" % vrfName )
            return
         print( _dashMeaning )
         print()
         print( 'VRF : %s' % vrfName )
         if vrfModel.ipv6PrefixSupported:
            headings = ( 'Segment', 'Interfaces', 'Prefix IPv4', 'Prefix IPv6',
                          'From Segment', 'Policy' )
         else:
            headings = ( 'Segment', 'Interfaces', 'Prefix IPv4',
                         'From Segment', 'Policy' )
         self.renderVrf( headings, vrfName )
      if self.segments:
         print( _dashMeaning )
         print()
         headings = ( 'Segment', 'Interfaces', 'From Segment', 'Policy' )
         self.renderNonVrf( headings )

def _sortedWithDashLast( coll, isDict=False ):
   if isDict:
      return sorted( coll, key=lambda k: ( k[ 0 ] == "-", k[ 0 ] ) )
   return sorted( coll, key=lambda k: ( k == "-", k ) )

class FwPolicyToInfoSourceSegment( Model ):
   policy = Str( help="Policy associated with segment", optional=True )
   # Possible additional information

class FwPolicyToInfoSourceSegments( Model ):
   fromSegments = Dict( keyType=str, valueType=FwPolicyToInfoSourceSegment,
                     help="Mappings from source segment name to policy" )

class FwPolicyToInfo( Model ):

   vrfName = Str( help="Relevant VRF name", optional=True )
   segments = Dict( keyType=str, valueType=FwPolicyToInfoSourceSegments,
            help="Policy mappings from destination segment(s) to source segment(s)" )
   enabled = Bool( help="Segment-security is enabled" )
   vrfExist = Bool( help="VRF exists" )
   _showDashMeaning = Bool( help="Explain '-' in output" )
   _showNaMeaning = Bool( help="Explain 'n/a' in output" )

   def setShowDashMeaning( self, val ):
      self._showDashMeaning = val

   def setShowNaMeaning( self, val ):
      self._showNaMeaning = val

   def render( self ):
      if self.vrfName is None:
         return
      if not self.enabled:
         print( "Segment Security is not enabled.\n" )
         return
      if not self.vrfExist:
         self.vrfName = '' if self.vrfName is None else self.vrfName
         print( "Segment Security not configured in VRF '%s'\n" % self.vrfName )
         return
      if self._showDashMeaning:
         print( _dashMeaning )
      if self._showNaMeaning:
         print( _naMeaning )
      print()
      print( 'VRF: %s' % self.vrfName )
      if not self.segments.keys():
         print( "Destination IP address does not match a segment\n" )
         return
      headings = ( 'To Segment', 'From Segment', 'Policy' )
      table = createTable( headings )
      f1 = Format( justify='left' )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )
      table.formatColumns( f1, f1, f1 )
      for ( destination, info ) in _sortedWithDashLast(
            self.segments.items(), isDict=True ):
         for ( source, sourceInfo ) in _sortedWithDashLast(
               info.fromSegments.items(), isDict=True ):
            policy = sourceInfo.policy if sourceInfo.policy else 'n/a'
            table.newRow( destination, source, policy )
      print( table.output() )

class FwCounterInfo( Model ):
   __revision__ = 2

   vrf = Str( help="VRF in which segment security is defined", optional=True )
   vrfExist = Bool( help="Whether VRF is created" )
   flowsCreated = Int( "Number of flows created", optional=True )
   invalidPackets = Int( "Invalid packets dropped counter", optional=True )
   bypassPackets = Int( "Packets not destined to any segment", optional=True )

   class PolicyCounter( Model ):
      hits = Int( "Number of packets evaluated" )
      drops = Int( "Explicit drop action counter" )
      defaultDrops = Int( "No matching rule in policy" )

   policies = Dict( keyType=str, valueType=PolicyCounter,
                    help="A mapping of policy names to counters for that policy" )

   class DestSegment( Model ):
      class SourceSegment( Model ):
         policyName = Str( "Policy name", optional=True )
         hits = Int( "Number of packets evaluated" )
         drops = Int( "Explicit drop action counter", optional=True )
         defaultDrops = Int( "No matching rule in policy", optional=True )

      # NB: "-" represents packets not hitting a source segment
      srcSegments = Dict( keyType=str, valueType=SourceSegment,
                          help="A mapping of source segment to counters" )

   dstSegments = Dict( keyType=str, valueType=DestSegment,
                       help="A mapping of destination segment to source segment" )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # previously non-optional attributes
         if 'vrf' not in dictRepr:
            dictRepr[ 'vrf' ] = 'default'
         for k in [ 'flowsCreated', 'invalidPackets', 'bypassPackets' ]:
            if k not in dictRepr:
               dictRepr[ k ] = 0
         for dstSeg in dictRepr[ 'dstSegments' ].values():
            for srcSeg in dstSeg[ 'srcSegments' ].values():
               k = 'policyName'
               if k not in srcSeg:
                  srcSeg[ k ] = ''
      return dictRepr

   def render( self ):

      hasGlobalCounters = ( ( self.flowsCreated is not None ) or
                            ( self.invalidPackets is not None ) or
                            ( self.bypassPackets is not None ) )

      if hasGlobalCounters:
         print( "Bypass: Packets not matching any destination segment" )
      print( "Default Drop: Packets dropped matching no rule in the policy" )
      print( "Drop: Number of packets explicitly dropped" )
      print( "Hit: Number of packets evaluated" )
      print( _dashMeaning )
      print()

      if self.vrfExist:
         print( "VRF: %s" % self.vrf )
         print()

      if hasGlobalCounters:
         print( "Global Counters" )
         print( "Flows created: %s" % self.flowsCreated )
         print( "Invalid pkt drop: %s" % self.invalidPackets )
         print( "Bypass: %s" % self.bypassPackets )
         print()

      headings = ( 'Policy', 'Hit', 'Drop', 'Default Drop' )
      table = createTable( headings )
      nameFormat = Format( justify='left' )
      nameFormat.noPadLeftIs( True )
      nameFormat.padLimitIs( True )
      numberFormat = Format( justify='right' )
      numberFormat.noPadLeftIs( True )
      numberFormat.padLimitIs( True )
      table.formatColumns( nameFormat, numberFormat, numberFormat, numberFormat )

      for policyName in sorted( self.policies ):
         policy = self.policies[ policyName ]
         table.newRow( policyName, policy.hits, policy.drops,
                       policy.defaultDrops )

      if self.policies:
         print( table.output() )

      def _hasSegmentDrops():
         for dstSegment in self.dstSegments.values():
            for srcSegment in dstSegment.srcSegments.values():
               return srcSegment.drops is not None
         return False

      hasSegmentDrops = _hasSegmentDrops()

      headings = [ 'Dest Segment', 'Source Segment', 'Policy', 'Hit' ]
      formats = [ nameFormat, nameFormat, nameFormat, numberFormat ]
      if hasSegmentDrops:
         headings += [ 'Drop', 'Default Drop' ]
         formats += [ numberFormat, numberFormat ]
      table = createTable( headings )
      table.formatColumns( *formats )

      def addRow( dstSegName, srcSegName, srcSeg ):
         columns = [ dstSegName,
                     srcSegName,
                     srcSeg.policyName if srcSeg.policyName is not None else "n/a",
                     srcSeg.hits ]
         if hasSegmentDrops:
            columns += [ srcSeg.drops, srcSeg.defaultDrops ]
         table.newRow( *columns )

      for dstSegName in sorted( self.dstSegments ):
         dstSeg = self.dstSegments[ dstSegName ]
         # sort and put "-" last
         for srcSegName in _sortedWithDashLast( dstSeg.srcSegments ):
            addRow( dstSegName, srcSegName, dstSeg.srcSegments[ srcSegName ] )

      if self.dstSegments:
         print( table.output() )

class FwHwStatusInfo( Model ):
   class Vrf( Model ):
      class DestSegment( Model ):
         class SourceSegment( Model ):
            status = Enum( values=( 'waiting', 'success', 'failed' ),
                           help='Segment status' )
         ipv4PrefixListName = Str( help='Name of IPv4 prefix list' )
         ipv6PrefixListName = Str( help='Name of IPv6 prefix list' )
         # NB: "-" represents packets not hitting a source segment
         srcSegments = Dict( keyType=str, valueType=SourceSegment,
                             help='A mapping of source segment name to its status' )
      dstSegments = Dict( keyType=str, valueType=DestSegment,
                          help='Destination segments keyed by their name' )
   ipv6PrefixSupported = Bool(
         help='IPv6 prefix supoort is enabled' )
   vrfs = Dict( keyType=str, valueType=Vrf,
         help='A mapping of VRF name to its segments status' )

   def render( self ):
      if self.ipv6PrefixSupported:
         headings = ( 'Segment', 'Prefix IPv4', 'Prefix IPv6',
                      'From Segment', 'Status' )
      else:
         headings = ( 'Segment', 'Prefix IPv4', 'From Segment', 'Status' )
      f1 = Format( justify='left' )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      def addNewRow( table, dstSegName, v4List, v6List, srcSegName, status ):
         if self.ipv6PrefixSupported:
            table.newRow( dstSegName, v4List, v6List, srcSegName, status )
         else:
            table.newRow( dstSegName, v4List, srcSegName, status )

      if self.vrfs:
         print( _dashMeaning )
         print()

      for vrfName in sorted( self.vrfs ):
         print( 'VRF: %s' % vrfName )
         table = createTable( headings )
         table.formatColumns( *( [ f1 ] * len( headings ) ) )
         vrfModel = self.vrfs[ vrfName ]
         for dstSegName in sorted( vrfModel.dstSegments ):
            firstLine = True
            dstSegModel = vrfModel.dstSegments[ dstSegName ]
            for srcSegName in _sortedWithDashLast( dstSegModel.srcSegments ):
               if firstLine:
                  addNewRow( table, dstSegName, dstSegModel.ipv4PrefixListName,
                             dstSegModel.ipv6PrefixListName, srcSegName,
                             dstSegModel.srcSegments[ srcSegName ].status )
                  firstLine = False
               else:
                  addNewRow( table, '', '', '', srcSegName,
                             dstSegModel.srcSegments[ srcSegName ].status )
         print( table.output() )

class FwHwErrorsInfo( Model ):
   class Vrf( Model ):
      class DestSegment( Model ):
         class SourceSegment( Model ):
            error = Str( help="Error causing this segment to fail to install" )

         # NB: "-" represents packets not hitting a source segment
         srcSegments = Dict( keyType=str, valueType=SourceSegment,
                             help="A mapping of source segment name to its error" )
      dstSegments = Dict( keyType=str, valueType=DestSegment,
                          help='Destination segments keyed by their name',
                          optional=True )
      error = Str( help="Error causing policies for this VRF to fail to install",
                   optional=True )

   vrfs = Dict( keyType=str, valueType=Vrf, help="A mapping from VRF to its errors" )
   # TODO: L2 Policy when platforms with L2 support this command

   def render( self ):
      headings = ( 'Segment', 'From Segment', 'Error' )
      f1 = Format( justify='left' )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )
      formats = [ f1 ] * len( headings )

      usesDashSegment = False

      body = ''
      for vrfName in sorted( self.vrfs ):
         body += 'VRF: ' + vrfName + '\n'
         vrfModel = self.vrfs[ vrfName ]
         if vrfModel.error is not None:
            body += vrfModel.error + '\n'
         if vrfModel.dstSegments is not None:
            table = createTable( headings )
            table.formatColumns( *formats )
            for dstSegName in sorted( vrfModel.dstSegments ):
               dstSegModel = vrfModel.dstSegments[ dstSegName ]
               usesDashSegment = usesDashSegment or (
                     '-' in dstSegModel.srcSegments )
               for srcSegName in _sortedWithDashLast( dstSegModel.srcSegments ):
                  table.newRow( dstSegName,
                                srcSegName,
                                dstSegModel.srcSegments[ srcSegName ].error )
            body += table.output() + '\n'
         else:
            body += '\n'

      if usesDashSegment:
         print( _dashMeaning, end='\n\n' )
      print( body, end='' )
