# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliUtil
import ConfigMount
import Cell
import CliCommand
import CliGlobal
import CliMatcher
from CliMode.FlowWatcher import (
   MonitorSecurityAwakeModeBase,
   NdrModeBase,
   NucleusModeBase,
)
from CliPlugin.Ssl import (
   profileMatcher,
   profileNameMatcher,
   sslMatcher
)
from CliPlugin import IntfCli
from CliPlugin.FlowWatcherCliLib import (
   MirroringConstants,
   SysUtil,
   nucleusNameMatcher,
)
import ExtensionMgrLib
from FlowWatcherCliUtil import (
   FlowTableSize,
   IpfixCollectorPort,
   MpId,
   NucleusPort,
)
from HostnameCli import IpAddrOrHostnameMatcher
import LazyMount
from MirroringCliLib import isMaxSessionsReached
import Tac
import Tracing
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'FlowWatcherCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

constants = Tac.newInstance( "FlowWatcher::Constants" )
L4Port = TacLazyType( "Ipfix::L4Port" )
ThreadConfig = TacLazyType( "FlowWatcher::ThreadConfig" )

# Global variable holder.
gv = CliGlobal.CliGlobal(
   dict(
      capabilities=None,
      extensionStatus=None,
      fwConfig=None,
      fwConfigReq=None,
      fwStatus=None,
      mirroringConfig=None,
      mirroringHwCapability=None,
   )
)

class MonitorSecurityAwakeMode( MonitorSecurityAwakeModeBase,
                                BasicCli.ConfigModeBase ):
   name = 'Monitor security awake configuration'

   def __init__( self, parent, session, context ):
      self.context_ = context
      self.session_ = session
      MonitorSecurityAwakeModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def context( self ):
      return self.context_

class NucleusMode( NucleusModeBase, BasicCli.ConfigModeBase ):
   name = 'Nucleus configuration'

   def __init__( self, parent, session, context, nucleus ):
      self.session_ = session
      self.context_ = context
      NucleusModeBase.__init__( self, nucleus )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def context( self ):
      return self.context_

class NucleusContext:
   def __init__( self, fwCtx, nucleus, config=None ):
      self.config_ = config

   def nucleusConfig( self ):
      return self.config_

class NdrMode( NdrModeBase, BasicCli.ConfigModeBase ):
   name = 'NDR configuration'

   def __init__( self, parent, session, context ):
      self.session_ = session
      self.context_ = context
      NdrModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def context( self ):
      return self.context_

class NdrContext:
   def __init__( self, fwCtx, config=None ):
      self.config_ = config

   def ndrConfig( self ):
      return self.config_

class FlowWatcherContext:
   def __init__( self, config, status ):
      self.config_ = config
      self.status_ = status
      self.nucleusCtx = {}
      self.ndrCtx = None
      for nucleus, nucleusConfig in config.nucleus.items():
         self.nucleusCtx[ nucleus ] = NucleusContext(
               self, nucleus, nucleusConfig )

   def nucleusContext( self, nucleus ):
      # Config could be changed after entering monitor security mode.
      # Create context with existing Sysdb config.
      nucleusConfig = self.config_.nucleus.get( nucleus )
      if nucleusConfig is None:
         nucleusConfig = self.config_.nucleus.newMember( nucleus )
      self.nucleusCtx[ nucleus ] = NucleusContext(
                                          self, nucleus, nucleusConfig )
      return self.nucleusCtx[ nucleus ]

   def ndrContext( self ):
      # Config could be changed after entering monitor security mode.
      # Create context with existing Sysdb config.
      if self.config_.ndrConfig is None:
         self.config_.ndrConfig = ()
      self.ndrCtx = NdrContext( self, self.config_.ndrConfig )
      return self.ndrCtx

   def flowWatcherConfig( self ):
      return self.config_

   def flowWatcherStatus( self ):
      return self.status_

# --------------------------------------------------------------------------
# "[no|default] monitor security awake"  in config mode
# --------------------------------------------------------------------------

def monitorSecurityAwakeConfigCmdHandler( mode, args ):
   t1( 'goto MonitorSecurityAwakeMode' )
   context = FlowWatcherContext( gv.fwConfig, gv.fwStatus )
   childMode = mode.childMode( MonitorSecurityAwakeMode, context=context )
   mode.session_.gotoChildMode( childMode )

def monitorSecurityAwakeConfigCmdNoHandler( mode, args ):
   t1( 'no MonitorSecurityAwakeMode' )
   lastMode = mode.session_.modeOfLastPrompt()
   if ( isinstance( lastMode, MonitorSecurityAwakeMode ) and
         lastMode.context() ):
      # If 'no monitor security awake' is issued in
      # monitor-security-awake mode then delete context
      lastMode.context_ = None
   config = gv.fwConfig
   config.enabled = False
   config.topic = gv.fwConfig.topicDefault
   config.flowTableSize = FlowTableSize.sizeDefault
   config.monitorPointId = MpId.mpIdDefault
   config.ipfixCollectorPort = L4Port.ipfixPortDefault
   config.nucleus.clear()
   del gv.mirroringConfig.session[ MirroringConstants.msaMirrorSession ]
   if config.ndrConfig:
      config.ndrConfig.unknownUdp = False

# --------------------------------------------------------------------------
# "[no|default] disabled" command in "config-monitor-security-awake" mode
# --------------------------------------------------------------------------

class DisabledCmd( CliCommand.CliCommandClass ):
   syntax = '''disabled'''
   noOrDefaultSyntax = syntax

   data = {
      'disabled': 'Enable or disable the monitor security awake feature'
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().flowWatcherConfig().enabled = False
      del gv.mirroringConfig.session[ MirroringConstants.msaMirrorSession ]

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      msaMirrorSession = MirroringConstants.msaMirrorSession
      sessionCfg = gv.mirroringConfig.session.get( msaMirrorSession )
      if not sessionCfg:
         # display error if creating more than max sessions
         if isMaxSessionsReached( mode, msaMirrorSession, gv.mirroringHwCapability,
                                  gv.mirroringConfig ):
            errMsg = ( "Could not create session '" + msaMirrorSession +
                       "'. Maximum mirroring sessions limit reached" )
            mode.addError( errMsg )
            return
         sessionCfg = gv.mirroringConfig.session.newMember( msaMirrorSession )
         sessionCfg.isHiddenSession = True
         sessionCfg.owner = "MonitorSecurityAwake"
         sessionCfg.targetIntf.newMember( "Cpu" )
      mode.context().flowWatcherConfig().enabled = True

# ----------------------------------------------------------------------------
# "[no|default] topic <name>" command in "config-monitor-security-awake" mode
# ----------------------------------------------------------------------------

class TopicCmd( CliCommand.CliCommandClass ):
   syntax = '''topic TOPIC_NAME'''
   noOrDefaultSyntax = '''topic ...'''
   data = {
      'topic': 'Configure topic name',
      'TOPIC_NAME': CliMatcher.PatternMatcher( pattern=r'[a-zA-Z0-9\._-]+',
                                                helpname='TOPIC',
                                                helpdesc='Topic name' ),
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.context().flowWatcherConfig()
      topic = args.get( 'TOPIC_NAME', config.topicDefault )
      nameMaxLen = constants.confNameMaxLen
      if len( topic ) > nameMaxLen:
         mode.addError( 'Topic name is too long (maximum %d)' % nameMaxLen )
         return
      config.topic = topic

   noOrDefaultHandler = handler

# --------------------------------------------------------------------------
# "[no|default] monitor-point identifier <id>" command
# in "config-monitor-security-awake" mode
# --------------------------------------------------------------------------

class MonitorPointIdCmd( CliCommand.CliCommandClass ):
   syntax = '''monitor-point identifier ID'''
   noOrDefaultSyntax = '''monitor-point identifier ...'''

   data = {
      'monitor-point': 'Configure monitor point',
      'identifier': 'Configure monitor point identifier',
      'ID': CliMatcher.IntegerMatcher( MpId.minMpId, MpId.maxMpId,
                           helpdesc='Monitor point ID' ),
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.context().flowWatcherConfig()
      monitorPointId = args.get( 'ID', MpId.mpIdDefault )
      config.monitorPointId = monitorPointId

   noOrDefaultHandler = handler

# -----------------------------------------------------------------------------------
# [no|default] flow table size <size> entries in "config-monitor-security-awake" mode
# -----------------------------------------------------------------------------------

class FlowTableSizeCmd( CliCommand.CliCommandClass ):
   syntax = '''flow table size SIZE entries'''
   noOrDefaultSyntax = '''flow table size ...'''

   data = {
      'flow': 'Configure flow parameters',
      'table': 'Configure flow table parameters',
      'size': 'Configure maximum flow table size',
      'SIZE': CliMatcher.IntegerMatcher(
                  FlowTableSize.minSize,
                  FlowTableSize.maxSize,
                  helpdesc='Maximum number of entries in flow table' ),
      'entries': 'Flow table entries'
   }
   @staticmethod
   def handler( mode, args ):
      config = mode.context().flowWatcherConfig()
      status = mode.context().flowWatcherStatus()
      size = args.get( 'SIZE', FlowTableSize.sizeDefault )

      if config.flowTableSize == size:
         return

      restartNeeded = True
      lowMemoryMode = ( gv.fwStatus.lowMemoryMode if gv.fwStatus.enabled
                        else SysUtil.useLowMemoryMode() )
      if lowMemoryMode and size > FlowTableSize.sizeDefault:
         warningPrompt = ( 'The flow table will be restricted to %d entries on'
                           ' this system (low memory mode)'
                           % FlowTableSize.sizeDefault )
         mode.addWarning( warningPrompt )
         if status.flowTableSize == FlowTableSize.sizeDefault:
            restartNeeded = False

      # check whether FlowWatcher agent is running using the same logic as
      # launcher plugin
      flowWatcherEnabled = ( config.enabled and gv.capabilities.awakeSupported and
                             constants.swixName in
                             gv.extensionStatus.installedPrimaryPkg )
      if ( flowWatcherEnabled or status.enabled ) and restartNeeded:
         warningPrompt = 'The flow table size configuration change will cause ' \
         'the FlowWatcher agent restart and all active flows to be lost.'
         promptText = 'Do you wish to proceed with this command? [y/N]'
         mode.addWarning( warningPrompt )
         if not BasicCliUtil.confirm( mode, promptText, answerForReturn=False ):
            mode.addError( 'Command aborted by user' )
            return
      config.flowTableSize = size

   noOrDefaultHandler = handler

# --------------------------------------------------------------------------
# The "[ no | default ] ipfix listener port UDP_PORT" command,
# in "config-monitor-security-awake" mode
# --------------------------------------------------------------------------
class IpfixCollectorPortCmd( CliCommand.CliCommandClass ):
   syntax = 'ipfix listener port UDP_PORT'
   noOrDefaultSyntax = 'ipfix listener ...'
   data = {
      'ipfix': 'IPFIX collector information',
      'listener': 'IPFIX collector listener',
      'port': 'IPFIX collector listener UDP port',
      'UDP_PORT': CliMatcher.IntegerMatcher( IpfixCollectorPort.minPort,
                                             IpfixCollectorPort.maxPort,
                                             helpdesc="UDP port number" ),
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.context().flowWatcherConfig()
      port = args.get( 'UDP_PORT', IpfixCollectorPort.ipfixPortDefault )
      config.ipfixCollectorPort = port

   noOrDefaultHandler = handler

# -------------------------------------------------------------------------------
# "[no|default] ndr application" command, in "config-monitor-security-awake" mode
# -------------------------------------------------------------------------------

class NdrCommand( CliCommand.CliCommandClass ):
   syntax = '''ndr application'''
   noOrDefaultSyntax = syntax

   data = {
      'ndr': 'Configure network detection and response',
      'application' : 'Configure application to monitor',
   }

   @staticmethod
   def handler( mode, args ):
      context = mode.context()
      t1( 'goto NdrMode' )

      ndrCtx = context.ndrContext()

      childMode = mode.childMode( NdrMode, context=ndrCtx )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      context = mode.context()
      config = context.flowWatcherConfig()
      t1( 'noNdrMode' )
      config.ndrConfig = None

# --------------------------------------------------------------------------
# "[no|default] unknown-udp" command
# in "config-monitor-security-awake-ndr" mode
# --------------------------------------------------------------------------

class NdrUnknownUdpCmd( CliCommand.CliCommandClass ):
   syntax = '''unknown-udp'''
   noOrDefaultSyntax = syntax

   data = {
      'unknown-udp': 'Monitor unknown UDP flows',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().ndrConfig().unknownUdp = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().ndrConfig().unknownUdp = False

# -------------------------------------------------------------------------------
# "[no|default] nucleus <name>" command, in "config-monitor-security-awake" mode
# -------------------------------------------------------------------------------

def getNucleusName( mode ):
   return list( mode.context().nucleusCtx )

class NucleusCommand( CliCommand.CliCommandClass ):
   syntax = '''nucleus NUCLEUS_NAME'''
   noOrDefaultSyntax = syntax

   data = {
      'nucleus': 'Configure nucleus',
      'NUCLEUS_NAME': nucleusNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      nucleus = args[ 'NUCLEUS_NAME' ]
      context = mode.context()
      config = context.flowWatcherConfig()
      t1( 'goto NucleusMode of', nucleus )

      nameMaxLen = constants.confNameMaxLen
      if len( nucleus ) > nameMaxLen:
         mode.addError( 'Nucleus name is too long (maximum %d)'
                        % nameMaxLen )
         return

      maxNucleus = constants.maxNucleus
      if nucleus not in config.nucleus and \
            len( config.nucleus ) >= maxNucleus:
         mode.addError( 'Maximum nucleus configuration limit: %d reached'
                        % maxNucleus )
         return

      nucleusCtx = context.nucleusContext( nucleus )

      childMode = mode.childMode( NucleusMode, context=nucleusCtx,
                                  nucleus=nucleus )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nucleus = args[ 'NUCLEUS_NAME' ]
      context = mode.context()
      config = context.flowWatcherConfig()
      t1( 'noNucleusMode of', nucleus )
      if nucleus in config.nucleus:
         del config.nucleus[ nucleus ]
      if nucleus in context.nucleusCtx:
         del context.nucleusCtx[ nucleus ]

# --------------------------------------------------------------------------
# "[no|default] local interface <intf>" command
# in "config-monitor-security-awake-nucleus" mode
# --------------------------------------------------------------------------

class NucleusLocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = '''local interface INTERFACE'''
   noOrDefaultSyntax = '''local interface ...'''

   data = {
      'local': 'Configure the local interface',
      'interface': 'Configure the local interface',
      'INTERFACE': IntfCli.Intf.matcherWithIpSupport,
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().nucleusConfig().localIntf = args[ 'INTERFACE' ].name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.context().nucleusConfig()
      config.localIntf = config.localIntfDefault

# --------------------------------------------------------------------------
# "[no|default] destination <ipv4/6-addr> [ port <port>]" command
# in "config-monitor-security-awake-nucleus" mode
# --------------------------------------------------------------------------

class NucleusDestinationCmd( CliCommand.CliCommandClass ):
   syntax = '''destination NUCLEUS [ port PORT ]'''
   noOrDefaultSyntax = '''destination ...'''

   data = {
      'destination': 'Configure nucleus destination',
      'NUCLEUS': IpAddrOrHostnameMatcher( ipv6=True,
         helpdesc='IPv4 address or IPv6 address or fully qualified domain name' ),
      'port': 'Nucleus port',
      'PORT': CliMatcher.IntegerMatcher( NucleusPort.minPort,
                                         NucleusPort.maxPort,
                                         helpdesc='Nucleus port' ),
   }

   @staticmethod
   def handler( mode, args ):
      hostname = args[ 'NUCLEUS' ]
      port = args.get( 'PORT', NucleusPort.nucleusPortDefault )
      hostnameStr = constants.ipAddrDefault if not hostname else str( hostname )
      config = mode.context().nucleusConfig()
      t1( 'Nucleus', config.name, 'set destination: ', hostname, port, hostnameStr )
      config.nucleusHostAndPort = Tac.Value( 'Arnet::HostAndPort',
                                             hostname=hostnameStr, port=port )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.context().nucleusConfig()
      t1( 'Nucleus', config.name, 'delete destination' )
      config.nucleusHostAndPort = Tac.Value( 'Arnet::HostAndPort' )

# --------------------------------------------------------------------------
# The "[ no | default ] ssl profile PROFILE_NAME command,
# in "config-monitor-security-awake-nucleus" mode
# --------------------------------------------------------------------------
class NucleusSslProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILE_NAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': sslMatcher,
      'profile': profileMatcher,
      'PROFILE_NAME': profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.context().nucleusConfig()
      sslProfileName = args.get( 'PROFILE_NAME', config.sslProfileDefault )
      config.sslProfile = sslProfileName

   noOrDefaultHandler = handler

# -------------------------------------------------------------------------------
# "clear monitor security awake counters | ( memory statistics )"
# -------------------------------------------------------------------------------

def msaClearCmdHandler( mode, args ):
   if "counters" in args:
      gv.fwConfigReq.clearCountersReqTime = Tac.now()
   else:
      gv.fwConfigReq.clearMemoryStatsReqTime = Tac.now()

# -------------------------------------------------------------------------------
# "[no|default] agent flowwatcher threads ( single | ( dpi tx ) )"
# -------------------------------------------------------------------------------

def threadConfigCmdHandler( mode, args, no=False ):
   threadConfig = None if no else ThreadConfig( 'single' in args )
   if threadConfig != gv.fwConfig.threadConfig:
      mode.addWarning( 'Change will take effect only after agent restart' )
   gv.fwConfig.threadConfig = threadConfig

def noOrDefaultThreadConfigCmdHandler( mode, args ):
   threadConfigCmdHandler( mode, args, no=True )

MonitorSecurityAwakeMode.addCommandClass( DisabledCmd )
MonitorSecurityAwakeMode.addCommandClass( TopicCmd )
MonitorSecurityAwakeMode.addCommandClass( MonitorPointIdCmd )
MonitorSecurityAwakeMode.addCommandClass( FlowTableSizeCmd )
MonitorSecurityAwakeMode.addCommandClass( IpfixCollectorPortCmd )
MonitorSecurityAwakeMode.addCommandClass( NucleusCommand )
NucleusMode.addCommandClass( NucleusLocalInterfaceCmd )
NucleusMode.addCommandClass( NucleusSslProfileCmd )
NucleusMode.addCommandClass( NucleusDestinationCmd )
MonitorSecurityAwakeMode.addCommandClass( NdrCommand )
NdrMode.addCommandClass( NdrUnknownUdpCmd )

# --------------------------
def Plugin( em ):
   gv.capabilities = LazyMount.mount( em, 'hardware/flowwatcher/capabilities',
                                      'HwFlowWatcher::Capabilities', 'r' )

   gv.extensionStatus = LazyMount.mount( em, ExtensionMgrLib.statusPath(),
                                         'Extension::Status', 'r' )

   gv.fwConfig = ConfigMount.mount( em, 'flowwatcher/config',
                                    'FlowWatcher::Config', 'w' )

   gv.fwConfigReq = LazyMount.mount( em, 'flowwatcher/configReq',
                                     'FlowWatcher::ConfigReq', 'w' )

   gv.fwStatus = LazyMount.mount( em, Cell.path( 'flowwatcher/status' ),
                                  'FlowWatcher::Status', 'r' )

   gv.mirroringConfig = ConfigMount.mount( em, "mirroring/config",
                                           "Mirroring::Config", "w" )

   gv.mirroringHwCapability = LazyMount.mount( em, "mirroring/hwcapability",
                                               "Mirroring::HwCapability", "r" )
