# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import sortIntf
from ArnetModel import (
   IpGenericAddrAndPort,
   IpGenericAddress,
)
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from CliPlugin.FlowWatcherCliLib import (
   FlowWatcherConstants,
   FlowWatcherInactiveReasonEnum,
   IpProtoType,
   NucleusInactiveReasonEnum,
   utcTimeRelativeToNowStr,
)
from collections import OrderedDict
from FlowTrackerCliUtil import (
   addressStr,
   protocolStr,
   renderCounter,
   timeStr,
)
from Intf.IntfRange import intfListToCanonical
from IntfModels import Interface
from natsort import natsorted, natsort_key
import TableOutput
import TacSigint

flowWatcherInactiveReasonStr = {
   "eosExtensionNotInstalled": "EOS extension not installed",
   "monitorSecurityAwakeDisabled": "Monitor security awake disabled",
   "monitorSecurityAwakeInitializing": "Monitor security awake initializing",
   "hwFlowTrackingNotRunning": "Hardware flow tracking not running",
   "hwExporterNotActive": "Hardware flow tracking exporter not active",
   "monitorSessionNotActive": "Monitor session not active",
   "nucleusNotConnected": "Nucleus not connected",
}

nucleusInactiveReasonStr = {
   "hostNotConfigured": "Host not configured",
   "portNotConfigured": "Port not configured",
   "localAddrNotConfigured": "Local address not configured",
   "vrfNotConfigured": "VRF not configured",
   "sslProfileNameNotConfigured": "SSL profile name not configured",
   "sslProfileNotConfigured": "SSL profile does not exist",
   "sslProfileNotValid": "SSL profile not valid",
   "topicNotConfigured": "Topic name not configured",
   "certNotConfigured": "SSL certificate not configured",
   "trustedCertsNotConfigured": "SSL trusted certificates not configured",
   "tlsVersionsNotConfigured": "SSL versions not configured",
   "cipherSuiteNotConfigured": "SSL cipher suite not configured",
   "cipherSuiteV1_3NotConfigured": "TLSv1.3 cipher suite not configured",
   "dnsResolutionInProgress": "DNS resolution in progress",
   "connectionInProgress": "Connection in progress",
   "connectionLost": "Connection lost",
   "connectionFailed": "Connection failed",
   "duplicateDestination": "Duplicate nucleus destination",
}

interval5min = FlowWatcherConstants.interval5min
interval1hour = FlowWatcherConstants.interval1hour
interval24hour = FlowWatcherConstants.interval24hour

MB = 1024.0 * 1024.0

formatLeft = TableOutput.Format( justify="left" )
formatLeft.noPadLeftIs( True )
formatLeft.padLimitIs( True )
formatRight = TableOutput.Format( justify="right" )
formatRight.noPadLeftIs( True )
formatRight.padLimitIs( True )

def renderRate( rate ):
   if rate >= 1000:
      return renderCounter( int( rate ), humanReadableOnly=True )
   elif float( rate ).is_integer() or rate == 0.0:
      return str( int( rate ) )
   # round rate to 0.1 because we only show one decimal digit
   rate = max( rate, 0.1 )
   return f"{rate:.1f}"

class MsaNucleusInactiveReason( Model ):
   inactiveReason = Enum( values=NucleusInactiveReasonEnum.attributes,
                          help="Nucleus inactive reason" )
   errorReason = Str( optional=True, help="TCP or SSL error reason" )

class MsaNucleus( Model ):
   status = Enum( help="Status of the nucleus connection",
                  values=[ "connected", "inactive" ] )
   inactiveReasons = List( help="Reasons why the nucleus is not active",
                           valueType=MsaNucleusInactiveReason )
   vrf = Str( help="VRF name" )
   localIntf = Interface( help="Local interface from which to connect" )
   localAddr = IpGenericAddress( help="Local IP address from which to connect" )
   destinationAddr = IpGenericAddress( help="IP address of the nucleus" )
   destinationPort = Int( help="Port number of the nucleus" )
   sslProfile = Str( help="SSL profile name" )
   lastEstablished = Float( help="Last time that connection was established "
                                 "successfully" )

   def render( self, nucleusName ): # pylint: disable=arguments-differ
      print( "Nucleus:", nucleusName )
      print( "Status:", self.status )
      if self.inactiveReasons:
         inactiveReasons = []
         for reason in self.inactiveReasons:
            inactiveReasons.append(
               nucleusInactiveReasonStr[ reason.inactiveReason ] )
            if reason.errorReason:
               inactiveReasons.append( reason.errorReason )
         print( "Inactive reason(s):", ", ".join( inactiveReasons ) )
      print( "VRF:", self.vrf )
      if self.localIntf:
         print( f"Local interface: {self.localIntf.stringValue} ({self.localAddr})" )
      else:
         print( "Local interface:" )
      print( "Destination:", self.destinationAddr, "port", self.destinationPort )
      print( "SSL profile:", self.sslProfile )
      print( "Last established:", utcTimeRelativeToNowStr( self.lastEstablished ) )

class MsaInactiveReason( Model ):
   inactiveReason = Enum( values=FlowWatcherInactiveReasonEnum.attributes,
                          help="Nucleus inactive reason" )

class MsaStatus( Model ):
   active = Bool( help="Indicates whether the monitor is active or not" )
   lowMemoryMode = Bool( help="Indicates whether the monitor is running in "
                         "low memory mode" )
   inactiveReasons = List( help="Reasons why monitor is not active",
                           valueType=MsaInactiveReason )
   topicName = Str( help="Configured name of topic" )
   monitorPointId = Int( help="Configured monitor point identifier" )
   flowTableSize = Int( help="Maximum flow table size" )
   flowTableInactiveTimeout = Float( help="Timeout of flows in seconds" )
   activeIntfs = List( help="Interfaces on which monitor is active",
                       valueType=Interface )

   def render( self ):
      if self.active:
         statusStr = "active (low memory mode)" if self.lowMemoryMode else "active"
      else:
         statusStr = "inactive"
      print( "Monitor security awake status:", statusStr )
      if self.inactiveReasons:
         print( "Inactive reason(s):",
                ", ".join( flowWatcherInactiveReasonStr[ reason.inactiveReason ]
                           for reason in self.inactiveReasons ) )
      print( "Topic identifier:", self.topicName )
      print( "Monitor point identifier:", self.monitorPointId )
      print( "Flow table size:", self.flowTableSize, "entries" )
      print( "Flow table inactive timeout:", self.flowTableInactiveTimeout,
             "seconds" )
      activeIntfs = sortIntf( intf.shortName for intf in self.activeIntfs )
      print( "Active interfaces:", ", ".join( intfListToCanonical( activeIntfs ) ) )

class Msa( Model ):
   status = Submodel( help="Monitor security awake status", valueType=MsaStatus,
                      optional=True )
   nucleuses = Dict( help="A mapping of nucleus name to its status",
                     keyType=str, valueType=MsaNucleus )

   def render( self ):
      if self.status:
         self.status.render()
      for i, ( nucleusName, nucleus ) in (
            enumerate( natsorted( self.nucleuses.items() ) ) ):
         if self.status or i:
            print( "" )
         nucleus.render( nucleusName )

class AppCounters( Model ):
   flows = Int( help="Cumulative created flow count", default=0 )
   activeFlows = Int( help="Number of active flows", default=0 )
   expiredFlows = Int( help="Cumulative expired flow count", default=0 )

class FlowCounters( Model ):
   flows = Int( help="Cumulative created flow count", default=0 )
   activeFlows = Int( help="Number of active flows", default=0 )
   expiredFlows = Int( help="Cumulative expired flow count", default=0 )
   packets = Int( help="Cumulative count of packets received", default=0 )
   flowsRates = Dict(
      help="A mapping of time interval to flows created per second",
      keyType=int, valueType=float )
   packetsRates = Dict(
      help="A mapping of time interval to packets received per second",
      keyType=int, valueType=float )
   ipv4AppCounters = Dict(
      help="A mapping of IPv4 application name to flow counters",
      keyType=str, valueType=AppCounters )
   ipv6AppCounters = Dict(
      help="A mapping of IPv6 application name to flow counters",
      keyType=str, valueType=AppCounters )
   lastCountersClearedTime = Float( help="Last time that the flows counters were"
                                         "cleared",
                                    default=0.0 )

class IpfixCounters( Model ):
   exporter = Submodel( help="Exporter IP address and port number",
                        valueType=IpGenericAddrAndPort )
   observationDomainId = Int( help="Observation domain ID" )
   rxMessages = Int( help="Messages received" )
   rxTemplateRecords = Int( help="Template records received" )
   rxOptionsTemplateRecords = Int( help="Options template records received" )
   rxDataRecords = Int( help="Data records received" )
   rxOptionsDataRecords = Int( help="Options data records received" )
   templateIdErrors = Int( help="Unknown template ID errors" )
   invalidIpfixMsgs = Int( help="Invalid IPFIX messages received" )
   flowRecordQueueFull = Int( help="Flow record queue full" )
   messagesRates = Dict(
      help="A mapping of time interval to messages received per second",
      keyType=int, valueType=float )
   dataRecordsRates = Dict(
      help="A mapping of time interval to data records received per second",
      keyType=int, valueType=float )
   lastCountersClearedTime = Float( help="Last time that the IPFIX counters "
                                         "were cleared" )

class NucleusCounters( Model ):
   numActivityRecordsTransmitted = Int( help="Number of activity records sent" )
   lastActivityTransmissionTimestamp = Float(
      help="Time when the last activity record was sent" )
   numProgressRecordsTransmitted = Int( help="Number of progress records sent" )
   lastProgressTransmissionTimestamp = Float(
      help="Time when the last progress record was sent" )
   lastConnectionSuccessTimestamp = Float( help="Time of the last successful "
                                                "connection" )
   connectionSuccessCount = Int( help="Number of successful connections" )
   lastConnectionErrorTimestamp = Float( help="Time of the last failed connection" )
   connectionErrorCount = Int( help="Number of failed connections" )
   numActivityRecordsInQueue = Int(
      help="Number of activity records currently in the queue" )
   numProgressRecordsInQueue = Int(
      help="Number of progress records currently in the queue" )
   activityRecordsRates = Dict(
      help="A mapping of time interval to activity records transmitted per second",
      keyType=int, valueType=float )
   progressRecordsRates = Dict(
      help="A mapping of time interval to progress records transmitted per second",
      keyType=int, valueType=float )
   throughputs = Dict(
      help="A mapping of time interval to outgoing activity and progress records"
      " throughput in Mbps",
      keyType=int, valueType=float )
   lastCountersClearedTime = Float( help="Last time that the nucleus counters "
                                         "were cleared" )

class MsaCounters( Model ):
   flowCounters = Submodel( help="Cumulative flow counters", valueType=FlowCounters,
                            optional=True )
   ipfixCounters = List( help="A list of per-exporter IPFIX counters",
                         valueType=IpfixCounters )
   nucleusCounters = Dict( help="A list of per-nucleus counters",
                           keyType=str, valueType=NucleusCounters )
   activityRecordQueueFull = Int( help="Activity record queue full", optional=True )
   packetBundleQueueFull = Int( help="Packet bundle queue full", optional=True )
   flowTableFull = Int( help="Flow table full", optional=True )

   def printAppCounters( self, appCounters, ipVersion ):
      headings = ( "Application", "Flows Active", "Flows Created", "Flows Expired" )

      table = TableOutput.createTable( headings, tableWidth=100 )
      table.formatColumns(
         formatLeft, # Application
         formatRight, # Flows Active
         formatRight, # Flows Created
         formatRight, # Flows Expired
      )

      def otherLast( keyValuePair ):
         name = keyValuePair[ 0 ]
         return ( name == "Other", natsort_key( name ) )

      # Always print other application type at the end of the table
      for app, counter in sorted( appCounters.items(), key=otherLast ):
         table.newRow( app,
                       renderCounter( counter.activeFlows ),
                       renderCounter( counter.flows ),
                       renderCounter( counter.expiredFlows ) )
         TacSigint.check()

      print( f"{ipVersion} flows:" )
      print( table.output() )

   def printRates( self, heading, rows ):
      headings = ( heading, "Last 5 mins", "Last 1 hr", "Last 24 hrs" )

      table = TableOutput.createTable( headings, tableWidth=100 )
      table.formatColumns( formatLeft, * 3 * ( formatRight, ) )
      for row, rates in rows.items():
         table.newRow( row,
                       renderRate( rates[ interval5min ] ),
                       renderRate( rates[ interval1hour ] ),
                       renderRate( rates[ interval24hour ] )
                      )
      print( table.output() )

   def renderFlowCounters( self ):
      print( "Active flows: {}, RX packets: {}".format(
         renderCounter( self.flowCounters.activeFlows ),
         renderCounter( self.flowCounters.packets ) ) )
      print( "Flows created: {}, expired: {}".format(
         renderCounter( self.flowCounters.flows ),
         renderCounter( self.flowCounters.expiredFlows ) ) )
      if self.flowCounters.lastCountersClearedTime:
         print( "Last clearing of flows counters:", utcTimeRelativeToNowStr(
            self.flowCounters.lastCountersClearedTime ) )
      rates = OrderedDict()
      rates[ "Flows created (per second)" ] = self.flowCounters.flowsRates
      rates[ "Packets received (per second)" ] = self.flowCounters.packetsRates
      self.printRates( "Processed", rates )
      self.printAppCounters( self.flowCounters.ipv4AppCounters, "IPv4" )
      self.printAppCounters( self.flowCounters.ipv6AppCounters, "IPv6" )

   def renderIpfixCounters( self ):
      print( "IPFIX counters:" )
      for counter in natsorted( self.ipfixCounters ):
         print( "Exporter: %s Source port: %d Observation domain ID: %d" %
                ( counter.exporter.ip.stringValue, counter.exporter.port,
                  counter.observationDomainId ) )
         print( "Messages received: {}".format(
            renderCounter( counter.rxMessages ) ) )
         print( "Template records received: {}".format(
            renderCounter( counter.rxTemplateRecords ) ) )
         print( "Options template records received: {}".format(
                renderCounter( counter.rxOptionsTemplateRecords ) ) )
         print( "Data records received: {}".format(
            renderCounter( counter.rxDataRecords ) ) )
         print( "Options data records received: {}".format(
            renderCounter( counter.rxOptionsDataRecords ) ) )
         print( "Unknown template ID errors: {}".format(
            renderCounter( counter.templateIdErrors ) ) )
         print( "Invalid IPFIX messages received: {}".format(
            renderCounter( counter.invalidIpfixMsgs ) ) )
         print( "Flow record queue full: {}".format(
            renderCounter( counter.flowRecordQueueFull ) ) )
         if counter.lastCountersClearedTime:
            print( "Last clearing of IPFIX counters:",
                   utcTimeRelativeToNowStr( counter.lastCountersClearedTime ) )
         rates = OrderedDict()
         rates[ "Messages (per second)" ] = counter.messagesRates
         rates[ "Data records (per second)" ] = counter.dataRecordsRates
         self.printRates( "Received", rates )

   def renderNucleusCounters( self ):
      for nucleusName, counter in natsorted( self.nucleusCounters.items() ):
         print( "Nucleus:", nucleusName )
         print( "Activity records sent: {},".format(
            renderCounter( counter.numActivityRecordsTransmitted ) ),
                "last sent",
                utcTimeRelativeToNowStr(
                   counter.lastActivityTransmissionTimestamp ) )
         print( "Progress records sent: {},".format(
            renderCounter( counter.numProgressRecordsTransmitted ) ),
                "last sent",
                utcTimeRelativeToNowStr(
                   counter.lastProgressTransmissionTimestamp ) )
         print( "Last successful connection:",
                utcTimeRelativeToNowStr( counter.lastConnectionSuccessTimestamp ) )
         print( "Successful connections: {}".format(
            renderCounter( counter.connectionSuccessCount ) ) )
         print( "Last connection failure:",
                utcTimeRelativeToNowStr( counter.lastConnectionErrorTimestamp ) )
         print( "Connection failures: {}".format(
            renderCounter( counter.connectionErrorCount ) ) )
         print( "Activity records in queue: {}".format(
            renderCounter( counter.numActivityRecordsInQueue ) ) )
         print( "Progress records in queue: {}".format(
            renderCounter( counter.numProgressRecordsInQueue ) ) )
         if counter.lastCountersClearedTime:
            print( "Last clearing of nucleus counters:",
                   utcTimeRelativeToNowStr( counter.lastCountersClearedTime ) )
         rates = OrderedDict()
         rates[ "Activity records (per second)" ] = counter.activityRecordsRates
         rates[ "Progress records (per second)" ] = counter.progressRecordsRates
         rates[ "Aggregate (Mbps)" ] = counter.throughputs
         self.printRates( "Sent", rates )

   def render( self ):
      if self.flowCounters:
         self.renderFlowCounters()
      self.renderNucleusCounters()
      if self.ipfixCounters:
         self.renderIpfixCounters()
      if self.activityRecordQueueFull is not None:
         print( "Activity record queue full: {}".format(
            renderCounter( self.activityRecordQueueFull ) ) )
      if self.packetBundleQueueFull is not None:
         print( "Packet bundle queue full: {}".format(
            renderCounter( self.packetBundleQueueFull ) ) )
      if self.flowTableFull is not None:
         print( "Flow table full: {}".format(
            renderCounter( self.flowTableFull ) ) )

class MsaDpiStructMemoryAllocations( Model ):
   numAlloc = Int( help="Current number of allocations" )
   maxAlloc = Int( help="Maximum number of allocations observed" )
   minAlloc = Int( help="Minimum number of allocations observed" )
   numBytes = Int( help="Current allocation in bytes" )
   maxBytes = Int( help="Maximum byte allocation observed" )
   minBytes = Int( help="Minimum byte allocation observed" )
   maxTime = Float( help="Time when the maximum number of allocations was observed" )
   minTime = Float( help="Time when the minimum number of allocations was observed" )

   def renderRow( self, table, ctxName, structName, withTs ):
      table.newRow(
         ctxName,
         structName,
         f"{self.numBytes / MB:.2f} MB, {self.numAlloc}",
         f"{self.minBytes / MB:.2f} MB, {self.minAlloc}",
         f"{self.maxBytes / MB:.2f} MB, {self.maxAlloc}" )
      if withTs:
         table.newRow(
            "",
            "",
            "",
            utcTimeRelativeToNowStr( self.minTime ),
            utcTimeRelativeToNowStr( self.maxTime ) )

class MsaDpiCtxMemoryAllocations( Model ):
   structures = Dict( keyType=str, valueType=MsaDpiStructMemoryAllocations,
                      help="Mapping from structure to allocations" )

class MsaDpiMemoryAllocations( Model ):
   contexts = Dict( keyType=str, valueType=MsaDpiCtxMemoryAllocations,
                    help="Mapping from context to allocations" )
   _sort = Str( help="Sort order" )
   _nt = Bool( help="No timestamps" )

   def _sortedItems( self ):
      items = []
      sort = { 'current': lambda item: item[ 2 ].numBytes,
               'minimum': lambda item: item[ 2 ].minBytes,
               'maximum': lambda item: item[ 2 ].maxBytes, }.get( self._sort )
      if sort:
         for ctxName, ctxAllocations in self.contexts.items():
            items += [ ( ctxName, structName, structAllocations )
                       for structName, structAllocations
                       in ctxAllocations.structures.items() ]
         return sorted( items, key=sort )
      else:
         for ctxName, ctxAllocations in sorted( self.contexts.items() ):
            items += [ ( ctxName, structName, structAllocations )
                       for structName, structAllocations
                       in sorted( ctxAllocations.structures.items() ) ]
         return items

   def render( self ):
      if not self.contexts:
         return
      headers = ( "Context", "Structure", "Current", "Minimum", "Maximum" )
      formats = [ formatLeft ] * len( headers )
      table = TableOutput.createTable( headers, tableWidth=120 )
      table.formatColumns( *formats )
      totalAlloc = 0
      totalMinAlloc = 0
      totalMaxAlloc = 0
      totalBytes = 0
      totalMinBytes = 0
      totalMaxBytes = 0
      for ctxName, structName, structAllocations in self._sortedItems():
         structAllocations.renderRow( table, ctxName, structName, not self._nt )
         totalAlloc += structAllocations.numAlloc
         totalMinAlloc += structAllocations.minAlloc
         totalMaxAlloc += structAllocations.maxAlloc
         totalBytes += structAllocations.numBytes
         totalMinBytes += structAllocations.minBytes
         totalMaxBytes += structAllocations.maxBytes
      table.newRow(
         "Total",
         "",
         "%.2f MB, %d" % ( totalBytes / MB, totalAlloc ),
         "%.2f MB, %d" % ( totalMinBytes / MB, totalMinAlloc ),
         "%.2f MB, %d" % ( totalMaxBytes / MB, totalMaxAlloc ) )
      print( table.output() )

class MsaDpiMemoryPoolDetails( Model ):
   blockSize = Int( help="Block size of this memory pool in bytes" )
   allocCount = Int( help="Number of blocks allocated" )
   allocSize = Int( help="Number of bytes allocated" )
   usedCount = Int( help="Number of blocks in use" )
   usedSize = Int( help="Number of bytes in use" )
   minimumCount = Int( help="Minimum number of blocks in use observed" )
   minimumSize = Int( help="Minimum number of bytes in use observed" )
   minimumTime = Float( help="Time when the minimum use was observed" )
   maximumCount = Int( help="Maximum number of blocks in use observed" )
   maximumSize = Int( help="Maximum number of bytes in use observed" )
   maximumTime = Float( help="Time when the maximum use was observed" )

   def renderRow( self, table, poolId ):
      if poolId == "total":
         label = "Total"
      else:
         label = "%d B" % self.blockSize

      usedPercent = self.usedSize * 100 // self.allocSize if self.allocSize else 0
      minPercent = self.minimumSize * 100 // self.allocSize if self.allocSize else 0
      maxPercent = self.maximumSize * 100 // self.allocSize if self.allocSize else 0
      table.newRow(
         label,
         f"{self.allocSize / MB:.1f} MB, {self.allocCount}",
         f"{self.usedSize / MB:.1f} MB, {self.usedCount}, {usedPercent}%",
         f"{self.minimumSize / MB:.1f} MB, {self.minimumCount}, {minPercent}%",
         f"{self.maximumSize / MB:.1f} MB, {self.maximumCount}, {maxPercent}%" )
      table.newRow(
         "",
         "",
         "",
         utcTimeRelativeToNowStr( self.minimumTime ),
         utcTimeRelativeToNowStr( self.maximumTime ) )

class MsaDpiMemoryPools( Model ):
   memoryPools = Dict( keyType=str, valueType=MsaDpiMemoryPoolDetails,
                       help="Mapping from pool ID to pool details" )

   def render( self ):
      if not self.memoryPools:
         return # do not print table headers when there are no rows
      headers = ( "Pool", "Capacity, Blocks", "Current Use", "Minimum Use",
                  "Maximum Use" )
      table = TableOutput.createTable( headers )
      table.formatColumns( formatRight, * 4 * ( formatLeft, ) )
      for poolId, memoryInfo in sorted( self.memoryPools.items() ):
         memoryInfo.renderRow( table, poolId )
      print( table.output() )


class MsaDpiMemoryDetails( Model ):
   current = Int( help="Current memory use in bytes" )
   minimum = Int( help="Minimum memory use observed in bytes" )
   maximum = Int( help="Maximum memory use observed in bytes" )
   minimumTime = Float( help="Time when the minimum memory use was observed" )
   maximumTime = Float( help="Time when the maximum memory use was observed" )

   def renderRow( self, table, label ):
      table.newRow( label,
                    "%.1f" % ( self.current / MB ),
                    "{:.1f} ({})".format( self.minimum / MB,
                                    utcTimeRelativeToNowStr( self.minimumTime ) ),
                    "{:.1f} ({})".format( self.maximum / MB,
                                    utcTimeRelativeToNowStr( self.maximumTime ) ) )

class MsaDpiMemory( Model ):
   allocated = Submodel( help="Allocated memory details", optional=True,
                         valueType=MsaDpiMemoryDetails )
   inUse = Submodel( help="In use memory details", optional=True,
                     valueType=MsaDpiMemoryDetails )

   def render( self ):
      if not any( ( self.allocated, self.inUse ) ):
         return # do not print table headers when there are no rows
      headers = ( "Memory", "Current (MB)", "Minimum (MB)", "Maximum (MB)" )
      table = TableOutput.createTable( headers, tableWidth=100 )
      table.formatColumns( *[ formatLeft ] * len( headers ) )
      for memoryInfo, label in ( ( self.allocated, "Allocated" ),
                                 ( self.inUse, "In use" ) ):
         if memoryInfo:
            memoryInfo.renderRow( table, label )
      print( table.output() )

class FlowDetail( Model ):
   lastUpdateTime = Float( help='Last flow update time' )

class Flow( Model ):
   lowerIpAddress = Submodel( help="Lower IP address",
                              valueType=IpGenericAddrAndPort )
   higherIpAddress = Submodel( help="Higher IP address",
                                valueType=IpGenericAddrAndPort )
   startTime = Float( help="Flow start time" )
   ipProtocol = Enum( help="IP protocol", values=IpProtoType.attributes )
   ipProtocolNumber = Int( help="IP protocol number" )
   numPacketsH2L = Int( help="Number of packets from higher IP to lower IP" )
   numPacketsL2H = Int( help="Number of packets from lower IP to higher IP" )
   numBytesH2L = Int( help="Number of bytes from higher IP to lower IP" )
   numBytesL2H = Int( help="Number of bytes from lower IP to higher IP" )
   flowDetail = Submodel( valueType=FlowDetail, optional=True,
                          help="Flow detailed information" )

class FlowGroup( Model ):
   flows = List( help="A list of active flows", valueType=Flow )

class FlowTable( Model ):
   groups = Dict( keyType=str, valueType=FlowGroup,
         help="A mapping between group name and group" )
   activeFlows = Int( help="Number of active flows", default=0 )

   def render( self ):
      ipv4Flows = self.groups.get( "IPv4", FlowGroup() ).flows
      ipv6Flows = self.groups.get( "IPv6", FlowGroup() ).flows
      self.printFlowTable( ipv4Flows, "IPv4" )
      self.printFlowTable( ipv6Flows, "IPv6" )

   @staticmethod
   def totalPackets( flowModel ):
      return flowModel.numPacketsH2L + flowModel.numPacketsL2H

   @staticmethod
   def totalBytes( flowModel ):
      return flowModel.numBytesH2L + flowModel.numBytesL2H

   def printFlowTable( self, flows, ipVersion ):
      headings = ( "Lower IP address", "Higher IP address", "Protocol",
         "Start Time", "Packets", "Bytes" )

      table = TableOutput.createTable( headings, tableWidth=100 )
      table.formatColumns(
         formatLeft,  # Lower IP Address
         formatLeft,  # Higher IP Address
         formatLeft,  # Protocol
         formatRight, # Start time
         formatRight, # Packets
         formatRight, # Bytes
      )
      for flow in flows:
         table.newRow(
            addressStr( flow.lowerIpAddress.ip, flow.lowerIpAddress.port ),
            addressStr( flow.higherIpAddress.ip, flow.higherIpAddress.port ),
            protocolStr( flow.ipProtocol, flow.ipProtocolNumber ),
            timeStr( flow.startTime ),
            self.totalPackets( flow ),
            self.totalBytes( flow ) )
         TacSigint.check()

      print( "{ipVersion} flows: {flowCount}".format(
            ipVersion=ipVersion,
            flowCount=len( flows ),
      ) )
      print( table.output() )

class FlowTableDetail( FlowTable ):
   def render( self ):
      ipv4Flows = self.groups.get( "IPv4", FlowGroup() ).flows
      ipv6Flows = self.groups.get( "IPv6", FlowGroup() ).flows
      if ipv4Flows or ipv4Flows:
         print( "Flow table detail codes: L2H - Lower to higher IP address, " +
                "H2L - Higher to lower IP address\n" )
      self.printFlowTableDetail( ipv4Flows, "IPv4" )
      self.printFlowTableDetail( ipv6Flows, "IPv6" )

   def printFlowTableDetail( self, flows, ipVersion ):
      print( "{ipVersion} flows: {flowCount}".format(
         ipVersion=ipVersion,
         flowCount=len( flows ) ),
      )
      if flows:
         for flow in flows:
            lowAddr = addressStr( flow.lowerIpAddress.ip, flow.lowerIpAddress.port )
            highAddr = addressStr( flow.higherIpAddress.ip,
                                   flow.higherIpAddress.port )
            print( "Flow: {proto} {lowAddr} - {highAddr}".format(
                  proto=protocolStr( flow.ipProtocol, flow.ipProtocolNumber ),
                  lowAddr=lowAddr,
                  highAddr=highAddr,
               )
            )
            print( "Start time: {st}, Last packet time: {lpt}".format(
                  st=timeStr( flow.startTime ),
                  lpt=timeStr( flow.flowDetail.lastUpdateTime ),
               )
            )
            print( "Packets L2H: {plh}, Bytes L2H: {blh}, "
                   "Packets H2L: {phl}, Bytes H2L: {bhl}".format(
                      plh=flow.numPacketsL2H,
                      blh=flow.numBytesL2H,
                      phl=flow.numPacketsH2L,
                      bhl=flow.numBytesH2L,
                   ),
            )
            print()
      else:
         print()
