# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Cell
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import LazyMount
import Smash
import SmashLazyMount
import Tac

entityManager = None
vrfBgpAttrInfoStatus = None
vrfBgpGlobalInfoStatus = None
vrfBgpPeerInfoStatusEntryTable = None
bgpExportVrfStatusV4 = None
bgpExportVrfStatusV6 = None
l3Config = None

#=====================================================
# Table pretty printer
#=====================================================
class TablePrint:
   def __init__( self, header ):
      self.data_ = [ header ]

   def addRow( self, row ):
      self.data_.append( [str(s) for s in row] )

   def printTable( self ):
      widths = [ max( map( len, col ) ) for col in zip( *self.data_ ) ]
      for row in self.data_:
         print( "  ".join( val.rjust( width ) for val, width in
                           zip( row, widths ) ) )

#=====================================================
# Mark commands unsupported in multi-agent mode.
#=====================================================
def unsupportedInMultiAgent( mode ):
   protocolAgentModel = l3Config.protocolAgentModel
   if protocolAgentModel == ProtoAgentModel.multiAgent:
      mode.addError( 'Not supported' )
      return True
   return False

#=====================================================
# show bgp export-table bgp-nexthops ( vrf <vrfName> )
#=====================================================
def showBgpNexthopsTable( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if unsupportedInMultiAgent( mode ):
      return
   if not vrfBgpAttrInfoStatus.entity.get( vrfName ):
      return
   bgpNexthopInfo = vrfBgpAttrInfoStatus[ vrfName ].bgpNexthopInfo

   print( "Bgp Nexthops Table" )
   with Tac.ActivityLockHolder():
      for rnh, bgpNhEntry in bgpNexthopInfo.items():
         nexthops = " ".join( bnh.stringValue for bnh in bgpNhEntry.bgpNexthop )
         print( "Resolved Nexthop: {} BGP Nexthops: {}".format( rnh.stringValue,
                                                            nexthops ) )

#=====================================================
# show bgp export-table bgp-attributes-id ( vrf <vrfName> )
#=====================================================
def showBgpAttrsIdTable( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if unsupportedInMultiAgent( mode ):
      return
   if not vrfBgpAttrInfoStatus.entity.get( vrfName ):
      return
   bgpAttrSetIdInfo = vrfBgpAttrInfoStatus[ vrfName ].bgpAttrSetIdInfo

   def getBgpAttrString( bgpNh, bgpAttr ):
      return "%s,%d" % ( bgpNh.stringValue, bgpAttr )

   print( "Bgp Attributes Id Table" )
   with Tac.ActivityLockHolder():
      for setId, bgpAttrIds in bgpAttrSetIdInfo.items():
         bgpAttrsString = ""
         for bgpNh, bgpAttrId in bgpAttrIds.attrInfoId.items():
            bgpAttrsString += getBgpAttrString( bgpNh, bgpAttrId )
            bgpAttrsString += " "
         print( "Id: %d Bgp Attr Ids: %s" % ( setId, bgpAttrsString ) )

#==========================================================
# show bgp export-table rib ( vrf <vrfName> )
#==========================================================

def showBgpAttrSmashTable( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   addrFamily = 'ipv4-unicast' if 'ipv4-unicast' in args else 'ipv6-unicast'
   if unsupportedInMultiAgent( mode ):
      return
   if not vrfName:
      vrfName = DEFAULT_VRF
   if addrFamily == 'ipv4-unicast':
      if ( not bgpExportVrfStatusV4.entity.get( vrfName ) ) or \
         bgpExportVrfStatusV4[ vrfName ].state == \
         bgpExportVrfStatusV4[ vrfName ].defaultState:
         return
      smi = Smash.mountInfo( 'reader' )
      bgpRouteInfoStatusStatus = SmashLazyMount.mount(
         entityManager,
         'routing/bgp/bgpRouteInfoStatus/%s' % vrfName,
         'Smash::Routing::BgpRouteInfoStatus', smi )
      print( "Bgp Export Route Info IPv4" )
      print( "Route\tAttr-Id\tLocalPref" )
      for index, entry in \
          bgpRouteInfoStatusStatus.bgpRouteInfoStatusEntry.items():
         print( "%s\t%d\t%d\t%s" % ( index.stringValue, entry.id,
                                     entry.localPref, entry.ecmp ) )

   else:
      if ( not bgpExportVrfStatusV6.entity.get( vrfName ) ) or \
         bgpExportVrfStatusV6[ vrfName ].state == \
         bgpExportVrfStatusV6[ vrfName ].defaultState:
         return
      smi = Smash.mountInfo( 'reader' )
      bgpRouteInfoStatusStatus = SmashLazyMount.mount(
         entityManager,
         'routing6/bgp/bgpRouteInfoStatus/%s' % vrfName,
         'Smash::Routing::BgpRouteInfoStatus', smi )
      print( "Bgp Export Route Info IPv6" )
      print( "Route\tAttr-Id\tLocalPref" )
      for index, entry in \
            bgpRouteInfoStatusStatus.bgpRouteInfoStatusEntry.items():
         print( "%s\t%d\t%d\t%s" % ( index.stringValue, entry.id,
                                     entry.localPref, entry.ecmp ) )

#==========================================================
# show bgp export-table peer ( vrf <vrfName> )
#==========================================================
def showBgpPeerSmashTable( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if not vrfBgpPeerInfoStatusEntryTable.entity.get( vrfName ):
      return
   smi = Smash.mountInfo( 'reader' )
   AfiSafiIndexEnum = Tac.Type( 'Routing::Bgp::AfiSafiIndexEnum' )
   ipv4Index = Tac.enumValue( AfiSafiIndexEnum, 'ipv4Unicast' )
   ipv6Index = Tac.enumValue( AfiSafiIndexEnum, 'ipv6Unicast' )
   bgpPeerInfoStatusEntryTable = vrfBgpPeerInfoStatusEntryTable[ vrfName ]
   bgpPeerInfoStatus = SmashLazyMount.mount( entityManager,
         'routing/bgp/bgpPeerInfoStatus/%s' % vrfName,
         'Smash::Routing::BgpPeerInfoStatus', smi )
   print( "Bgp Export Peer Info" )
   tp = TablePrint( ["Peer Address", "Id", "AS", "Bgp State(Sysdb)",
                     "Bgp State(Smash)", "Peer Router Id", "TCP Local Addr",
                     "Established Time", "Established Transitions", "InUpdates",
                     "OutUpdates", "Prefix Sent V4", "Prefix Sent V6",
                     "Prefix Rcvd V4", "Prefix Rcvd V6", "Admin Status",
                     "bgpPeerInitExportFin" ] )

   for index, entry in bgpPeerInfoStatusEntryTable.bgpPeerInfoStatusEntry.items():
      if index not in bgpPeerInfoStatus.bgpPeerStatisticsEntry or \
         index not in bgpPeerInfoStatus.bgpPeerStatusEntry:
         continue
      statEntry = bgpPeerInfoStatus.bgpPeerStatisticsEntry[ index ]
      statusEntry = bgpPeerInfoStatus.bgpPeerStatusEntry[ index ]
      ipv4_stats = statEntry.bgpPeerAfiSafiStats[ ipv4Index ]
      ipv6_stats = statEntry.bgpPeerAfiSafiStats[ ipv6Index ]
      tp.addRow( [index.stringValue, entry.bgpPeerId, entry.bgpPeerAs,
                  entry.bgpState, statusEntry.bgpState, entry.bgpPeerRouterId,
                  entry.bgpPeerLocalAddr, entry.bgpPeerIntoOrOutOfEstablishedTime,
                  entry.bgpPeerEstablishedTransitions, statEntry.bgpPeerInUpdates,
                  statEntry.bgpPeerOutUpdates, ipv4_stats.prefixOut,
                  ipv6_stats.prefixOut, ipv4_stats.prefixIn, ipv6_stats.prefixIn,
                  entry.bgpPeerAdminShutDown, entry.bgpPeerInitExportFin] )

   tp.printTable()

#==========================================================
# show bgp export-table adj-rib-in ( vrf <vrfName> )
#==========================================================
def showBgpPathSmashTable( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if unsupportedInMultiAgent( mode ):
      return
   smi = Smash.mountInfo( 'reader' )
   mountPath = 'routing/bgp/export/allPeerAdjRibIn/%s' % vrfName
   allPeerAdjRibIn = SmashLazyMount.mount( entityManager,
         mountPath, 'Smash::Routing::AllPeerAdjRibIn',
         smi )
   print( "Bgp Export AdjRibIn Path Info" )

   # TODO: Eliminate afi/safi columns
   # template = "{0:<20}  {1:<%d}  {2:<%d}  {3:<%d}  {4:<%d}" % (
   #      len( 'Peer ID' ), len( 'Add Path ID' ),
   #      len( 'Post-Policy ID' ), len( 'Pre-Policy ID' ) )
   # print template.format( 'Prefix', 'Peer ID', 'Add Path ID',
   #      'Post-Policy ID', 'Pre-Policy ID' )
   template = "{0:<%d}  {1:<%d}  {2:<20}  {3:<%d}  {4:<%d}  {5:<%d}" % (
         len( 'Afi' ), len( 'Safi' ), len( 'PeerId ' ),
         len( 'Post-Policy ID' ), len( 'Pre-Policy ID' ) )
   print( template.format( 'Afi', 'Safi', 'Prefix', 'Peer ID',
                           'Post-Policy ID', 'Pre-Policy ID' ) )

   # print "IPv4 Unicast"
   for entry in allPeerAdjRibIn.v4UnicastPathEntry.values():
      print( template.format( 1, 1, str( entry.key.prefix ),
                              entry.key.peerId, entry.postPolicyPathAttrId,
                              entry.prePolicyPathAttrId ) )

   # print "IPv6 Unicast"
   for entry in allPeerAdjRibIn.v6UnicastPathEntry.values():
      print( template.format( 2, 1, str( entry.key.prefix ),
                              entry.key.peerId, entry.postPolicyPathAttrId,
                              entry.prePolicyPathAttrId ) )

#==========================================================
# show bgp export-table aspath ( vrf <vrfName> )
#==========================================================
def showBgpAsPathSmashTable( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if unsupportedInMultiAgent( mode ):
      return
   smi = Smash.mountInfo( 'reader' )
   mountPath = 'routing/bgp/export/pathAttrTableInfo/%s' % vrfName
   pathAttrTableInfo = SmashLazyMount.mount( entityManager, mountPath,
                                        'Smash::Routing::PathAttrTableInfo', smi )
   print( "Bgp Export AdjRibIn AsPath Info" )
   print( "Id\tAsPath\t" )
   for entry in pathAttrTableInfo.asPathAttrInfoEntry.values():
      print( f"{entry.key}\t{entry.toStrep()}\t" )

#==========================================================
# show bgp export-table community ( vrf <vrfName> )
#==========================================================
def showBgpCommSmashTable( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if unsupportedInMultiAgent( mode ):
      return
   smi = Smash.mountInfo( 'reader' )
   mountPath = 'routing/bgp/export/pathAttrTableInfo/%s' % vrfName
   pathAttrTableInfo = SmashLazyMount.mount( entityManager, mountPath,
                                        'Smash::Routing::PathAttrTableInfo', smi )
   print( "Bgp Export AdjRibIn Community Info" )
   print( "Id\tCommunity\t" )
   for entry in pathAttrTableInfo.commListInfoEntry.values():
      print( f"{entry.key}\t{entry.toStrep()}\t" )

#==========================================================
# show bgp export-table path-attribute ( vrf <vrfName> )
#==========================================================
def showBgpAttrTable( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if unsupportedInMultiAgent( mode ):
      return
   smi = Smash.mountInfo( 'reader' )
   mountPath = 'routing/bgp/export/pathAttrTableInfo/%s' % vrfName
   pathAttrTableInfo = SmashLazyMount.mount( entityManager, mountPath,
                                        'Smash::Routing::PathAttrTableInfo', smi )
   print( "Bgp Attribute Info Table" )
   tp = TablePrint( [ "key", "nextHop", "origin", "pathFlags", "originatorId",
                      "med", "localPref", "asPathAttrInfoId",
                      "commListId", "extCommListId" ] )
   for entry in pathAttrTableInfo.pathAttrEntry.values():
      tp.addRow( [ entry.key, entry.nextHop, entry.origin, entry.pathFlags,
                   entry.originatorId, entry.med,
                   entry.localPref, entry.asPathAttrInfoId, entry.commListId,
                   entry.extCommListId ] )
   tp.printTable()

#==========================================================
# show bgp export-table ext-community ( vrf <vrfName> )
#==========================================================
def showBgpExtCommSmashTable( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   raw = 'raw' in args
   if unsupportedInMultiAgent( mode ):
      return
   if not vrfName:
      vrfName = DEFAULT_VRF
   smi = Smash.mountInfo( 'reader' )
   mountPath = 'routing/bgp/export/pathAttrTableInfo/%s' % vrfName
   pathAttrTableInfo = SmashLazyMount.mount( entityManager, mountPath,
                                        'Smash::Routing::PathAttrTableInfo', smi )
   print( "Bgp Export AdjRibIn Extended Community Info" )
   print( "Id\tExtended Community\t" )
   if raw:
      for entry in pathAttrTableInfo.extCommListInfoEntry.values():
         print( f"{entry.key}\t{entry.toRawStrep()}\t" )
   else:
      for entry in pathAttrTableInfo.extCommListInfoEntry.values():
         print( f"{entry.key}\t{entry.toStrep()}\t" )

#==========================================================
# show bgp export-table global-info ( vrf <vrfName> )
#==========================================================
def showBgpGlobalInfo( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if not vrfBgpGlobalInfoStatus.entity.get( vrfName ):
      return
   bgpGlobalInfoStatus = vrfBgpGlobalInfoStatus[ vrfName ]
   template = "{0:<%d}  {1:<%d}  {2:<%d}  {3:<%d}  {4:<%d}" % (
         len( 'Vrf Name' ), len( 'AS Number' ), len( 'Router Id' ),
         len( 'Total Paths' ), len( 'InitDumpComplete' ) )
   print( "Bgp Export Global Info" )
   print( template.format( 'Vrf Name', 'AS Number', 'Router Id',
                           'Total Paths', 'InitDumpComplete' ) )
   print( template.format( vrfName, bgpGlobalInfoStatus.asNumber,
                           bgpGlobalInfoStatus.routerId,
                           bgpGlobalInfoStatus.totalPaths,
                           bgpGlobalInfoStatus.bgpInitExpComplete ) )

def Plugin( entMan ):
   global entityManager
   global vrfBgpAttrInfoStatus, bgpExportVrfStatusV4, bgpExportVrfStatusV6
   global vrfBgpGlobalInfoStatus, vrfBgpPeerInfoStatusEntryTable
   global l3Config

   entityManager = entMan
   vrfBgpAttrInfoStatus = LazyMount.mount(
      entMan,
      Cell.path( 'routing/bgp/export/vrfBgpAttrInfoStatus' ),
      'Tac::Dir', 'ri' )
   vrfBgpGlobalInfoStatus = LazyMount.mount(
      entMan,
      Cell.path( 'routing/bgp/export/vrfBgpGlobalInfoStatus' ),
      'Tac::Dir', 'ri' )
   vrfBgpPeerInfoStatusEntryTable = LazyMount.mount(
      entMan,
      Cell.path( 'routing/bgp/export/vrfBgpPeerInfoStatusEntryTable' ),
      'Tac::Dir', 'ri' )
   bgpExportVrfStatusV4 = LazyMount.mount(
      entMan,
      Cell.path( 'routing/bgp/export/vrfStatus/ipv4' ),
      'Tac::Dir', 'ri' )
   bgpExportVrfStatusV6 = LazyMount.mount(
      entMan,
      Cell.path( 'routing/bgp/export/vrfStatus/ipv6' ),
      'Tac::Dir', 'ri' )
   l3Config = LazyMount.mount( entityManager, 'l3/config',
                            'L3::Config', 'r' )
