# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# -----------------------------------------------------------------------------------
# This module provides an interface to retrieve show tech state
# for multicast (across default and non-default VRFs)
# -----------------------------------------------------------------------------------

from __future__ import absolute_import, division, print_function

import sys
import json
import ArPyUtils
import BasicCliModes
import BasicCliSession
import CliParser
import CliCommon
import CliModel
import CliApi
import TacSigint

from sortedcontainers import SortedSet
from IpLibConsts import ALL_VRF_NAME
from CliPlugin.VrfCli import (
      getAllVrfNames,
      vrfExists,
)
from CliPlugin.GmpShowTechCli import (
      showTechMulticastCallbacks,
      showTechMulticastVrfCallbacks,
      showTechSummaryCallbacks,
)

class ShowTechMulticastModel( CliModel.DeferredModel ):
   __public__ = False
   showTechMulticast = CliModel.Str( help="show tech-support multicast model" )

def handlerShowCommandSet( cmdsSet, disabledMcastCmds, mode,
                           jsonFmt, capiExecutor=None, vrf=None, firstCmd=True ):
   assert cmdsSet

   for cmd in ( cmdsSet ):
      if cmd in disabledMcastCmds:
         continue
      if vrf:
         currCmd = cmd.format( vrf )
      else:
         currCmd = cmd
      if jsonFmt:
         if currCmd in CliCommon.skippedJsonCmds:
            # commands are buggy, skip them
            continue
         if not firstCmd:
            print( ',' )
         print( f"\"{currCmd}\":" )
      else:
         print( f"\n------------- {currCmd} -------------\n" )
      sys.stdout.flush()
      try:
         if jsonFmt:
            print( '{ "json":' )
            sys.stdout.flush()
            result = capiExecutor.executeCommands( [ CliApi.CliCommand( currCmd ) ],
                                                   textOutput=False,
                                                   autoComplete=True,
                                                   streamFd=sys.stdout.fileno() )
            if result[ 0 ].status == CliApi.CapiStatus.NOT_CAPI_READY:
               with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno() ] ) \
                    as capturedStdout:
                  capiExecutor.executeCommands( [ CliApi.CliCommand( currCmd ) ],
                                                textOutput=True,
                                                autoComplete=True,
                                                streamFd=sys.stdout.fileno() )
                  output = capturedStdout.contents()
                  if isinstance( output, bytes ):
                     output = output.decode( 'utf-8', 'replace' )
                  strJson = json.dumps( output )
                  print( ', "text": { "output": ' )
                  print( f"{strJson}" )
                  print( '}' )
            print( '}' )
            sys.stdout.flush()
            firstCmd = False
         else:
            mode.session_.runCmd( currCmd, aaa=False )
      except CliParser.GuardError as e:
         if not jsonFmt:
            print( f"(unavailable: {cmd} {e.guardCode})" )
      except CliParser.AlreadyHandledError:
         pass
      except KeyboardInterrupt:
         raise
      except: # pylint: disable=bare-except
         # Catch all errors, so that one command failure doesn't cause
         # output from others to be skipped.
         if not jsonFmt:
            print( f"(unavailable command: {cmd})" )
         disabledMcastCmds.add( cmd )
      TacSigint.check()

   return firstCmd

def handlerShowTechMulticastSupport( mode, args ):
   # parse vrf commands
   vrfName = args.get( "VRF_NAME", ALL_VRF_NAME )
   if vrfName == ALL_VRF_NAME:
      vrfList = getAllVrfNames()
   else:
      if not vrfExists( vrfName ):
         return CliModel.noValidationModel( ShowTechMulticastModel )
      vrfList = [ vrfName ]

   jsonFmt = mode.session.outputFormat_ == 'json'

   cmdsSet = SortedSet()
   for cmd in ( showTechMulticastCallbacks ):
      cmdsSet.update( cmd() )

   cmdsVrfSet = SortedSet()
   for cmd in ( showTechMulticastVrfCallbacks ):
      cmdsVrfSet.update( cmd() )

   cmdsSummarySet = SortedSet()
   for cmd in ( showTechSummaryCallbacks ):
      cmdsSummarySet.update( cmd() )

   if not cmdsVrfSet and not cmdsSummarySet and not cmdsSet:
      return CliModel.noValidationModel( ShowTechMulticastModel )

   # Handle json
   capiExecutor = None
   if jsonFmt:
      session = BasicCliSession.Session( BasicCliModes.EnableMode,
                                         mode.session.entityManager,
                                         privLevel=CliCommon.MAX_PRIV_LVL,
                                         disableAaa=True,
                                         disableAutoMore=True,
                                         isEapiClient=True,
                                         shouldPrint=False,
                                         disableGuards=not
                                         mode.session.guardsEnabled(),
                                         interactive=False,
                                         cli=mode.session.cli,
                                         aaaUser=mode.session.aaaUser() )
      capiExecutor = CliApi.CapiExecutor( mode.session.cli, session,
                                          stateless=False )
      print( '{\n' )
      sys.stdout.flush()

   disabledMcastCmds = set()
   firstCmd = True
   if cmdsSet:
      firstCmd = handlerShowCommandSet( cmdsSet, disabledMcastCmds, mode,
                                        jsonFmt, capiExecutor )

   if cmdsVrfSet:
      # parse vrf commands
      for vrf in sorted( vrfList ):
         if not jsonFmt:
            print( f"\n------------- VRF: {vrf} -------------\n" )
         firstCmd = handlerShowCommandSet( cmdsVrfSet, disabledMcastCmds, mode,
                                           jsonFmt, capiExecutor, vrf, firstCmd )

   if cmdsSummarySet:
      # FIXME: json issue with "show pim evpn gateway dr"
      # FIXME: json issue with "show multicast fib ipv4 vrf {} summary"
      if not jsonFmt:
         handlerShowCommandSet( cmdsSummarySet, disabledMcastCmds, mode,
                                jsonFmt, firstCmd )

   if jsonFmt:
      print( '}\n' )
      sys.stdout.flush()
   else:
      # Clear any errors, that pertain to the last run command
      mode.session_.clearMessages()
   return CliModel.noValidationModel( ShowTechMulticastModel )
