# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from HttpServiceConstants import ServerConstants
from CliMode.HttpService import HttpServerVrfConfigModeBase
import CliCommand
from CliPlugin import AclCli
import CliGlobal
import ConfigMount
from socket import IPPROTO_TCP
import AclCliLib

vrfShutdownHelpdesc = 'Disable server access in this VRF'
aclServiceName = ServerConstants.serviceName

# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
gv = CliGlobal.CliGlobal(
   dict(
      capiConfig = None,
      aclConfig = None,
      aclCpConfig = None,
   )
)

#-------------------------------------------------------------------------------
# The "[no | default] shutdown" command,
# in "vrf <vrf name>" mode
# under "management http-server" mode.
#-------------------------------------------------------------------------------

class HttpServerVrfConfigMode( HttpServerVrfConfigModeBase,
                               BasicCli.ConfigModeBase ):
   name = "HTTP server VRF configuration"

   def __init__( self, parent, session, vrfName ):
      HttpServerVrfConfigModeBase.__init__( self, ( vrfName, gv.capiConfig ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setServiceAcl( self, aclType, aclName, remove=False ):
      if remove:
         AclCliLib.noServiceAcl( self, aclServiceName, gv.aclConfig, gv.aclCpConfig,
                                 aclName, aclType, self.vrfName_ )
      else:
         AclCliLib.setServiceAcl( self, aclServiceName, IPPROTO_TCP,
                                  gv.aclConfig, gv.aclCpConfig, aclName, 
                                  aclType, self.vrfName_,
                                  port=[ gv.capiConfig.httpConfig.port,
                                         gv.capiConfig.httpsConfig.port ] )

   def setIpAcl( self, args ):
      self.setServiceAcl( 'ip', args[ 'ACL_NAME' ] )

   def noIpAcl( self, args=None ):
      aclName = args.get( 'ACL_NAME' ) if args else None
      self.setServiceAcl( 'ip', aclName, remove=True )

   def setIp6Acl( self, args ):
      self.setServiceAcl( 'ipv6', args[ 'ACL_NAME' ] )

   def noIp6Acl( self, args=None ):
      aclName = args.get( 'ACL_NAME' ) if args else None
      self.setServiceAcl( 'ipv6', aclName, remove=True )


class VrfModeShutdown( CliCommand.CliCommandClass ):
   syntax = '''shutdown'''
   noSyntax = '''shutdown ...'''
   defaultSyntax = '''shutdown ...'''
   data = { 'shutdown': vrfShutdownHelpdesc }
   hidden = True

   handler = "HttpServiceHandler.vrfModeShutDownHandler"
   noHandler = "HttpServiceHandler.vrfModeShutDownNoHandler"
   defaultHandler = "HttpServiceHandler.vrfModeShutDownDefaultHandler"

HttpServerVrfConfigMode.addCommandClass( VrfModeShutdown )


#-----------------------------------------------------------------------------------
# [ no | default ] ip access-group <name> [ vrf VRF ] in
#-----------------------------------------------------------------------------------
class IpAccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_NAME in'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] in'
   data = {
            'ip': AclCli.ipKwForServiceAclMatcher, 
            'access-group': AclCli.accessGroupKwMatcher, 
            'ACL_NAME': AclCli.ipAclNameMatcher,
            'in': AclCli.inKwMatcher 

          }
   handler = "HttpServiceHandler.doSetIpAclCmd"
   noOrDefaultHandler = "HttpServiceHandler.noIpAclCmd"

HttpServerVrfConfigMode.addCommandClass( IpAccessGroup )

#-----------------------------------------------------------------------------------
# [ no | default ] ipv6 access-group <name> [ vrf VRF ] in
#-----------------------------------------------------------------------------------
class Ipv6AccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME in'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] in'
   data = {
            'ipv6': AclCli.ipv6KwMatcherForServiceAcl, 
            'access-group': AclCli.accessGroupKwMatcher, 
            'ACL_NAME': AclCli.ip6AclNameMatcher,
            'in': AclCli.inKwMatcher 

          }
   handler = "HttpServiceHandler.doSetIp6AclCmd"
   noOrDefaultHandler = "HttpServiceHandler.noIp6AclCmd"

HttpServerVrfConfigMode.addCommandClass( Ipv6AccessGroup )



# Plug-in definition:
def Plugin( entityManager ):
   gv.capiConfig = ConfigMount.mount( entityManager,
                                   "mgmt/capi/config",
                                   "HttpService::Config",
                                   "w" )
   gv.aclConfig = ConfigMount.mount( entityManager,
                                  "acl/config/cli",
                                  "Acl::Input::Config",
                                  "w" )
   gv.aclCpConfig = ConfigMount.mount( entityManager,
                                    "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig",
                                    "w" )