# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
      Bool,
      Dict,
      Int,
      Model,
      Str,
      Submodel )
from Toggles.IcmpResponderToggleLib import (
      toggleIcmpExtendedErrorResponderEnabled,
)

INDENT_SIZE = 2

def afActiveState( v4Active, v6Active ):
   activeStates = []
   if v4Active:
      activeStates.append( 'IPv4' )
   if v6Active:
      activeStates.append( 'IPv6' )
   if activeStates:
      afActive = ', '.join( activeStates )
   else:
      afActive = 'none'
   return afActive

class ErrorResponderVrfModel( Model ):
   __public__ = toggleIcmpExtendedErrorResponderEnabled()
   v4Responding = Bool( help='Responding for IPv4 address family' )
   v6Responding = Bool( help='Responding for IPv6 address family' )

   v4ExtensionsActive = Bool( help='Extensions active for IPv4 address family',
                              optional=True )
   v6ExtensionsActive = Bool( help='Extensions active for IPv6 address family',
                              optional=True )

   v4AclName = Str( help='Name of ACL configured for IPv4 traffic',
                  optional=True )
   v4AclActive = Bool( help='IPv4 ACL active on VRF',
                      optional=True )

   v6AclName = Str( help='Name of ACL configured for IPv6 traffic',
                    optional=True )
   v6AclActive = Bool( help='IPv6 ACL active on VRF',
                       optional=True )

   def render( self ):
      afRespondingDesc = 'Responding address family'
      afResponding = afActiveState( self.v4Responding, self.v6Responding )
      print( f'{afRespondingDesc}: {afResponding}' )

      afInfo = []
      if self.v4Responding:
         afInfo.append( ( 'IPv4', self.v4ExtensionsActive,
                          self.v4AclName, self.v4AclActive ) )
      if self.v6Responding:
         afInfo.append( ( 'IPv6', self.v6ExtensionsActive,
                          self.v6AclName, self.v6AclActive ) )

      for af, extensionsActive, aclName, aclActive in afInfo:
         if aclName:
            status = 'applied' if aclActive else 'not found'
            aclSuffix = f', ACL {aclName} {status}'
         else:
            aclSuffix = ''
         activeState = 'active' if extensionsActive else 'inactive'
         extensionsActiveMsg = f'{af} extensions: {activeState}{aclSuffix}'
         print( extensionsActiveMsg )

class ErrorResponderModel( Model ):
   __public__ = toggleIcmpExtendedErrorResponderEnabled()
   v4Active = Bool( help='Active for IPv4 address family' )
   v6Active = Bool( help='Active for IPv6 address family' )
   vrfs = Dict( keyType=str, valueType=ErrorResponderVrfModel,
                help='ICMP Error responder information for each enabled VRF' )

   def render( self ):
      afActiveDesc = 'Active address family'
      afActive = afActiveState( self.v4Active, self.v6Active )
      print( f'{afActiveDesc}: {afActive}' )

      for vrf in sorted( self.vrfs ):
         # Print a newline between each vrf,
         # and between AF state and first state
         print( '' )
         print( f'VRF {vrf}' )
         vrfModel = self.vrfs[ vrf ]
         vrfModel.render()

class ErrorResponderVrfCountersModel( Model ):
   __public__ = toggleIcmpExtendedErrorResponderEnabled()
   packetsProcessed = Int( 'Valid packet objects received from kernel on this VRF' )
   socketNotFound = Int(
         'Packet objects dropped because socket was not yet initialized' )
   extendedIcmpErrorsSent = Int( 'ICMP errors sent out with extensions' )
   icmpErrorsSent = Int( 'ICMP errors sent out without extensions' )
   socketErrors = Int( 'ICMP error packets dropped by the socket' )

   def render( self ):
      print( f'Packets processed: {self.packetsProcessed}' )
      print( f'Output socket uninitialized drops: {self.socketNotFound}' )
      print( f'Extended ICMP errors sent: {self.extendedIcmpErrorsSent}' )
      print( f'ICMP errors sent: {self.icmpErrorsSent}' )
      print( f'Output socket errors: {self.socketErrors}' )

class ErrorResponderAfCountersModel( Model ):
   __public__ = toggleIcmpExtendedErrorResponderEnabled()
   enqueued = Int( 'Packet objects enqueued by kernel' )
   queueDropped = Int( 'Packet objects queue-dropped by kernel' )
   dequeued = Int( 'Total packet objects dequeued by IcmpResponder' )

   invalidPktObj = Int( 'Invalid packet objects dequeued by IcmpResponder' )
   originalDatagramInPktObjTooShort = Int( 'Original datagaram too short in the '
                                           'dequeued packet object' )
   invalidVrfInPktObj = Int( 'Packet info object could not '
                             'be mapped to a valid VRF' )
   packetsProcessed = Int( 'Packet info objects that pass all checks' )
   vrfs = Dict( keyType=str, valueType=ErrorResponderVrfCountersModel,
                help="ICMP Error responder counters for each enabled VRF" )

   def render( self ):
      print( f'Enqueued: {self.enqueued}' )
      print( f'Queue-dropped: {self.queueDropped}' )
      print( f'Dequeued: {self.dequeued}' )
      print( f'Invalid packet-objects: {self.invalidPktObj}' )
      print( 'Original datagram too short: '
             f'{self.originalDatagramInPktObjTooShort}' )
      print( f'Invalid VRF: {self.invalidVrfInPktObj}' )
      print( f'Packets processed: {self.packetsProcessed}' )

      for vrf in sorted( self.vrfs ):
         vrfCountersModel = self.vrfs[ vrf ]
         # Print a newline between each vrf,
         # and between AF counters and first vrf counter
         print( '' )

         print( f'VRF {vrf}' )
         vrfCountersModel.render()

class ErrorResponderCountersModel( Model ):
   __public__ = toggleIcmpExtendedErrorResponderEnabled()
   ipv4Counters = Submodel( valueType=ErrorResponderAfCountersModel,
                            help='ICMPv4 Error responder counters',
                            optional=True )
   ipv6Counters = Submodel( valueType=ErrorResponderAfCountersModel,
                            help='ICMPv6 Error responder counters',
                            optional=True )

   def render( self ):
      if self.ipv4Counters:
         print( 'IPv4 counters' )
         self.ipv4Counters.render()

      if self.ipv4Counters and self.ipv6Counters:
         # Print a newline between IPv4 and IPv6 output
         print( '' )

      if self.ipv6Counters:
         print( 'IPv6 counters' )
         self.ipv6Counters.render()
