# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Submodel, Dict, Enum, Int, Str, List
import sys
from itertools import chain
from CliPlugin.InotifyModel import ALL_QUERIES

class InotifyProc( Model ):
   count = Int( help='The number of inotify FDs owned by the process' )
   cmdline = Str( help='The /proc/PID/cmdline of the process', optional=True )
   owner = Str( help='The owner of the process', optional=True )

class InotifyFile( Model ):
   filepath = Str( help='The path of the file being watched' )
   watchDescriptor = Int( help='The watch descriptor associated with the watch' )

class InotifyFiles( Model ):
   files = List( valueType=InotifyFile,
                 help='Filepaths and corresponding watch descriptors' )

class InotifyPid( Model ):
   cmdline = Str( help='The /proc/PID/cmdline of the process', optional=True )
   fds = Dict( keyType=int, valueType=InotifyFiles, help='A mapping from '
         'an inotify instance file descriptor to its watched files' )

class InotifyPids( Model ):
   pids = Dict( keyType=int, valueType=InotifyPid, help='A mapping from a '
         'process ID to its inotify instances' )

class InotifyQueryFilesModel( Model ):
   _pid = Int( help='The process identifier to query for ' )
   files = List( valueType=InotifyFile, help='Files in var/shmem being watched '
                                             'by the process using inotify' )

   def render( self ):
      print( f'Watch descriptor and files in /var/shmem watched by the process '
             f'with PID {self._pid}:\n' )
      title = ( "Watch Descriptor", "File Path" )
      formatStr = f'{{:>{len( title[0] )}}}   {{}}'
      print( formatStr.format( *title ) + '\n' )
      for f in self.files:
         print( formatStr.format( f.watchDescriptor, f.filepath ) )
      print( f'-----\nTotal files: {len( self.files )}' )

class InotifyQueryModel( Model ):
   _query = Enum( chain( ALL_QUERIES, [ 'all' ] ), help='The issued inotify query' )
   count = Int( optional=True,
                help='Count of total inotify FDs currently allocated' )
   limits = Dict( valueType=int, optional=True,
                  help='A mapping of the inotify limit to its value' )
   ps = Dict( valueType=InotifyProc, optional=True,
              help='A mapping of the /proc/PID/fd of each process using inotify '
                   'to its process information' )
   user = Dict( valueType=InotifyProc, keyType=int, optional=True,
                 help='A mapping of the number of inotify FDs in use by each '
                      'user' )
   watch = Submodel( valueType=InotifyPids, optional=True,
                        help='Inotify watches grouped by PID' )

   def renderHelper( self, query ):
      if query == 'count':
         print( f'Number of FDs Used: {self.count}' )
      elif query == 'limits':
         for category in self.limits:
            print( f'{category}: {self.limits[ category ]}\n' )
      elif query == 'ps':
         print( 'FDs Used by Each Process:\n' )
         for procPid in self.ps:
            procInfo = self.ps[ procPid ]
            print( f'{procInfo.count} {procPid} {procInfo.cmdline}' )
      elif query == 'user':
         print( 'FDs Used by Each User:\n' )
         for uid in self.user:
            procInfo = self.user[ uid ]
            print( f'{procInfo.count} {procInfo.owner}' )
      elif query == 'watch':
         print( 'Number of inotify watches in /var/shmem used by each '
                'process:\n' )
         totalWatches = 0
         title = ( "Count", "Pid", "Name" )
         formatStr = '{:>10} {:>7}   {}'
         output = []
         for pid, pidObj in self.watch.pids.items():
            count = sum( len( fd.files ) for fd in pidObj.fds.values() )
            totalWatches += count
            output.append( ( count, pid, pidObj.cmdline ) )
         output.sort()

         print( formatStr.format( *title ) + '\n' )
         for outputLine in output:
            print( formatStr.format( *outputLine ) )
         print( f'-----\nTotal watches: {totalWatches}' )

   def render( self ):
      queries = ALL_QUERIES if self._query == 'all' else [ self._query ]

      for query in queries:
         # Only print the section header for the 'all' command
         if self._query == 'all':
            print( f'\n----- show debugging inotify {query} -----\n' )
            sys.stdout.flush()
         self.renderHelper( query )
