# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
from CliMode.IpLocking import (
   IpLockingVlanAddressLockingMode,
   IpLockingEthIntfAddressLockingMode,
   IpLockingLagIntfAddressLockingMode,
   AddressLockingMode,
)
from CliPlugin import IntfCli
from CliPlugin import VlanCli
from CliPlugin.IpLockingCliConstants import vlanTagValidationWarning
import ConfigMount
import LazyMount
import SharedMem
import Smash
import Tac

# globals
cliConfig = None
cliStatus = None
hwConfig = None
enabledInterfaceStatus = None
enabledVlanStatus = None
leaseStatus = None
installedConfig = None
arpInspectionStatus = None
ipsgConfig = None
dhcpRelayStatus = None
dhcpSnoopingStatus = None
dhcpSnooping6Status = None
dhcpServerVrfStatus = None
bridgingInputCliConfig = None
intfVlanListMap = None

def checkOrCreateClearLeaseInfo():
   cliClearLeaseInfo = cliConfig.cliClearLeaseInfo
   if not cliClearLeaseInfo:
      cliConfig.cliClearLeaseInfo = ( "LeaseClearEntity", )

def updateQueryTimeoutInfo( retryInterval, timeout ):
   cliConfig.queryTimeoutInfo = Tac.Value( 'IpLocking::QueryTimeoutInfo',
         retryInterval, timeout )

def updateConfiguredAndActive( config ):
   config.configured = bool( config.disabled != config.disabledDefault or
                             config.localInterface != config.localInterfaceDefault or
                             config.dhcpV4Server or
                             config.dhcpV6Server or
                             config.staticLease or
                             config.macAgingCleanUp !=
                                config.macAgingCleanUpDefault or
                             config.ipv4LockedAddressEnforcementDisabled or
                             config.ipv6LockedAddressEnforcementDisabled or
                             config.queryTimeoutInfo.timeout )

   # IPv4 Locking can be active in locked address enforcement disabled mode
   # based on just the interface or level vlan level configuration.
   allConfigs = list( config.interface.values() )
   ipv4FullyDisabledConfExists = (
         any( physIntfOrVlan.ipv4Disabled for
              physIntfOrVlan in allConfigs ) )

   ipv6FullyDisabledConfExists = (
         any( physIntfOrVlan.ipv6Disabled for
              physIntfOrVlan in allConfigs ) )

   allConfigs.extend( config.vlan.values() )
   ipv4EnforcementEnabledConfExists = (
         any( physIntfOrVlan.ipv4 and
              not physIntfOrVlan.ipv4LockedAddressEnforcementDisabled for
              physIntfOrVlan in allConfigs ) )

   ipv4EnforcementDisabledConfExists = (
         any( physIntfOrVlan.ipv4 and
              physIntfOrVlan.ipv4LockedAddressEnforcementDisabled for
              physIntfOrVlan in allConfigs ) )

   # Locked-address enforcement can only be disabled globally for IPv6,
   # so for uniformity with IPv4, calling the collection of interfaces which are
   # configured for IPv6 Locking as `ipv6ConfExists`
   ipv6ConfExists = (
         any( physIntfOrVlan.ipv6 for physIntfOrVlan in allConfigs ) )

   config.active = bool( config.disabled == config.disabledDefault and
                         ( ( config.localInterface !=
                                config.localInterfaceDefault and
                             ( ( config.dhcpV4Server and
                                 ipv4EnforcementEnabledConfExists ) or
                               ( config.dhcpV6Server and
                                 ipv6ConfExists ) ) ) or
                           ipv4FullyDisabledConfExists or
                           ipv6FullyDisabledConfExists or
                           ( config.ipv4LockedAddressEnforcementDisabled and
                             ipv4EnforcementEnabledConfExists ) or
                           ( config.ipv6LockedAddressEnforcementDisabled and
                             ipv6ConfExists ) or
                           ipv4EnforcementDisabledConfExists ) )

   config.ipv4EnforcedDisabledActive = bool(
         not config.disabled and
         ( ipv4EnforcementDisabledConfExists or
           ( config.ipv4LockedAddressEnforcementDisabled and
             ipv4EnforcementEnabledConfExists ) ) )

   config.ipv4EnforcedEnabledActive = bool(
         not config.disabled and
         not config.ipv4LockedAddressEnforcementDisabled and
         config.localInterface != config.localInterfaceDefault and
         config.dhcpV4Server and ipv4EnforcementEnabledConfExists )

   config.ipv6EnforcedDisabledActive = bool(
         not config.disabled and
         config.ipv6LockedAddressEnforcementDisabled and
         ipv6ConfExists )

   # note: ipv6 enforcement enabled mode is not implemented
   config.ipv6EnforcedEnabledActive = bool(
         not config.disabled and
         not config.ipv6LockedAddressEnforcementDisabled and
         config.localInterface != config.localInterfaceDefault and
         config.dhcpV6Server and ipv6ConfExists )

def doAddressLockingModeEnter( mode, args ):
   childMode = mode.childMode( AddressLockingMode )
   mode.session_.gotoChildMode( childMode )
   firstTime = not mode.session.sessionData( 'IpLocking.vlanTagWarn', False )
   if firstTime and not bridgingInputCliConfig.vlanTagValidationEnabled:
      mode.addWarning( vlanTagValidationWarning )
      mode.session.sessionDataIs( 'IpLocking.vlanTagWarn', True )

def noAddressLockingModeEnter( mode, args ):
   cliConfig.disabled = cliConfig.disabledDefault
   cliConfig.localInterface = cliConfig.localInterfaceDefault
   cliConfig.macAgingCleanUp = cliConfig.macAgingCleanUpDefault
   cliConfig.queryTimeoutInfo = cliConfig.queryTimeoutInfoDefault
   cliConfig.ipv4LockedAddressEnforcementDisabled = False
   cliConfig.ipv6LockedAddressEnforcementDisabled = False
   cliConfig.dhcpV4Server.clear()
   cliConfig.dhcpV6Server.clear()
   cliConfig.staticLease.clear()
   cliConfig.useOldSyntax.clear()
   updateConfiguredAndActive( cliConfig )
   mode.session.sessionDataIs( 'IpLocking.vlanTagWarn', False )

def setIpLockingDisabled( mode, args ):
   cliConfig.disabled = True
   updateConfiguredAndActive( cliConfig )

def noIpLockingDisabled( mode, args ):
   cliConfig.disabled = cliConfig.disabledDefault
   updateConfiguredAndActive( cliConfig )

def setIpLockingLocalInterface( mode, args ):
   intf = args.get( 'INTF' )
   cliConfig.localInterface = intf.name
   updateConfiguredAndActive( cliConfig )

def noIpLockingLocalInterface( mode, args ):
   cliConfig.localInterface = cliConfig.localInterfaceDefault
   updateConfiguredAndActive( cliConfig )

def setIpLockingDhcpServer( mode, args ):
   v4addr = args[ 'V4ADDR' ]
   cliConfig.dhcpV4Server[ v4addr ] = 'leaseQuery'
   updateConfiguredAndActive( cliConfig )

def noIpLockingDhcpServer( mode, args ):
   if 'V4ADDR' in args:
      v4addr = args[ 'V4ADDR' ]
      del cliConfig.dhcpV4Server[ v4addr ]
   else:
      cliConfig.dhcpV4Server.clear()
   updateConfiguredAndActive( cliConfig )

def setIpLockingServer( mode, args ):
   v4addr = args[ 'V4ADDR' ]
   protocol = 'arIpQuery' if 'arista-query' in args else 'leaseQuery'
   cliConfig.dhcpV4Server[ v4addr ] = protocol
   updateConfiguredAndActive( cliConfig )

def noIpLockingServer( mode, args ):
   v4addr = args.get( 'V4ADDR' )
   if v4addr:
      del cliConfig.dhcpV4Server[ v4addr ]
   else:
      if 'arista-query' in args:
         for server in cliConfig.dhcpV4Server:
            if cliConfig.dhcpV4Server[ server ] == 'arIpQuery':
               del cliConfig.dhcpV4Server[ server ]
      else:
         cliConfig.dhcpV4Server.clear()
   updateConfiguredAndActive( cliConfig )

def setIpLockingV6Server( mode, args ):
   v6addr = args[ 'V6ADDR' ]
   cliConfig.dhcpV6Server[ v6addr ] = True
   updateConfiguredAndActive( cliConfig )

def noIpLockingV6Server( mode, args ):
   v6addr = args.get( 'V6ADDR' )
   if v6addr:
      del cliConfig.dhcpV6Server[ v6addr ]
   else:
      cliConfig.dhcpV6Server.clear()
   updateConfiguredAndActive( cliConfig )

def setIpLockingStaticLease( mode, args ):
   ip = args.get( 'V4ADDR' ) or args.get( 'V6ADDR' )
   ipGenAddr = Tac.Value( 'Arnet::IpGenAddr', str( ip ) )
   macAddr = Tac.Value( 'Arnet::EthAddr', stringValue=args[ 'MAC_ADDR' ] )
   if ( ipGenAddr in cliConfig.staticLease and
        cliConfig.staticLease[ ipGenAddr ].mac != args[ 'MAC_ADDR' ] ):
      del cliConfig.staticLease[ ipGenAddr ]
      cliConfig.staticLease.newMember( ipGenAddr, macAddr, '' )
   else:
      cliConfig.staticLease.newMember( ipGenAddr, macAddr, '' )
   updateConfiguredAndActive( cliConfig )

def noIpLockingStaticLease( mode, args ):
   ip = args.get( 'V4ADDR' ) or args.get( 'V6ADDR' )
   ipGenAddr = Tac.Value( 'Arnet::IpGenAddr', str( ip ) )
   if ipGenAddr in cliConfig.staticLease:
      del cliConfig.staticLease[ ipGenAddr ]
   updateConfiguredAndActive( cliConfig )

def vlanConfigured( vlanConfig ):
   return ( vlanConfig and
            ( vlanConfig.ipv4 or vlanConfig.ipv6 or
              vlanConfig.ipv4LockedAddressEnforcementDisabled ) )

def updateVlanMode( config, vlan ):
   if not vlanConfigured( config ):
      del cliConfig.vlan[ vlan ]
   updateConfiguredAndActive( cliConfig )

def handlerHelperIntfVlanLockedAddressEnforcementDisabled( mode, vlan ):
   config = cliConfig.newVlan( vlan )
   config.ipv4LockedAddressEnforcementDisabled = True
   updateVlanMode( config, vlan )

def noOrDefaultIntfVlanLockedAddressEnforcementDisabled( mode, vlan ):
   config = cliConfig.vlan.get( vlan )
   if not config:
      return
   config.ipv4LockedAddressEnforcementDisabled = False
   updateVlanMode( config, vlan )

def updateIntfMode( mode, config ):
   if not intfConfigured( config ):
      del cliConfig.interface[ mode.intf ]
   clearUseOldSyntax( mode.intf )
   updateConfiguredAndActive( cliConfig )

def setIpLockingIntfVlanLockedAddressEnforcementDisabled( mode, args ):
   if isinstance( mode, IpLockingVlanAddressLockingMode ):
      if mode.multiInstance:
         for m in mode.vlan.individualVlans_:
            handlerHelperIntfVlanLockedAddressEnforcementDisabled(
                  mode, m.vlan.id_ )
      else:
         handlerHelperIntfVlanLockedAddressEnforcementDisabled(
               mode, mode.vlan.id )
   else:
      config = mode.newConfig()
      config.ipv4LockedAddressEnforcementDisabled = True
      updateIntfMode( mode, config )

def noIpLockingIntfVlanLockedAddressEnforcementDisabled( mode, args ):
   if isinstance( mode, IpLockingVlanAddressLockingMode ):
      if mode.multiInstance:
         for m in mode.vlan.individualVlans_:
            noOrDefaultIntfVlanLockedAddressEnforcementDisabled( mode, m.vlan.id_ )
      else:
         noOrDefaultIntfVlanLockedAddressEnforcementDisabled( mode, mode.vlan.id )
   else:
      config = mode.getConfig()
      if not config:
         return
      config.ipv4LockedAddressEnforcementDisabled = False
      updateIntfMode( mode, config )

def setIpLockingGlobalLockedAddressEnforcementDisabled( mode, args ):
   attr = args.get( 'ipv4', 'ipv6' ) + 'LockedAddressEnforcementDisabled'
   setattr( cliConfig, attr, True )
   updateConfiguredAndActive( cliConfig )

def noIpLockingGlobalLockedAddressEnforcementDisabled( mode, args ):
   attr = args.get( 'ipv4', 'ipv6' ) + 'LockedAddressEnforcementDisabled'
   setattr( cliConfig, attr, False )
   updateConfiguredAndActive( cliConfig )

def setIpLockingLockedAddressExpirationMacDisabled( mode, args ):
   cliConfig.macAgingCleanUp = False
   updateConfiguredAndActive( cliConfig )

def noIpLockingLockedAddressExpirationMacDisabled( mode, args ):
   cliConfig.macAgingCleanUp = cliConfig.macAgingCleanUpDefault
   updateConfiguredAndActive( cliConfig )

def addressFamilyHandler( args, key, func ):
   afs = args.get( 'ADDRESS_FAMILY' )
   config = func( key )
   config.ipv4 = 'ipv4' in afs
   config.ipv6 = 'ipv6' in afs
   updateConfiguredAndActive( cliConfig )

def intfConfigured( intfConfig ):
   return ( intfConfig and
            ( intfConfig.ipv4 or intfConfig.ipv6 or
              intfConfig.ipv4Disabled or intfConfig.ipv6Disabled or
              intfConfig.ipv4LockedAddressEnforcementDisabled or
              intfConfig.ipv4DeniedAddr ) )

def addressFamilyNoOrDefaultHandler( args, key, configs ):
   if key not in configs:
      return
   afs = args.get( 'ADDRESS_FAMILY', [ 'ipv4', 'ipv6' ] )
   config = configs[ key ]
   if 'ipv4' in afs:
      config.ipv4 = False
   if 'ipv6' in afs:
      config.ipv6 = False
   if not intfConfigured( config ):
      del configs[ key ]
   updateConfiguredAndActive( cliConfig )

def clearUseOldSyntax( intf ):
   del cliConfig.useOldSyntax[ intf ]

def addUseOldSyntax( intf ):
   cliConfig.useOldSyntax[ intf ] = True

def doEnterLagInterfaceAddressLockingMode( mode, args ):
   intf = mode.intf.name
   childMode = mode.childMode( IpLockingLagIntfAddressLockingMode,
                               param=intf,
                               cliConfig=cliConfig )
   mode.session_.gotoChildMode( childMode )

def noEnterLagInterfaceAddressLockingMode( mode, args ):
   intf = mode.intf.name
   del cliConfig.interface[ intf ]

# ---------------------------------------------------------
# Address locking sub-mode within the "config-if-Po*" mode.
# ---------------------------------------------------------
def setIpLockingLagInterfaceAddressFamilyDisabled( mode, args ):
   config = mode.newConfig()
   config.ipv4Disabled = True
   updateIntfMode( mode, config )

def noIpLockingLagInterfaceAddressFamilyDisabled( mode, args ):
   config = mode.getConfig()
   if not config:
      return
   config.ipv4Disabled = False
   updateIntfMode( mode, config )

# ---------------------------------------------------------
# Address locking sub-mode within the "config-if-Et*" mode.
# ---------------------------------------------------------
def setIpLockingInterfaceAddressFamily( mode, args ):
   config = mode.newConfig()
   attr = args.get( 'ipv4', 'ipv6' )
   setattr( config, attr, 'disabled' not in args )
   setattr( config, attr + 'Disabled', 'disabled' in args )
   updateIntfMode( mode, config )

def noIpLockingInterfaceAddressFamily( mode, args ):
   config = mode.getConfig()
   if not config:
      return
   attrs = args.get( 'ipv4', args.get( 'ipv6', [ 'ipv4', 'ipv6' ] ) )
   if not isinstance( attrs, list ):
      attrs = [ attrs ]
   for attr in attrs:
      attr += 'Disabled' if 'disabled' in args else ''
      setattr( config, attr, False )
   updateIntfMode( mode, config )

def setIpLockingInterfaceEnable( mode, args ):
   intf = mode.intf.name
   if not args.get( 'ADDRESS_FAMILY' ):
      childMode = mode.childMode( IpLockingEthIntfAddressLockingMode,
                                  param=intf,
                                  cliConfig=cliConfig )
      # Do not enter IP Locking interface address locking submode when
      # configured via interface range config mode
      if mode.session.mode_ != childMode.parent_:
         mode.addWarning( 'Command cannot run in interface range config mode' )
         return
      mode.session_.gotoChildMode( childMode )
   else:
      intfConfig = cliConfig.interface.get( intf )
      if not intfConfigured( intfConfig ):
         addUseOldSyntax( intf )
      addressFamilyHandler( args, intf, cliConfig.newInterface )

def noIpLockingInterfaceEnable( mode, args ):
   intf = mode.intf.name
   clearUseOldSyntax( intf )
   if not args.get( 'ADDRESS_FAMILY' ):
      del cliConfig.interface[ intf ]
   else:
      addressFamilyNoOrDefaultHandler( args, intf, cliConfig.interface )

def setIpLockingIntfDenyAddress( mode, args ):
   config = mode.newConfig()
   deniedAddr = args.get( 'IP_ADDR' )
   config.ipv4DeniedAddr[ IpGenAddr( deniedAddr ) ] = True
   updateIntfMode( mode, config )

def noIpLockingIntfDenyAddress( mode, args ):
   config = mode.getConfig()
   if not config:
      return
   deniedAddr = args.get( 'IP_ADDR' )
   del config.ipv4DeniedAddr[ IpGenAddr( deniedAddr ) ]
   updateIntfMode( mode, config )

def doEnterVlanAddressLockingMode( mode, args ):
   childMode = mode.childMode( IpLockingVlanAddressLockingMode )
   mode.session_.gotoChildMode( childMode )

def noEnterVlanAddressLockingMode( mode, args ):
   if mode.multiInstance:
      for individualVlanMode in mode.vlan.individualVlans_:
         del cliConfig.vlan[ individualVlanMode.vlan.id ]
   else:
      del cliConfig.vlan[ mode.vlan.id ]

def handlerHelperIpLockingVlanAddressFamily( mode, args, vlan ):
   config = cliConfig.newVlan( vlan )
   attr = args.get( 'ipv4', 'ipv6' )
   setattr( config, attr, True )
   updateVlanMode( config, vlan )

def noOrDefaultHandlerHelperIpLockingVlanAddressFamily( mode, args, vlan ):
   config = cliConfig.vlan.get( vlan )
   if not config:
      return
   attrs = args.get( 'ipv4', args.get( 'ipv6', [ 'ipv4', 'ipv6' ] ) )
   if not isinstance( attrs, list ):
      attrs = [ attrs ]
   for attr in attrs:
      setattr( config, attr, False )
   updateVlanMode( config, vlan )

def setIpLockingVlanAddressFamily( mode, args ):
   if mode.multiInstance:
      for m in mode.vlan.individualVlans_:
         handlerHelperIpLockingVlanAddressFamily( mode, args, m.vlan.id_ )
   else:
      handlerHelperIpLockingVlanAddressFamily( mode, args, mode.vlan.id )

def noIpLockingVlanAddressFamily( mode, args ):
   if mode.multiInstance:
      for m in mode.vlan.individualVlans_:
         noOrDefaultHandlerHelperIpLockingVlanAddressFamily(
               mode, args, m.vlan.id )
   else:
      noOrDefaultHandlerHelperIpLockingVlanAddressFamily(
            mode, args, mode.vlan.id )

def setIpLockingClearLease( mode, args ):
   checkOrCreateClearLeaseInfo()
   # Reset the members of cliClearLeaseInfo everytime it is configured.
   cliConfig.cliClearLeaseInfo.ip = cliConfig.cliClearLeaseInfo.ipDefault
   cliConfig.cliClearLeaseInfo.intf = cliConfig.cliClearLeaseInfo.intfDefault
   cliConfig.cliClearLeaseInfo.clearAll = (
         cliConfig.cliClearLeaseInfo.clearAllDefault )
   if 'V4ADDR' in args or 'V6ADDR' in args:
      addr = args.get( 'V4ADDR' ) or args.get( 'V6ADDR' )
      ipGenAddr = Tac.Value( "Arnet::IpGenAddr", str( addr ) )
      cliConfig.cliClearLeaseInfo.ip = ipGenAddr
   elif 'INTF' in args:
      intf = args[ 'INTF' ]
      cliConfig.cliClearLeaseInfo.intf = intf.name
   elif 'all' in args:
      cliConfig.cliClearLeaseInfo.clearAll = True
   cliConfig.cliClearLeaseInfo.versionNumber = \
      ( cliConfig.cliClearLeaseInfo.versionNumber + 1 ) % 256

def setIpLockingClearCounters( mode, args ):
   cliConfig.resetCountersTrigger = ( cliConfig.resetCountersTrigger + 1 ) % 256

def setIpLockingLeaseQueryTimeout( mode, args ):
   retryInterval = args[ 'RETRY_INTERVAL' ]
   timeout = args[ 'TIMEOUT' ]
   if timeout < retryInterval:
      mode.addError(
            'Lease query retry interval is greater than timeout' )
      return
   updateQueryTimeoutInfo( retryInterval, timeout )
   updateConfiguredAndActive( cliConfig )

def noIpLockingLeaseQueryTimeout( mode, args ):
   updateQueryTimeoutInfo( 0, 0 )
   updateConfiguredAndActive( cliConfig )

# Remove interface config when running "[no|default] interface <interface>"
class IpLockingIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del cliConfig.interface[ self.intf_.name ]

# Remove vlan config when running "[no|default] vlan <VLANID>"
class IpLockingVlan( VlanCli.VlanDependentBase ):
   def destroy( self ):
      del cliConfig.vlan[ self.vlan_.id_ ]

#-------------------------------------------------------------------------------
# Have the Cli agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig
   global cliStatus
   global hwConfig
   global enabledInterfaceStatus
   global enabledVlanStatus
   global leaseStatus
   global installedConfig
   global arpInspectionStatus
   global ipsgConfig
   global dhcpRelayStatus
   global dhcpSnoopingStatus
   global dhcpSnooping6Status
   global dhcpServerVrfStatus
   global bridgingInputCliConfig
   global intfVlanListMap

   cliConfig = ConfigMount.mount( entityManager, "iplocking/cliConfig",
                                  "IpLocking::CliConfig", "w" )
   cliStatus = LazyMount.mount( entityManager, "iplocking/cliStatus",
                                "IpLocking::CliStatus", "r" )
   hwConfig = LazyMount.mount( entityManager, "iplocking/hardware/config",
                               "IpLocking::Hardware::Config", "r" )
   enabledInterfaceStatus = LazyMount.mount( entityManager,
         "iplocking/interfaceStatus", "IpLocking::EnabledInterfaceStatus", "r" )
   enabledVlanStatus = LazyMount.mount( entityManager,
         "iplocking/vlanStatus", "IpLocking::EnabledVlanStatus", "r" )
   leaseStatus = LazyMount.mount( entityManager, "iplocking/leaseStatus",
                                  "IpLocking::LeaseStatus", "r" )
   arpInspectionStatus = LazyMount.mount( entityManager,
                                          "security/arpInspection/status",
                                          "ArpInsp::Status", "r" )
   ipsgConfig = LazyMount.mount( entityManager, "security/ipsg/config",
                                 "Ipsg::Config", "r" )
   dhcpRelayStatus = LazyMount.mount( entityManager, "ip/helper/dhcprelay/status",
                                      "Ip::Helper::DhcpRelay::Status", "r" )
   dhcpSnoopingStatus = LazyMount.mount( entityManager,
                                         "bridging/dhcpsnooping/status",
                                         "Bridging::DhcpSnooping::Status", "r" )
   dhcpSnooping6Status = LazyMount.mount( entityManager,
                                          "bridging/dhcpsnooping/dhcp6Status",
                                          "Bridging::DhcpSnooping::Status", "r" )
   dhcpServerVrfStatus = LazyMount.mount( entityManager, "dhcpServer/vrf/status",
                                       "DhcpServer::VrfStatusDir", "r" )
   bridgingInputCliConfig = LazyMount.mount( entityManager,
                                             "bridging/input/config/cli",
                                             "Bridging::Input::CliConfig", "r" )
   intfVlanListMap = LazyMount.mount( entityManager,
                                      "iplocking/intfVlanListMap",
                                      "IpLocking::IntfVlanListMap", "r" )

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   installedConfig = shmemEm.doMount( "iplocking/hardware/installedConfig",
                                      "IpLocking::Hardware::InstalledConfig",
                                      Smash.mountInfo( 'keyshadow' ) )
   IntfCli.Intf.registerDependentClass( IpLockingIntf )
   VlanCli.Vlan.registerDependentClass( IpLockingVlan )
