# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Dict, Int
from IntfModels import Interface
from ArnetModel import IpGenericAddress
from TableOutput import createTable, Format

class ServerCounters( Model ):
   leaseQuery = Int( help="Number of lease query packets" )
   leaseActiveReceived = Int(
         help="Number of lease active packets that were received" )
   leaseActiveDropped = Int(
         help="Number of lease active packets that were dropped" )
   leaseUnknownReceived = Int(
         help="Number of lease unknown packets that were received" )
   leaseUnknownDropped = Int(
         help="Number of lease unknown packets that were dropped" )
   leaseUnassignedReceived = Int(
         help="Number of lease unassigned packets that were received" )
   leaseUnassignedDropped = Int(
         help="Number of lease unassigned packets that were dropped" )
   unknown = Int(
         help="Number of packets dropped due to unrecognized packet type" )

class IntfCounters( Model ):
   leaseQuery = Int( help="Number of lease query packets" )
   leaseActive = Int( help="Number of lease active packets" )
   leaseUnknown = Int( help="Number of lease unknown packets" )
   leaseUnassigned = Int( help="Number of lease unassigned packets" )

class IpLockingCountersModel( Model ):
   intfCounters = Dict( keyType=Interface,
         valueType=IntfCounters,
         help="A mapping of interfaces to their counters" )
   serverCounters = Dict( keyType=IpGenericAddress,
         valueType=ServerCounters,
         help="A mapping of server addresses to their counters" )

   def render( self ):
      f1 = Format( justify='right' )
      f1.padLimitIs( True )

      serverHeader = [ 'DHCP Server', 'Query',
            ( 'Lease Active', 'ch', ( 'Rcvd', 'Drop' ) ),
            ( 'Lease Unknown', 'ch', ( 'Rcvd', 'Drop' ) ),
            ( 'Lease Unassigned', 'ch', ( 'Rcvd', 'Drop' ) ),
            'Unknown' ]
      serverTable = createTable( serverHeader )
      for server in self.serverCounters:
         serverTable.newRow( server,
               self.serverCounters[ server ].leaseQuery,
               self.serverCounters[ server ].leaseActiveReceived,
               self.serverCounters[ server ].leaseActiveDropped,
               self.serverCounters[ server ].leaseUnknownReceived,
               self.serverCounters[ server ].leaseUnknownDropped,
               self.serverCounters[ server ].leaseUnassignedReceived,
               self.serverCounters[ server ].leaseUnassignedDropped,
               self.serverCounters[ server ].unknown )

      serverTable.formatColumns( f1, f1, f1, f1, f1, f1, f1, f1, f1 )
      print( '\n' + serverTable.output() )

      intfHeader = [ 'Interface', 'Query', 'Lease Active', 'Lease Unknown',
            'Lease Unassigned' ]
      intfTable = createTable( intfHeader )
      for intf in self.intfCounters:
         intfTable.newRow( intf,
               self.intfCounters[ intf ].leaseQuery,
               self.intfCounters[ intf ].leaseActive,
               self.intfCounters[ intf ].leaseUnknown,
               self.intfCounters[ intf ].leaseUnassigned )
      intfTable.formatColumns( f1, f1, f1, f1, f1 )
      print( "\n" + intfTable.output() )
