#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import LazyMount
import ConfigMount
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.IraNexthopGroupCli import nexthopGroupSupportedGuard
from CliPlugin.IraDlbCli import routingHwDlbStatus
from CliPlugin.IraFlowsetTableReadCallback import iraFlowsetTableReadCallback

routingHwDlbConfig = None
routingNhgConfig = None

IraDlbModel = CliDynamicPlugin( "IraDlbModel" )

# -------------------------------------------------------------------------------
# "[no/default] ip hardware fib load-balance distribution [ hash|dynamic]"
#  command
# -------------------------------------------------------------------------------

def handlerFibLoadBalanceDistributionCmd( mode, args ):
   enabled = 'dynamic' in args
   routingHwDlbConfig.globalDlbEcmpEnable = enabled

# -------------------------------------------------------------------------------
# "show ip hardware fib load-balance distribution" command
# -------------------------------------------------------------------------------

def handlerShowFibLoadBalanceDistributionCmd( mode, args ):
   GlobalDlbModel = IraDlbModel.GlobalDlb()
   GlobalDlbModel.dlbEcmpNhg = []
   GlobalDlbModel.globalDlbEcmpEnabled = routingHwDlbConfig.globalDlbEcmpEnable
   GlobalDlbModel.dlbPortAssignmentMode = routingHwDlbStatus.portAssignmentMode
   GlobalDlbModel.dlbInactivityDuration = routingHwDlbStatus.inactivityDuration
   GlobalDlbModel.dlbSamplingPeriod = routingHwDlbStatus.samplingPeriod
   GlobalDlbModel.dlbPortLoadingWeight = routingHwDlbStatus.portLoadingWeight
   GlobalDlbModel.randomSelectionControlSeed =\
                  routingHwDlbStatus.randomSelectionControlSeed
   GlobalDlbModel.dlbFlowSetSize = routingHwDlbStatus.flowSetSize
   GlobalDlbModel.lbnBasedDlbHashing = False
   GlobalDlbModel.lbnBasedStaticEcmpHashing = False
   GlobalDlbModel.dlbAccessGroupIpv4 = routingHwDlbConfig.selectiveDlbIpv4Acl
   GlobalDlbModel.dlbAccessGroupIpv6 = routingHwDlbConfig.selectiveDlbIpv6Acl
   if nexthopGroupSupportedGuard( mode, None ) is None:
      nhgs = sorted( routingNhgConfig.nexthopGroup.items() )
      for nhgName, nhgInfo in nhgs:
         if nhgInfo.dlbEcmpEnable:
            GlobalDlbModel.dlbEcmpNhg.append( nhgName )
   return GlobalDlbModel

# ---------------------------------------------------------------------------
# "show ip hardware fib load-balance distribution dynamic group flows" command
# The CLI displays DLB ID and port assignment of macro-flows
# ---------------------------------------------------------------------------
def handlerShowFibLoadBalanceDistributionMacroFlowsCmd( mode, args ):
   dlbId = args.get( 'RANGE' )
   macroFlowsModel = IraDlbModel.MacroFlows()
   if dlbId is None:
      macroFlowsModel.dlbId = -1
   else:
      macroFlowsModel.dlbId = dlbId

   if routingHwDlbConfig.globalDlbEcmpEnable is False:
      macroFlowsModel.enabled = False
      return macroFlowsModel
   try:
      macroFlowsModel.flowList = \
      iraFlowsetTableReadCallback.displayDlbFlowsetTableEntries( dlbId )

   except ( Tac.Timeout, NotImplementedError ) as e:
      macroFlowsModel.exception = True
      if isinstance( e, Tac.Timeout ):
         # Tac.waitFor prints the exceptionMsg, no need to print another message
         macroFlowsModel.exceptionMsg = ""
      else:
         macroFlowsModel.exceptionMsg = str( e )
   return macroFlowsModel

# ---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic member-selection
#             optimal [ timer | always ] " in exec mode.
#   operational-mode is port_assignment_mode in broadcom specs
# Encoding of DLB operational modes are as follows :
#  0 - Eligibility [ default mode | <cmd> optimal timer ]
#  2 - Packet Spray [ <cmd> optimal always ]
# ---------------------------------------------------------------------------

def handlerFibLoadBalanceDistDynMemberSelCmd( mode, args ):
   if not routingHwDlbConfig.globalDlbEcmpEnable:
      mode.addWarning( "Dynamic load balance on ECMP is not enabled" )

   portAssignmentMode = 0
   memberSelMode = args[ 'MODE' ]
   if memberSelMode == "timer":
      portAssignmentMode = 0
   else :
      assert memberSelMode == "always"
      portAssignmentMode = 2
   routingHwDlbConfig.portAssignmentMode = portAssignmentMode

def noOrDefaultHandlerFibLoadBalanceDistDynMemberSelCmd( mode, args ):
   # to update param to default
   # Assign dlbConfig.<PARAM> to TAC model default values so in reactor,
   # it can assign default values to DlbStatus from chipAttr
   dlbCfg = Tac.newInstance( "Routing::Hardware::DlbConfig", "DlbConfig" )
   routingHwDlbConfig.portAssignmentMode = \
      dlbCfg.portAssignmentMode

# ---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic inactivity-threshold
#                 VALUE mircroseconds" in exec mode.
#   inactivity-threshold is inactivity-duration in broadcom specs
# ---------------------------------------------------------------------------

def handlerFibLoadBalanceDistDynInactDurCmd( mode, args ):
   if not routingHwDlbConfig.globalDlbEcmpEnable:
      mode.addWarning( "Dynamic load balance on ECMP is not enabled" )

   routingHwDlbConfig.inactivityDuration = args[ 'VALUE' ]

def noOrDefaultHandlerFibLoadBalanceDistDynInactDurCmd( mode, args ):
   # to update param to default
   # Assign dlbConfig.<PARAM> to TAC model default values so in reactor,
   # it can assign default values to DlbStatus from chipAttr
   dlbCfg = Tac.newInstance( "Routing::Hardware::DlbConfig", "DlbConfig" )
   routingHwDlbConfig.inactivityDuration =\
      dlbCfg.inactivityDuration

# ---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic sampling-period
#                 VALUE microseconds" in exec mode.
# ---------------------------------------------------------------------------

def handlerFibLoadBalanceDistDynSamplingPeriodCmd( mode, args ):
   if not routingHwDlbConfig.globalDlbEcmpEnable:
      mode.addWarning( "Dynamic load balance on ECMP is not enabled" )

   routingHwDlbConfig.samplingPeriod = args[ 'VALUE' ]

def noOrDefaultHandlerFibLoadBalanceDistDynSamplingPeriodCmd( mode, args ):
   # to update param to default
   # Assign dlbConfig.<PARAM> to TAC model default values so in reactor,
   # it can assign default values to DlbStatus from chipAttr
   dlbCfg = Tac.newInstance( "Routing::Hardware::DlbConfig", "DlbConfig" )
   routingHwDlbConfig.samplingPeriod = dlbCfg.samplingPeriod

# ---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic seed VALUE" in exec mode.
# ---------------------------------------------------------------------------

def handlerFibLoadBalanceDistDynSeedCmd( mode, args ):
   if not routingHwDlbConfig.globalDlbEcmpEnable:
      mode.addWarning( "Dynamic load balance on ECMP is not enabled" )

   routingHwDlbConfig.randomSelectionControlSeed =\
      args[ 'VALUE' ]

def noOrDefaultHandlerFibLoadBalanceDistDynSeedCmd( mode, args ):
   # to update param to default
   # Assign dlbConfig.<PARAM> to TAC model default values so in reactor,
   # it can assign default values to DlbStatus from chipAttr
   dlbCfg = Tac.newInstance( "Routing::Hardware::DlbConfig", "DlbConfig" )
   routingHwDlbConfig.randomSelectionControlSeed =\
         dlbCfg.randomSelectionControlSeed

# ---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic average-traffic-weight
#                 VALUE" in exec mode.
#    average-traffic-weight is port-loading-weight in broadcom term
# ---------------------------------------------------------------------------

def handlerFibLoadBalanceDistDynPortLoadingWtCmd( mode, args ):
   if not routingHwDlbConfig.globalDlbEcmpEnable:
      mode.addWarning( "Dynamic load balance on ECMP is not enabled" )

   routingHwDlbConfig.portLoadingWeight = args[ 'VALUE' ]

def noOrDefaultHandlerFibLoadBalanceDistDynPortLoadingWtCmd( mode, args ):
   # to update param to default
   # Assign dlbConfig.<PARAM> to TAC model default values so in reactor,
   # it can assign default values to DlbStatus from chipAttr
   dlbCfg = Tac.newInstance( "Routing::Hardware::DlbConfig", "DlbConfig" )
   routingHwDlbConfig.portLoadingWeight =\
      dlbCfg.portLoadingWeight

# ---------------------------------------------------------------------------
# " ip hardware fib load-balance distribution dynamic flow-set-size VALUE"
#                              in exec mode.
# ---------------------------------------------------------------------------

def handlerFibLoadBalanceDistDynFlowSetSizeCmd( mode, args ):
   if not routingHwDlbConfig.globalDlbEcmpEnable:
      mode.addWarning( "Dynamic load balance on ECMP is not enabled" )

   routingHwDlbConfig.flowSetSize = int( args[ 'VALUE' ] )

def noOrDefaultHandlerFibLoadBalanceDistDynFlowSetSizeCmd( mode, args ):
   # to update param to default
   # Assign dlbConfig.<PARAM> to TAC model default values so in reactor,
   # it can assign default values to DlbStatus from chipAttr
   dlbCfg = Tac.newInstance( "Routing::Hardware::DlbConfig", "DlbConfig" )
   routingHwDlbConfig.flowSetSize = dlbCfg.flowSetSize

# -------------------------------------------------------------------------------
# [no/default] ip hardware fib load-balance distribution dynamic
#              (ipv4|ipv6) access-group <acl>
# -------------------------------------------------------------------------------

def handlerFibLoadBalanceAclCmd( mode, args ):
   if not routingHwDlbConfig.globalDlbEcmpEnable:
      mode.addWarning( "Dynamic load balance on ECMP is not enabled" )

   aclName = args[ 'ACLNAME' ]
   if 'ipv4' in args:
      routingHwDlbConfig.selectiveDlbIpv4Acl = aclName
   else:
      routingHwDlbConfig.selectiveDlbIpv6Acl = aclName

def noOrDefaultHandlerFibLoadBalanceAclCmd( mode, args ):
   if 'ipv4' in args:
      routingHwDlbConfig.selectiveDlbIpv4Acl = ""
   else:
      routingHwDlbConfig.selectiveDlbIpv6Acl = ""

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHwDlbConfig
   global routingNhgConfig

   routingHwDlbConfig = ConfigMount.mount( entityManager,
                                           "routing/hardware/dlb/config",
                                           "Routing::Hardware::DlbConfig", "w" )
   routingNhgConfig = LazyMount.mount( entityManager,
                                       "routing/nexthopgroup/input/cli",
                                       "Routing::NexthopGroup::ConfigInput", "r" )
