#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys

import BasicCli
import CliCommand
import CliMatcher
import CliToken
import ConfigMount
import LazyMount
import ShowCommand
import Tracing
from TypeFuture import TacLazyType

from CliMode.Ira import IpHardwareEcmpFecMode

__defaultTraceHandle__ = Tracing.Handle( 'CliDynamicEcmpFecCli' )
t0 = Tracing.trace0

EcmpFecBankCount = TacLazyType( "Routing::Hardware::EcmpFecBankCount" )

routingHardwareConfig = None
routingHardwareStatus = None

class HierNhEcmpFecMode( IpHardwareEcmpFecMode, BasicCli.ConfigModeBase ):
   name = "Hierarchical ECMP FEC Bank configuration mode"

   def __init__( self, parent, session ):
      IpHardwareEcmpFecMode.__init__( self, param='ecmpFecConfig' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      # This is a temporary variable to store the bank values as the user
      # enters this mode
      self.bankConfig = {}
      currentConfig = routingHardwareConfig.ecmpFecReservedBankCount
      banks = [ currentConfig.level1, currentConfig.level2, currentConfig.level3 ]
      for lvl, bank in enumerate( banks ):
         if bank != 0:
            self.bankConfig[ lvl + 1 ] = bank

      self.levelCmd = "   level {} bank minimum count {}\n"
      self.cmdAbort = False

      self.errorCmd = "Configured banks exceed maximum available banks: {}"

   def addBankConfig( self, level, banks ):
      rhs = routingHardwareStatus
      if rhs is None:
         rhs = self.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]

      oldBanks = self.bankConfig.get( level, 0 )
      self.bankConfig[ level ] = banks

      pendingBankTotal = sum( self.bankConfig.values() )

      # If the number of configured banks exceed the available banks
      # throw an error.
      if rhs.flexibleEcmpSupported and pendingBankTotal > rhs.maxEcmpFecBanks:
         # Restore the old config
         self.bankConfig[ level ] = oldBanks
         self.addError( self.errorCmd.format( rhs.maxEcmpFecBanks ) )

   def delBankConfig( self, level ):
      if level in self.bankConfig:
         del self.bankConfig[ level ]

   def abort( self ):
      self.cmdAbort = True
      self.bankConfig = {}
      self.session_.gotoParentMode()

   def maybeCommit( self ):
      newConfig = EcmpFecBankCount( self.bankConfig.get( 1, 0 ),
                                    self.bankConfig.get( 2, 0 ),
                                    self.bankConfig.get( 3, 0 ) )

      currentConfig = routingHardwareConfig.ecmpFecReservedBankCount

      if newConfig.isInvalid() and newConfig == currentConfig:
         # They didn't enter any config, so silenty exit the mode
         return

      if currentConfig != newConfig:
         routingHardwareConfig.ecmpFecReservedBankCount = newConfig
         self.addWarning( "Restarting forwarding agent for "
                          "the changes to take effect" )
      self.bankConfig = {}

   def onExit( self ):
      if not self.cmdAbort:
         self.maybeCommit()
      BasicCli.ConfigModeBase.onExit( self )

   def deleteModeConfig( self, args ):
      # Not using self.clear method because it's faster to clear the config this way
      currentConfig = routingHardwareConfig.ecmpFecReservedBankCount
      if not currentConfig.isInvalid():
         routingHardwareConfig.ecmpFecReservedBankCount = EcmpFecBankCount.invalid
         self.addWarning( "Restarting forwarding agent for "
                          "the changes to take effect" )

   def showPending( self ):
      out = ""
      for lvl in range( 1, 4 ):
         if self.bankConfig.get( lvl, 0 ) != 0:
            out += self.levelCmd.format( lvl, self.bankConfig[ lvl ] )

      if out != "":
         sys.stdout.write( self.enterCmd() + '\n' )
         sys.stdout.write( out )

def gotoNexthopEcmpFecMode( mode, args ):
   t0( "Going into HierNhEcmpFecMode" )
   childMode = mode.childMode( HierNhEcmpFecMode )
   mode.session_.gotoChildMode( childMode )

# -------------------------------------------------------------------------------
# Range functions
# -------------------------------------------------------------------------------
def ecmpFecBankLevelRangeFn( mode, context=None ):
   return ( 1, 3 )

def ecmpFecBankCountRangeFn( mode, context=None ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]

   minBanks = 1
   maxBanks = rhs.maxEcmpFecBanks

   # If the feature is not supported, or during switch bootup,
   # we need to allow config even if maxEcmpFecBanks is not programmed
   if maxBanks == 0:
      maxBanks = 255

   return ( minBanks, maxBanks )

# -------------------------------------------------------------------------------
# ip hardware fib hierarchical next-hop ecmp
#   level <n> bank minimum count <n>
# -------------------------------------------------------------------------------
class EcmpFecBankConfigCmd( CliCommand.CliCommandClass ):
   syntax = "level LEVEL bank minimum count COUNT"
   noOrDefaultSyntax = "level LEVEL ..."

   data = {
      "level": "FEC hierarchy level configuration",
      "LEVEL": CliMatcher.DynamicIntegerMatcher( rangeFn=ecmpFecBankLevelRangeFn,
                                                 helpdesc="FEC hierarchy level" ),
      "bank": "ECMP FEC banks to reserve",
      "minimum": "Minimum ECMP FEC banks to reserve",
      "count": "Number of banks",
      "COUNT": CliMatcher.DynamicIntegerMatcher( rangeFn=ecmpFecBankCountRangeFn,
                                                 helpdesc="ECMP FEC bank count" )
   }

   @staticmethod
   def handler( mode, args ):
      mode.addBankConfig( args[ "LEVEL" ], args[ "COUNT" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delBankConfig( args[ "LEVEL" ] )

HierNhEcmpFecMode.addCommandClass( EcmpFecBankConfigCmd )

# -------------------------------------------------------------------------------
# ip hardware fib hierarchical next-hop ecmp
#   show pending
# Show command to show the current pending config
# -------------------------------------------------------------------------------
class ShowPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pending'
   data = {
      'pending': 'Show pending config in this session'
   }

   @staticmethod
   def handler( mode, args ):
      return mode.showPending()

HierNhEcmpFecMode.addShowCommandClass( ShowPendingCmd )

# -------------------------------------------------------------------------------
# ip hardware fib hierarchical next-hop ecmp
#   abort
# To abort the current changes and keep the existing changes
# -------------------------------------------------------------------------------
class AbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = { 'abort': CliToken.Cli.abortMatcher, }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

HierNhEcmpFecMode.addCommandClass( AbortCmd )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHardwareConfig
   global routingHardwareStatus

   routingHardwareConfig = ConfigMount.mount( entityManager,
                                              "routing/hardware/config",
                                              "Routing::Hardware::Config", "w" )
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
