#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import Arnet.MplsLib
from CliPlugin import VrfCli
from CliPlugin import IntfCli
from CliPlugin import IraCommonCli
from CliPlugin import IraIp6Cli
from CliPlugin import IraIpIntfCli
from CliPlugin import IraRouteCommon
from CliPlugin import IraUrpfCli
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.IraIpCliCommon import (
      setRoutesInExactMatch,
      unsetRoutesInExactMatch,
      IPV4,
      enableOptimizePrefixes,
      cliTokenToOptimizeProfile,
      vrfFilterWarningCheck,
)
from CliPlugin.IraIpModel import (
      IpRouteSummaryForVrf,
      IpModel,
      IpMulticastRouting,
      VrfIpRibRoutes,
      UrpfInterface,
      UrpfInterfaces,
)
from CliPlugin.VrfCli import (
      ALL_VRF_NAME,
      DEFAULT_VRF,
      getAllVrfNames,
      vrfExists,
)
from CliPlugin.IraIpRouteCliLib import (
      manageRoute,
      noRoute,
      noRouteTableForVrfMsg,
      routing,
      staticRoutingTable,
)
from CliPlugin.IraIpCli import (
   allVrfConfig,
   allVrfStatusLocal,
   fhrpVrrpIntfsHook,
   filterIntfByRouteable,
   filterIntfByVrf,
   mCastRoutingHook,
   printer,
   routingHardwareStatus,
   showRoute,
   vrfSummaryCommon,
)
import ConfigMount
import LazyMount
import Tac
from TypeFuture import TacLazyType
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import Toggles.IraToggleLib
import Toggles.ArnetToggleLib

routingHardwareConfig = None
ipConfig = None
ipGlobalConfigParams = None

TriStateBoolEnum = TacLazyType( "Ip::TristateBool" )
UrpfMode = TacLazyType( "Urpf::UrpfMode" )

# -------------------------------------------------------------------------------
# The "ip routing [ipv6 interfaces] [vrf]" and
#     "no|default ip routing [keep-static-routes | delete-static-routes]
#       [addressless] [vrf]"
#     commands.
# -------------------------------------------------------------------------------

def enableIpRouting( mode, args ):
   vrf = args.get( 'VRF', DEFAULT_VRF )
   if not vrfExists( vrf ):
      # According to AID/12, we shouldn't be failing for this, but we have
      # existing behavior which customers may depend on.
      mode.addErrorAndStop( f'No such VRF {vrf}.' )
   config = routing.config( vrf )
   config.routing = True
   if 'interfaces' in args:
      config.addresslessForwarding = TriStateBoolEnum.isTrue

def disableIpRouting( mode, args ):
   vrf = args.get( 'VRF', DEFAULT_VRF )
   if not vrfExists( vrf ):
      # vrf is deleted or not configured, nothing to do.
      mode.addWarning( f'No such VRF {vrf}.' )
      return vrf
   config = routing.config( vrf )
   if 'interfaces' in args:
      config.addresslessForwarding = TriStateBoolEnum.isFalse
   else:
      config.routing = False

   if ( 'keep-static-routes' not in args and
        'delete-static-routes' not in args and
        not mode.session_.startupConfig() and
        staticRoutingTable( vrf ).route ):
      mode.addWarning( "Preserving static routes.  "
                       "Use 'no ip routing delete-static-routes' to clear them." )
   return vrf # Possibly used by `disableIpRoutingMaybeClearStaticRoutes`.

def disableIpRoutingMaybeClearStaticRoutes( mode, args ):
   vrf = disableIpRouting( mode, args )
   # Delete all of the static routes if requested.
   if 'delete-static-routes' in args:
      staticRoutingTable( vrf ).route.clear()

def handlerIcmpMsgParamsCmd( mode, args ):
   if 'burst' in args:
      allVrfConfig.icmpBurstSize = args.get( 'BURSTRATE', 1000000 )
   elif 'rate' in args:
      allVrfConfig.icmpMsgsPerSec = args.get( 'MSGRATE', 1000000 )

def handlerIcmpUnreachableRateCmd( mode, args ):
   # Setting rate to 0 makes ICMP _not_ unreachable.
   routingHardwareConfig.icmpUnreachable = 'RATE' not in args

def manageIpRoute( mode, prefix, nexthop, routeOptions=None, vrfName=DEFAULT_VRF,
                   leakToVrf=None, egressVrf=None, viaResRibProfile=None ):
   assert vrfName != ''
   if not vrfName:
      vrfName = DEFAULT_VRF

   if not vrfExists( vrfName ):
      mode.addError( f"{noRouteTableForVrfMsg % vrfName} Create first." )
      return

   manageRoute( mode, prefix, nexthop, routeOptions, vrfName, leakToVrf, egressVrf,
               viaResRibProfile=viaResRibProfile )

def handlerIpRouteNexthopOrComboCmd( mode, args ):
   manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                  routeOptions=args.get( 'OPTION' ),
                  vrfName=args.get( 'VRF', DEFAULT_VRF ),
                  leakToVrf=args.get( 'LEAK_TO_VRF' ) )

def handlerIpRouteNexthopResRibProfileCmd( mode, args ):
   manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_ADDR' ],
                  routeOptions=args.get( 'OPTION' ),
                  vrfName=args.get( 'VRF', DEFAULT_VRF ),
                  viaResRibProfile=args.get( 'SYS_CON_ONLY_RIBS' ) )

def handlerIpRouteMplsLabelCmd( mode, args ):
   manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                  routeOptions=args.get( 'OPTION' ),
                  vrfName=args.get( 'VRF', DEFAULT_VRF ),
                  leakToVrf=args.get( 'LEAK_TO_VRF' ) )

def handlerIpRouteNexthopGroupCmd( mode, args ):
   manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                  routeOptions=args.get( 'OPTION' ),
                  vrfName=args.get( 'VRF', DEFAULT_VRF ),
                  leakToVrf=args.get( 'LEAK_TO_VRF' ) )

def handlerIpRouteInterfaceCmd( mode, args ):
   if 'NEXTHOP_INTF' not in args:
      args[ 'NEXTHOP_INTF' ] = {}
   args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
   args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ] = args.pop( 'INTF' )
   args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = None

   manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                  routeOptions=args.get( 'OPTION' ),
                  vrfName=args.get( 'VRF', DEFAULT_VRF ),
                  leakToVrf=args.get( 'LEAK_TO_VRF' ) )

def handlerIpRouteEgressVrfCmd( mode, args ):
   manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                  routeOptions=args.get( 'OPTION' ),
                  vrfName=args.get( 'VRF', DEFAULT_VRF ),
                  leakToVrf=args.get( 'LEAK_TO_VRF' ),
                  egressVrf=args.get( 'EGRESS_VRF' ) )

def handlerIpRouteEgressVrfNexthopOrComboCmd( mode, args ):
   manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                  routeOptions=args.get( 'OPTION' ),
                  vrfName=args.get( 'VRF', DEFAULT_VRF ),
                  leakToVrf=args.get( 'LEAK_TO_VRF' ),
                  egressVrf=args.get( 'EGRESS_VRF' ) )

def handlerIpRouteEgressVrfInterfaceCmd( mode, args ):
   manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                  routeOptions=args.get( 'OPTION' ),
                  vrfName=args.get( 'VRF', DEFAULT_VRF ),
                  leakToVrf=args.get( 'LEAK_TO_VRF' ),
                  egressVrf=args.get( 'EGRESS_VRF' ) )

def noOrDefaultHandlerNoIpRouteCmd( mode, args ):
   noIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
               args.get( 'PREFERENCE' ), args.get( 'VRF' ),
               args.get( 'EGRESS_VRF' ) )

def handlerIpRouteCachedCmd( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      mode.addWarning( "Routing protocols model multi-agent must be "
                        "configured for route cache configuration" )
   manageIpRoute( mode, args[ 'PREFIX' ], 'routeCacheConnected',
                  vrfName=args.get( 'VRF' ) )

def noOrDefaultHandlerNoIpRouteCachedCmd( mode, args ):
   noIpRoute( mode, args[ 'PREFIX' ], 'routeCacheConnected',
               preference=None, vrfName=args.get( 'VRF' ), egressVrf=None )

def generateIpModel( mode, args ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
   assert vrfName
   r = routing.config( vrfName )
   ipModel = IpModel()
   ipModel.v4RoutingEnabled = r.routing

   # FHRP VRRP Interfaces Support information
   for hook in fhrpVrrpIntfsHook.extensions():
      # Only one fhrpVrrpIntfs hook is expected
      assert len( fhrpVrrpIntfsHook.extensions() ) == 1
      ipModel.vrrpIntfs = hook()

   # Supported Multicast routing
   for hook in mCastRoutingHook.extensions():
      # Only one mCastRouting hook is expected
      assert len( mCastRoutingHook.extensions() ) == 1
      ipModel.multicastRouting = IpMulticastRouting()
      ipModel.multicastRouting.ipMulticastEnabled = hook().get( 'v4' )
      ipModel.multicastRouting.ip6MulticastEnabled = hook().get( 'v6' )

   # Addressless Forwarding information
   if r.addresslessForwarding == 'isTrue':
      forwardingState = True
   elif r.addresslessForwarding == 'isFalse':
      forwardingState = False
   else:
      forwardingState = None
   ipModel.v6IntfForwarding = forwardingState

   # Populate CAPI model with the IPv4 URPF state
   ipModel.v4uRpfState = IraUrpfCli.getV4UrpfStatus()

   # Populate CAPI model with the IPv6 Ecmp state
   ( ipModel.v6RoutingEnabled, ipModel.v6EcmpInfo ) = IraIp6Cli.\
         generateIp6EcmpModel( mode, vrfName )

   # Populate CAPI model with IPv6 URPF state
   ipModel.v6uRpfState = IraUrpfCli.getV6UrpfStatus()
   return ipModel

def handlerShowIpRouteNextHopGroupCmd( mode, args ):
   NexthopGroupId = Tac.Type( "Routing::NexthopGroup::NexthopGroupId" )
   routeFilterList = Tac.newInstance( "Ira::RouteFilterColl", True )
   nhgName = args.get( 'NHG_NAME' )
   if nhgName:
      wildcard = False
      nhgId = routing.getNexthopGroupId( nhgName )
      if nhgId == NexthopGroupId.null:
         routeFilterList.filterAll = True
   else:
      nhgId = NexthopGroupId.null
      wildcard = True
   routeFilterNhg = Tac.newInstance( "Ira::RouteFilterNextHopGroup", nhgId,
                                       wildcard )
   routeFilterList.filterColl.add( routeFilterNhg )
   return showRoute( mode, vrfName=args.get( 'VRF' ),
                     routeFilter=routeFilterList )

def handlerShowIpRouteNextHopVxlanCmd( mode, args ):

   routeFilterList = Tac.newInstance( "Ira::RouteFilterColl",
                                       'all-vias' not in args )
   genAddr = args.get( 'IP' )
   wildcard = genAddr is None
   if wildcard:
      genAddr = Tac.Value( 'Arnet::IpGenAddr' )
   routeFilterVtep = Tac.newInstance( "Ira::RouteFilterVtep", genAddr, wildcard )
   routeFilterList.filterColl.add( routeFilterVtep )
   return showRoute( mode, vrfName=args.get( 'VRF' ),
                     routeFilter=routeFilterList )

def showV4NotInstalledBfd( mode, vrfName=None, optionsSet=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )

   if not ( vrfName is None or vrfName == ALL_VRF_NAME or vrfExists( vrfName ) ):
      mode.addError( f"IP routing table {vrfName} does not exist." )
      return None

   vrfs = []
   if vrfName == ALL_VRF_NAME:
      for vrf in sorted( allVrfStatusLocal.vrf ):
         vrfs.append( vrf )
      vrfs.insert( 0, DEFAULT_VRF )
   elif vrfName is not None:
      vrfs.append( vrfName )
   else:
      vrfs.append( DEFAULT_VRF )

   return IraCommonCli.showCommonNotInstalledBfd( vrfs=vrfs )

def handlerShowNotInstalledBfdCmd( mode, args ):
   return showV4NotInstalledBfd( mode, vrfName=args.get( 'VRF' ) )

def showRouteFec( mode, vrfName=None, prefix=None ):
   vrfFilterWarningCheck( mode, vrfName, prefix )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   vrfFecRoutesModel = IraRouteCommon.VrfFecRoutesModel()
   if vrfName != ALL_VRF_NAME and not vrfExists( vrfName ):
      mode.addError( noRouteTableForVrfMsg % vrfName )
      return vrfFecRoutesModel
   if isinstance( prefix, str ):
      prefix = Arnet.Prefix( prefix )
   if vrfName == ALL_VRF_NAME:
      vrfFecRoutesModel.setShowVrfs()
      vrf = DEFAULT_VRF
      vrfFecRoutesModel.vrfs[ vrf ] = routing.showFecs( mode,
                                                        prefix=prefix,
                                                        vrfName=vrf )
      vrfs = allVrfStatusLocal.vrf
   else:
      vrfs = [ vrfName ]

   for vrf in vrfs:
      vrfFecRoutesModel.vrfs[ vrf ] = routing.showFecs( mode,
                                                        prefix=prefix,
                                                        vrfName=vrf )
   return vrfFecRoutesModel

def handlerShowIpRouteFecCmd( mode, args ):
   return showRouteFec( mode, prefix=args.get( 'PREFIX' ),
                        vrfName=args.get( 'VRF' ) )

def showIpFecSummary( mode, vrfName=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   return routing.showFecs( mode, vrfName=vrfName, summary=True )

def handlerShowIpFecSummaryCmd( mode, args ):
   return showIpFecSummary( mode, vrfName=args.get( 'VRF' ) )

def showRouteLongerPrefixes( mode, prefix=None, optionsSet=None, vrfName=None ):
   # pylint check fails if appended directly to set
   optionsDict = dict( optionsSet or [] )
   optionsDict[ 'longerPrefixes' ] = 'longer-prefixes'
   optionsSet = list( optionsDict.items() )
   return showRoute( mode, prefix, optionsSet, vrfName )

def handlerShowIpRouteLongerPrefixesCmd( mode, args ):
   return showRouteLongerPrefixes( mode, prefix=args.get( 'PREFIX' ),
                                    optionsSet=args[ 'optionSet' ],
                                    vrfName=args.get( 'VRF' ) )

def showIpRouteVrfAllSumBrief( mode, quantify=False ):
   beforeSum = Tac.now()
   vrfs = []

   for vrfName in sorted( allVrfStatusLocal.vrf ):
      vrfs.append( vrfName )
   vrfs.insert( 0, DEFAULT_VRF )

   ipRouteSummaryBriefModel = IpRouteSummaryForVrf()
   ipRouteSummaryBriefModel.maskLen = {}
   ipRouteSummaryBriefModel.totalRoutes = 0
   ipRouteSummaryBriefModel.vrfCount = len( vrfs )
   ipRouteSummaryBriefModel = vrfSummaryCommon( mode, ipRouteSummaryBriefModel,
                                                vrfs )

   # pylint: disable=protected-access
   if quantify:
      afterSum = Tac.now()
      ipRouteSummaryBriefModel._quantify = afterSum - beforeSum
   return ipRouteSummaryBriefModel

def handlerShowIpRouteSummaryBriefCmd( mode, args ):
   quantify = 'quantify' in args
   return showIpRouteVrfAllSumBrief( mode, quantify=quantify )

def resilientEcmpPrefixIs( mode, prefix, capacity, redundancy, ordered, marked ):

   # if C * R > maxEcmp value, throw max value error
   if ( routingHardwareStatus.maxResilientEcmp and
        ( capacity * redundancy ) > routingHardwareStatus.maxResilientEcmp ):
      mode.addError( "Maximum value supported for capacity * redundancy is" +
                     f" {routingHardwareStatus.maxResilientEcmp}" )
      return
   ecmpConfig = Tac.Value( "Routing::Hardware::ResilientEcmpConfig",
                           capacity, redundancy )
   ecmpConfig.orderedVias = ( ordered is not None )
   if marked:
      routingHardwareConfig.policyBasedResilientEcmp = ecmpConfig
   else:
      # IpAddr.IpPrefixRule makes string (unlike Ip6Addr.Ip6PRefixRule) so
      # prefix is a IpAddrWithFullMaskTest
      genPrefix = Arnet.IpGenPrefix( prefix )
      routingHardwareConfig.resilientEcmpPrefix[ genPrefix ] = ecmpConfig

def handlerResilientEcmpCmd( mode, args ):
   resilientEcmpPrefixIs( mode, args.get( "PREFIX" ), args[ "CAPACITY" ],
                           args[ "REDUNDANCY" ], args.get( 'ordered' ),
                           args.get( 'marked' ) )

def noResilientEcmp( mode, args ):
   prefix = args.get( "PREFIX" )
   if prefix:
      # IpAddr.IpPrefixRule makes string (unlike Ip6Addr.Ip6PRefixRule) so
      # prefix is a IpAddrWithFullMask
      genPrefix = Arnet.IpGenPrefix( prefix )
      del routingHardwareConfig.resilientEcmpPrefix[ genPrefix ]
   else:
      # config for policy marked needs to be cleared whether 'marked' keyword
      # is specified or not
      ecmpConfig = Tac.Value( "Routing::Hardware::ResilientEcmpConfig", 0, 0 )
      routingHardwareConfig.policyBasedResilientEcmp = ecmpConfig
      if 'marked' not in args:
         # user asked to clear config for all the prefix as well
         routingHardwareConfig.resilientEcmpPrefix.clear()

def enableRoutesInExactMatch( mode, prefixLenSet=None ):
   setRoutesInExactMatch( mode, IPV4, routingHardwareConfig, routingHardwareStatus,
                          prefixLenSet=prefixLenSet or [] )

def handlerExactMatchPrefixLenCmd( mode, args ):
   enableRoutesInExactMatch( mode, args[ "LENGTH" ] )

def noOrDefaultHandlerExactMatchPrefixLenCmd( mode, args ):
   unsetRoutesInExactMatch( mode, IPV4, routingHardwareConfig,
                              routingHardwareStatus )

def handlerOptimizePrefixLenCmd( mode, args ):
   setRoutesInExactMatch( mode, IPV4, routingHardwareConfig,
                           routingHardwareStatus, args[ "LENGTH" ],
                           args.get( "EXPANDLEN" ), args.get( "COMPRESSLEN" ),
                           "urpf" in args )

def noOrDefaultHandlerOptimizePrefixLenCmd( mode, args ):
   unsetRoutesInExactMatch( mode, IPV4, routingHardwareConfig,
                              routingHardwareStatus,
                              clearPrefixesProfileConfig=False )

def handlerVrfOptimizePrefixLenCmd( mode, args ):
   setRoutesInExactMatch( mode, IPV4, routingHardwareConfig,
                           routingHardwareStatus, args[ "LENGTH" ],
                           args.get( "EXPANDLEN" ),
                           args.get( "COMPRESSLEN" ), "urpf" in args,
                           vrfs=args[ "VRFS" ] )

def noOrDefaultHandlerVrfOptimizePrefixLenCmd( mode, args ):
   unsetRoutesInExactMatch( mode, IPV4, routingHardwareConfig,
                              routingHardwareStatus, vrfs=args[ "VRFS" ] )

def handlerOptimizePrefixesCmd( mode, args ):
   enableOptimizePrefixes(
      mode, IPV4, routingHardwareConfig,
      profileType=cliTokenToOptimizeProfile.get( args[ 'PROFILE' ], '' ) )

def noOrDefaultHandlerOptimizePrefixesCmd( mode, args ):
   unsetRoutesInExactMatch( mode, IPV4, routingHardwareConfig,
                              routingHardwareStatus,
                              clearPrefixLengthConfig=False )

def handlerVrfOptimizePrefixesCmd( mode, args ):
   enableOptimizePrefixes(
      mode, IPV4, routingHardwareConfig,
      profileType=cliTokenToOptimizeProfile.get( args[ 'PROFILE' ], '' ),
      vrfs=args[ 'VRFS' ] )

restartL3AgentWarnStrDisableVrf = "Please restart layer 3 forwarding agent to " \
      "ensure that the disable-vrf option change takes effect"

def setDisableVrf( mode, disableVrfSet ):
   # If previously disabled VRFs disagree with the current configuration,
   # then disableVrfConfigChanged = True
   disableVrfConfigChanged = False

   # disableVrfSet should always be non-empty
   assert disableVrfSet
   if set( disableVrfSet ) != set( routingHardwareConfig.optimizeDisabledVrf ):
      disableVrfConfigChanged = True
      routingHardwareConfig.optimizeDisabledVrf.clear()
      for vrf in set( disableVrfSet ):
         routingHardwareConfig.optimizeDisabledVrf[ vrf ] = True

   if disableVrfConfigChanged:
      mode.addWarning( restartL3AgentWarnStrDisableVrf )

def unsetDisableVrf( mode ):
   if routingHardwareConfig.optimizeDisabledVrf:
      routingHardwareConfig.optimizeDisabledVrf.clear()
      mode.addWarning( restartL3AgentWarnStrDisableVrf )

def handlerSetDisableVrfCmd( mode, args ):
   if "VRF" in args:
      setDisableVrf( mode, args[ "VRF" ] )

def noOrDefaultHandlerSetDisableVrfCmd( mode, args ):
   if "disable-vrf" not in args:
      unsetRoutesInExactMatch( mode, IPV4, routingHardwareConfig,
                                 routingHardwareStatus )
   unsetDisableVrf( mode )

def urpfInterfaceFromIntfConfig( intfConfig ):
   uRpfIntf = UrpfInterface()
   if not intfConfig:
      uRpfIntf.uRpfMode = UrpfMode.disable
      uRpfIntf.allowDefault = False
      return uRpfIntf
   urpfMode = intfConfig.urpf.mode
   allowDefaultRoute = intfConfig.urpf.allowDefaultRoute
   uRpfIntf.uRpfMode = urpfMode
   # TODO BUG642564: When strictDefault is removed from UrpfMode enum,
   # update the check here.
   uRpfIntf.allowDefault = \
         ( urpfMode == UrpfMode.strictDefault or
           ( Toggles.IraToggleLib.toggleUrpfIgnoreDefaultRouteEnabled() and
             routingHardwareStatus.urpfLooseNonDefaultSupported and
             urpfMode == UrpfMode.loose and allowDefaultRoute ) )
   uRpfIntf.lookupVrf = intfConfig.urpf.lookupVrf
   return uRpfIntf

def populateUrpfModel( mode, intfs=None ):
   uRpfModel = UrpfInterfaces()
   if intfs is None:
      uRpfModel.interfaces = None
      return uRpfModel
   for intf in intfs:
      if not intf.lookup():
         # The command has no interface filter so skip interfaces that have
         # been deleted before this loop could get to them.
         continue
      intf = IraIpIntfCli.IpIntf( intf, mode.sysdbRoot, createIfMissing=False )
      if intf.config().urpf.mode != UrpfMode.disable:
         uRpfIntf = urpfInterfaceFromIntfConfig( intf.config() )
         # pylint: disable-next=unsupported-assignment-operation
         uRpfModel.interfaces[ intf.name ] = uRpfIntf
   return uRpfModel

def populateUrpfPrefixModel( mode, prefix, vrfName ):
   uRpfModel = UrpfInterfaces()
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not ( vrfName is None or vrfExists( vrfName ) ):
      mode.addError( noRouteTableForVrfMsg % vrfName )
      uRpfModel.interfaces = None
      return uRpfModel

   prefix = Arnet.Prefix( prefix )
   routeEntry = routing.route( prefix, vrfName=vrfName )
   if routeEntry is None:
      uRpfModel.interfaces = None
      return uRpfModel

   uRpfModel.prefix = prefix.stringValue

   uRpfModel.viaDefaultRoute = \
      routeEntry.prefix == routing.ip.defaultRoute

   fec = routing.fGenStatus( vrfName ).fec.get( routeEntry.fecId )
   for via in fec.via.values():
      intf = via.intfId
      intfConfig = ipConfig.ipIntfConfig.get( intf )
      uRpfIntf = urpfInterfaceFromIntfConfig( intfConfig )
      # pylint: disable-next=unsupported-assignment-operation
      uRpfModel.interfaces[ intf ] = uRpfIntf
   return uRpfModel

def showUrpf( mode, vrfName=None, prefix=None ):

   # Get all routeable intefaces
   intfs = IntfCli.Intf.getAll( mode )
   intfs = filterIntfByRouteable( mode, intfs )

   # Filter interfaces by VRF
   if vrfName:
      intfs = filterIntfByVrf( mode, intfs, vrfName )

   if not prefix:
      return populateUrpfModel( mode, intfs )
   else:
      return populateUrpfPrefixModel( mode, prefix, vrfName )

def handlerShowUrpfCmd( mode, args ):
   return showUrpf( mode, vrfName=args.get( 'VRF' ), prefix=args.get( 'PREFIX' ) )

def setIpRedirect( mode, no ):
   vrfNames = getAllVrfNames( mode )
   for vrf in vrfNames:
      r = routing.config( vrf )
      r.sendRedirects = not no

def noIpRedirect( mode ):
   setIpRedirect( mode, no=True )

def ipRedirect( mode ):
   setIpRedirect( mode, no=False )

def handlerIpIcmpRedirectCmd( mode, args ):
   ipRedirect( mode )

def noHandlerIpIcmpRedirectCmd( mode, args ):
   noIpRedirect( mode )

def defaultHandlerIpIcmpRedirectCmd( mode, args ):
   ipRedirect( mode )

def setIcmpSrcIntf( mode, intf, vrfName=DEFAULT_VRF ):
   if not vrfName:
      vrfName = DEFAULT_VRF
   r = routing.config( vrfName )
   if r:
      if intf:
         if intf.config():
            r.icmpSrcIntf = intf.config().intfId
         else:
            mode.addError( f"Cannot find interface {intf}" )
      else:
         r.icmpSrcIntf = ""

def noIcmpSrcIntf( mode, vrfName=DEFAULT_VRF ):
   setIcmpSrcIntf( mode, None, vrfName )

def handlerSetIcmpSrcIntfCmd( mode, args ):
   setIcmpSrcIntf( mode, args[ 'SRC_INTF' ],
                     vrfName=args.get( 'VRF', DEFAULT_VRF ) )

def noOrDefaultHandlerSetIcmpSrcIntfCmd( mode, args ):
   noIcmpSrcIntf( mode, vrfName=args.get( 'VRF', DEFAULT_VRF ) )

def setFibFilter( mode, enable=True ):
   routingHardwareConfig.fibSuppression = enable

def noSetFibFilter( mode ):
   setFibFilter( mode, enable=False )

def handlerFibCompressionCmd( mode, args ):
   setFibFilter( mode )

def noOrDefaultHandlerFibCompressionCmd( mode, args ):
   noSetFibFilter( mode )

def handlerShowIpUserRouteCmd( mode, args ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, args.get( 'VRF' ) )
   if not vrfExists( vrfName ):
      mode.addError( noRouteTableForVrfMsg % vrfName )
      return None
   fmt = mode.session_.outputFormat()
   tag = args.get( 'TAGNUM', 0 )
   configTag = args.get( 'CONFIG_TAG', "" )
   routing.showUserRoutes( vrfName=vrfName, tag=tag,
                           configTag=configTag,
                           fmt=fmt, mlib=printer )
   return VrfIpRibRoutes

def handlerIpDuplicateAddressDetectionCmd( mode, args ):
   enableLogging = 'disabled' not in args
   ipGlobalConfigParams.duplicateAddressDetectionLogging = \
         enableLogging

def noIpRoute( mode, prefix, nexthop, preference, vrfName, egressVrf,
      configTag=False ):
   if not vrfName:
      vrfName = DEFAULT_VRF
   assert vrfName != ''

   if not vrfExists( vrfName ):
      # VRF is deleted, nothing more to do.
      mode.addWarning( f"No such VRF {vrfName}." )
      return

   noRoute( mode, prefix, nexthop, preference, vrfName, egressVrf, configTag )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHardwareConfig
   global ipConfig
   global ipGlobalConfigParams

   routingHardwareConfig = ConfigMount.mount( entityManager,
                                              "routing/hardware/config",
                                              "Routing::Hardware::Config", "w" )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   ipGlobalConfigParams = ConfigMount.mount( entityManager,
                                              "ip/globalConfigParams",
                                              "Ip::IpGlobalConfigParams",
                                              "w" )
