# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import os

import AgentCommandRequest
from CliDynamicSymbol import CliDynamicPlugin
from CliModel import cliPrinted
from CliPlugin.KernelFibCli import (
RouterKernelAfConfigMode,
KernelFibRcfCmd,
KernelFibRcfDetailCmd,
kernelfibConfig,
kernelfibConfig6
)
from CliPlugin.RcfCliLib import (
stripParens,
)
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin.IraIpRouteCliLib import isValidPrefixWithError
import KernelFibAgent
from TypeFuture import TacLazyType
import Tracing
import Tac
import Arnet

traceHandle = Tracing.Handle( 'KernelFibCli' )
t5 = traceHandle.trace5 # Info

KernelFibCliModels = CliDynamicPlugin( "KernelFibCliDynamicModels" )
kernelFibConstants = TacLazyType( 'KernelFib::KernelFibConstants' )

def getKernelFibVrfConfig( mode, addrFamily, vrfName ):
   assert addrFamily in [ 'ipv4', 'ipv6' ]
   config = kernelfibConfig if addrFamily == 'ipv4' else kernelfibConfig6
   vrfConfig = config.vrfConfig.get( vrfName )
   if not vrfConfig:
      # The parent 'router kernel' mode got deleted by another user, most likely
      mode.addError( f'Failed to find "router kernel'
                     f'{"" if vrfName == "default" else f" vrf {vrfName}"}"'
                     ' in running-config.' )
   return vrfConfig

# Set / Reset the preference value for kernel routes
def setKernelRouteAdminDistance( mode, addrFamily, vrfName, distance=None ):
   vrfConfig = getKernelFibVrfConfig( mode, addrFamily, vrfName )
   if vrfConfig:
      t5( 'Setting kernel', mode.addrFamily, 'route preference to',
          str( distance ) if distance else 'default', 'in VRF', mode.vrfName )
      vrfConfig.preference = distance or kernelFibConstants.defaultPreference()

# Set / Reset the name of the Route Control Function that makes decisions on
# the offloading of software forwarding to hardware
def setSwFwdingHwOffloadRcf( mode, addrFamily, vrfName, funcName=None ):
   vrfConfig = getKernelFibVrfConfig( mode, addrFamily, vrfName )
   if vrfConfig:
      t5( 'Setting hardware offload RCF name to', funcName or 'default',
          'for kernel', mode.addrFamily, 'routes in VRF', mode.vrfName )
      vrfConfig.hwOffloadRcfName = funcName or ''

# Set / Reset the interface providing preferred source address for kernel routes
def setSwFwdingHwOffloadLocalIntf( mode, addrFamily, vrfName, intfName=None ):
   vrfConfig = getKernelFibVrfConfig( mode, addrFamily, vrfName )
   if vrfConfig:
      t5( 'Setting hardware offload local interface to', intfName or 'default',
          'for kernel', mode.addrFamily, 'routes in VRF', mode.vrfName )
      vrfConfig.hwOffloadLocalIntfName = intfName or ''

# Enable / Disable Hardware Offload Route Lookup
def setSwFwdingHwOffloadRouteLookup( mode, addrFamily, vrfName, enable=True ):
   vrfConfig = getKernelFibVrfConfig( mode, addrFamily, vrfName )
   if vrfConfig:
      t5( 'Setting hardware offload route lookup for ', mode.vrfName,
          ' (', mode.addrFamily, ') to ', enable )
      mode.addWarning( "A hardware offload default route will be added if "
                       "necessary. The KernelFib agent needs to be restarted "
                       "in order to have the offload route changes take effect."
                       "This can be done by running 'agent KernelFib terminate'."
                       )
      vrfConfig.hwOffloadDefaultRoute = enable

# Delete all standby kernel routes
def resetKernelFibStandbyRoutes( mode, addrFamily, vrfName):
   t5( 'Deleting all standby kernel routes')
   routeCol = kernelfibConfig.vrfConfig[vrfName].standbyRouteCol
   routeCol.routes.clear()

def handlerRouterKernelAfConfigModeCmd( mode, args ):
   addrFamily = args[ 'ADDR_FAMILY' ]
   childMode = mode.childMode( RouterKernelAfConfigMode, addrFamily=addrFamily )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerRouterKernelAfConfigModeCmd( mode, args ):
   addrFamily = args[ 'ADDR_FAMILY' ]
   childMode = mode.childMode( RouterKernelAfConfigMode, addrFamily=addrFamily )
   childMode.clearConfig()

def handlerDistanceDistanceCmd( mode, args ):
   setKernelRouteAdminDistance( mode, mode.addrFamily, mode.vrfName,
                                 distance=args[ 'DISTANCE' ] )

def noOrDefaultHandlerDistanceDistanceCmd( mode, args ):
   setKernelRouteAdminDistance( mode, mode.addrFamily, mode.vrfName )

def handlerSwFwdingHwOffloadRouteRcfCmd( mode, args ):
   # rcfFunctionMatcher requires function names to end in "()", but we
   # don't store the trailing () internally so strip them off here
   args[ 'FUNCTION' ].removesuffix( '()' )
   setSwFwdingHwOffloadRcf( mode, mode.addrFamily, mode.vrfName,
                              funcName=args[ 'FUNCTION' ][ : -2 ] )

def noOrDefaultHandlerSwFwdingHwOffloadRouteRcfCmd( mode, args ):
   setSwFwdingHwOffloadRcf( mode, mode.addrFamily, mode.vrfName )

def handlerSwFwdingHwOffloadRouteLocalIntfCmd( mode, args ):
   setSwFwdingHwOffloadLocalIntf( mode, mode.addrFamily, mode.vrfName,
                                    intfName=args[ 'INTF' ].name )
def noOrDefaultHandlerSwFwdingHwOffloadRouteLocalIntfCmd( mode, args ):
   setSwFwdingHwOffloadLocalIntf( mode, mode.addrFamily, mode.vrfName )

def handlerSwFwdingHwOffloadRouteLookupCmd( mode, args ):
   setSwFwdingHwOffloadRouteLookup( mode, mode.addrFamily, mode.vrfName )

def noOrDefaultHandlerSwFwdingHwOffloadRouteLookupCmd( mode, args ):
   setSwFwdingHwOffloadRouteLookup( mode, mode.addrFamily, mode.vrfName,
         enable=False )

def handlerKernelSwFwdingEcmpVrfAllCmd( mode, args ):
   kernelfibConfig.programAllEcmpPaths = True

def noOrDefaultHandlerKernelSwFwdingEcmpVrfAllCmd( mode, args ):
   kernelfibConfig.programAllEcmpPaths = False

def handlerKernelSwFwdingHwOffloadSummaryCmd( mode, args ):
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
   command = f'SHOW_SW_FWDING_HW_OFFLOAD_SUMMARY {vrfName=!s}'

   AgentCommandRequest.runCliPrintSocketCommand( mode.entityManager,
                                                   KernelFibAgent.name(),
                                                   'sw-fwd-hw-offload-summary',
                                                   command,
                                                   mode )

   return cliPrinted( KernelFibCliModels.Fwd0EthTypeEntriesVrfs )

def handlerKernelSwFwdingHwOffloadConfigSanityCmd( mode, args ):
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
   command = f'SHOW_SW_FWDING_HW_OFFLOAD_CONFIG_SANITY {vrfName=!s}'

   AgentCommandRequest.runCliPrintSocketCommand(
      mode.entityManager,
      KernelFibAgent.name(),
      'sw-fwd-hw-offload-config-sanity',
      command,
      mode
   )

   return cliPrinted( KernelFibCliModels.Fwd0EthTypeConfigSanityEntriesVrfs )

def handlerBgpMonitoringInternalSchedulerResetCmd( mode, args ):
   # Since it's deprecated, this should never get called, but
   # if it is, let's make sure that the user knows that it didn't
   # work.
   mode.addError( 'Use "clear agent kernelFib task scheduler"' )

def handlerTechSupportKernelfibCmd( mode, args ):
   timeout = os.environ.get( 'KERNELFIB_BTEST', 120 )
   command = [ 'DUMP' ]
   if 'detail' in args:
      command += [ 'DETAILED' ]
   if 'rcf' in args:
      command += [ 'RCF' ]
   command += [ 'STATE' ]
   AgentCommandRequest.runSocketCommand( mode.entityManager,
                                         KernelFibAgent.name(),
                                         'tech-support',
                                         "_".join( command ),
                                         timeout=timeout )

def handlerKernelFibRcfCmd( mode, args ):
   command = 'SHOW_RCF'
   if 'FUNC_NAME' in args:
      funcNameParam = stripParens( args[ 'FUNC_NAME' ] )
      command += ' name=' + funcNameParam
   AgentCommandRequest.runCliPrintSocketCommand( mode.entityManager,
                                                 KernelFibAgent.name(),
                                                 'cli-print',
                                                 command,
                                                 mode )
   return cliPrinted( KernelFibRcfCmd.cliModel )

def handlerKernelFibRcfDetailCmd( mode, args ):
   command = 'SHOW_RCF_DETAIL'
   if 'FUNC_NAME' in args:
      funcNameParam = stripParens( args[ 'FUNC_NAME' ] )
      command += ' name=' + funcNameParam
   AgentCommandRequest.runCliPrintSocketCommand( mode.entityManager,
                                                 KernelFibAgent.name(),
                                                 'cli-print',
                                                 command,
                                                 mode )
   return cliPrinted( KernelFibRcfDetailCmd.cliModel )

def handlerKernelFibStandbyRouteCmd( mode, args ):
   if mode.addrFamily == "ipv6":
      mode.addError( "IPv6 mode not supported" )
      return
   prefixStr = args[ 'prefix' ]
   if not isValidPrefixWithError( mode, prefixStr ):
      return
   prefix = Arnet.IpGenPrefix( prefixStr )
   vrfName = mode.vrfName
   interface = Arnet.IntfId( args.get( 'interface', "" ) )
   nextHop = Arnet.IpGenAddr( args[ 'next-hop' ] )
   routeCollection = kernelfibConfig.vrfConfig[ vrfName ].standbyRouteCol
   if prefix not in routeCollection.routes:
      routeCollection.routes.newMember( prefix ) 
   route = routeCollection.routes[ prefix ]
   via = Tac.Value( "KernelFib::KernelVia", nextHop, interface )
   route.vias.add( via )
   
def noOrDefaultHandlerKernelFibStandbyRouteCmd( mode, args ):
   if mode.addrFamily == "ipv6":
      mode.addError( "IPv6 mode not supported" )
      return
   prefixStr = args[ 'prefix' ]
   if not isValidPrefixWithError( mode, prefixStr ):
      return
   prefix = Arnet.IpGenPrefix( prefixStr )
   vrfName = mode.vrfName
   nextHop = args.get( 'next-hop', "" )
   interface = args.get( 'interface', "" )
   routeCollection = kernelfibConfig.vrfConfig[ vrfName ].standbyRouteCol
   if prefix not in routeCollection.routes:
      return 
   route = routeCollection.routes[ prefix ]
   # delete all vias
   if ( not nextHop ) and ( not interface ):
      del routeCollection.routes[ prefix ]
   # delete a specific via
   elif nextHop: 
      via = Tac.Value( "KernelFib::KernelVia", Arnet.IpGenAddr( nextHop ),
                      Arnet.IntfId( interface ))
      route.vias.remove( via )
      if not route.vias:
         del routeCollection.routes[ prefix ]