# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module provides the CLI models used for rendering / capi-fying the L1 profile
card enumeration command.
'''

import CliModel
from Fru.ModuleRange import moduleListToModuleRangeString
import TableOutput

leftJustifyCol = TableOutput.Format( justify='left' )

# This is a handpicked constant that will determine when the Applicable Modules
# column header will wrap or not. If the maximum displayed module range is lower
# than this number, we will display the column header as two rows. Otherwise it
# will expand into 1 row. Example renders surrounding the limit are as follows:
# ..............v      ..............v
#                      Applicable
# Applicable Modules   Modules
# ------------------   -------------
# Linecard1,3,5,7      Linecard1,3,5
APPLICABLE_MODULE_WRAP_LEN = 15

class Enumeration( CliModel.Model ):
   '''A model representing a particular L1 profile to be enumerated. '''

   applicableModules = CliModel.List(
      valueType=str,
      optional=True,
      help='Applicable modules of L1 module profile' )
   description = CliModel.Str( help='The description of the L1 module profile' )

   def qualifiedApplicableModules( self ):
      if not self.applicableModules:
         return ''

      return ','.join( moduleListToModuleRangeString( self.applicableModules ) )

# TODO BUG790391: A CliModel.Dict is needed to model CLI defined modules, as well as
#                 a render loop. The Render order should be sorted by source then
#                 name, so we just need to add the render loop in the same fashion as
#                 builtin, but right after that (custom comes after builtin)
class Summary( CliModel.Model ):
   '''A model representing a summary of the L1 card profiles on the system. '''

   builtin = CliModel.Dict(
               valueType=Enumeration,
               optional=True,
               help='The builtin L1 module profiles installed on the system' )

   def maxModuleDisplayLength( self ):
      return max( ( len( profile.qualifiedApplicableModules() )
                  for profile in self.builtin.values() ),
                  default=0 )

   def render( self ):
      table = TableOutput.TableFormatter()
      applicableModulesHdr = 'Applicable Modules'
      if self.maxModuleDisplayLength() < APPLICABLE_MODULE_WRAP_LEN:
         applicableModulesHdr = 'Applicable\nModules'

      header = TableOutput.Headings( [ 'Name',
                                       'Source',
                                       applicableModulesHdr,
                                       'Summary' ] )
      header.doApplyHeaders( table )

      table.formatColumns( leftJustifyCol,
                           leftJustifyCol,
                           leftJustifyCol,
                           leftJustifyCol )

      for fmt in table.columnFormats_:
         fmt.padLimitIs( True )

      for profileName, profile in sorted( self.builtin.items() ):
         table.newRow( profileName,
                       'builtin',
                       profile.qualifiedApplicableModules().lower(),
                       profile.description )

      print( table.output() )
