# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
   Enum,
   Int,
   List,
   Model,
   Str
)
import TableOutput

class SerdesComponent( Model ):
   # Mark this model as not public so customers never see it
   __public__ = False

   sliceId = Str( help='Module' )
   chipType = Str( help='Type of chip' )
   chipId = Int( help='ID of the chip' )
   phyType = Str( help='Type of phy core' )
   coreId = Int( help='ID of phy core' )
   phyScope = Enum( values=[ 'Unknown', 'Line', 'System' ], help='Side of the chip' )
   direction = Enum( values=[ 'Tx', 'Rx' ], help='Direction of the lane' )
   physicalLane = Int( help='Physical SerDes ID' )
   logicalLane = Int( help='Logical SerDes ID' )
   polarity = Enum( values=[ 'Default', 'Swap' ],
                    help='SerDes Lane Polarity' )

   @staticmethod
   def populate( serdesDescriptor, logicalLane, polarity ):
      cliSerdes = SerdesComponent()
      cliSerdes.sliceId = serdesDescriptor.sliceId
      cliSerdes.chipType = serdesDescriptor.chipType
      cliSerdes.chipId = serdesDescriptor.chipId
      cliSerdes.phyType = serdesDescriptor.phyType
      cliSerdes.coreId = serdesDescriptor.coreId
      cliSerdes.phyScope = serdesDescriptor.phyScope.removeprefix( 'phyScope' )
      cliSerdes.direction = 'Tx' if serdesDescriptor.tx else 'Rx'
      cliSerdes.physicalLane = serdesDescriptor.physicalLane
      cliSerdes.logicalLane = logicalLane
      cliSerdes.polarity = polarity.removeprefix( 'polarity' )

      return cliSerdes

class L1TopologyComponent( Model ):
   # Mark this model as not public so customers never see it
   __public__ = False

   serdesGroups = List( valueType=SerdesComponent,
                        help="A list of SerdesTopology's" )

   def render( self ):
      print( 'WARNING! This is a debug command only for use under supervision '
             'of Arista customer support or engineering\n' )

      table = TableOutput.createTable( [ 'Module',
                                         'Chip',
                                         'Core',
                                         'Scope',
                                         'Direction',
                                         'Physical SerDes',
                                         'Logical SerDes',
                                         'Polarity' ] )

      fmtLeft = TableOutput.Format( justify='left' )
      fmtLeft.padLimitIs( True )

      fModule = fmtLeft
      fChip = fmtLeft
      fCore = fmtLeft
      fScope = fmtLeft
      fDirection = fmtLeft
      fPhysicalSerDes = TableOutput.Format( maxWidth=8, wrap=True )
      fPhysicalSerDes.padLimitIs( True )
      fLogicalSerDes = TableOutput.Format( maxWidth=7, wrap=True )
      fLogicalSerDes.padLimitIs( True )
      fPolarity = fmtLeft

      table.formatColumns( fModule, fChip, fCore, fScope,
                           fDirection, fPhysicalSerDes, fLogicalSerDes, fPolarity )

      for tableEntry in sorted( self.serdesGroups,
                                key=lambda t: ( t.sliceId, t.chipType, t.chipId,
                                                t.phyType, t.coreId, t.phyScope,
                                                t.direction, t.physicalLane,
                                                t.logicalLane, t.polarity ) ):
         table.newRow( tableEntry.sliceId,
                       f'{ tableEntry.chipType } { tableEntry.chipId }',
                       f'{ tableEntry.phyType } {tableEntry.coreId }',
                       tableEntry.phyScope,
                       tableEntry.direction,
                       tableEntry.physicalLane,
                       tableEntry.logicalLane,
                       tableEntry.polarity )

      print( table.output() )
