#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliDynamicSymbol
import CliGlobal
import CliMatcher
import CliMode.Ldap
import CliParser
import ConfigMount
import HostnameCli
import LazyMount
import LdapGroup
import Tac
from AaaDefs import roleNameRe
from CliPlugin import ConfigMgmtMode
from CliPlugin.VrfCli import VrfExprFactory
from LocalUserLib import usernameRe
import ReversibleSecretCli

DEFAULT_ROLE_PRIVLEVEL = 1
_timeoutMin = 1
_timeoutMax = 1000

gv = CliGlobal.CliGlobal( ldapConfig=None,
                          sslConfig=None,
                          localConfig=None )

# -------------------------------------------
#  (config)#[ no | default ] management ldap
# -------------------------------------------
class LdapConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = 'Ldap configuration'

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, 'ldap' )

   def enterCmd( self ):
      return "management ldap"

# ---------------------------------------------------------------
# (config-mgmt-ldap)# [ no | default ] group policy <POLICYNAME>
# ---------------------------------------------------------------
groupKwMatcher = CliMatcher.KeywordMatcher(
   'group',
   helpdesc='Configure LDAP group options' )
policyKwMatcher = CliMatcher.KeywordMatcher(
   'policy',
   helpdesc='Configure LDAP group policy' )
policyNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( name for name in gv.ldapConfig.groupPolicy ),
   helpname="WORD",
   helpdesc='Policy name',
   priority=CliParser.PRIO_LOW )

class GroupPolicyConfigMode( CliMode.Ldap.GroupPolicyConfigModeBase,
                             BasicCli.ConfigModeBase ):
   name = "Group policy configuration mode"

   def __init__( self, parent, session, pName=None ):
      CliMode.Ldap.GroupPolicyConfigModeBase.__init__( self, pName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.pName = pName
      sysdbGroupPolicy = gv.ldapConfig.groupPolicy.newMember( pName )
      # this symbol is lazy-loaded
      GroupPrivilegeSeq = CliDynamicSymbol.resolveSymbol(
         "LdapHandler.GroupPrivilegeSeq" )
      self.groupPolicy_ = GroupPrivilegeSeq( pName,
                                             sysdbGroupPolicy.groupRolePrivilege )

   def onExit( self ):
      sysdbGroupPolicy = gv.ldapConfig.groupPolicy.get( self.pName )
      if ( sysdbGroupPolicy and not sysdbGroupPolicy.groupRolePrivilege and
           sysdbGroupPolicy.searchFilter == Tac.Value( "Ldap::ObjectClassOptions",
                                                       "", "" ) ):
         del gv.ldapConfig.groupPolicy[ self.pName ]

      BasicCli.ConfigModeBase.onExit( self )

class EnterGroupPolicyConfig( CliCommand.CliCommandClass ):
   syntax = """group policy <POLICYNAME>"""
   noOrDefaultSyntax = syntax
   data = { 'group': groupKwMatcher,
            'policy': policyKwMatcher,
            "<POLICYNAME>": policyNameMatcher
          }
   handler = "LdapHandler.enterGroupPolicyMode"
   noOrDefaultHandler = "LdapHandler.noGroupPolicyMode"

LdapConfigMode.addCommandClass( EnterGroupPolicyConfig )

# -----------------------------------------------------------------------------------
# (config-mgmt-ldap-group-policy)# [ no | default ] group <GROUPNAME> role <ROLENAME>
# [ privilege <PRIVLEVEL> ]
# -----------------------------------------------------------------------------------
roleKwMatcher = CliMatcher.KeywordMatcher(
   'role',
   helpdesc='specify role' )

privilegeKwMatcher = CliMatcher.KeywordMatcher(
   'privilege',
   helpdesc='specify privilege' )

beforeKwMatcher = CliMatcher.KeywordMatcher(
   'before',
   helpdesc='Add this rule immediately before <before-rule>' )
afterKwMatcher = CliMatcher.KeywordMatcher(
   'after',
   helpdesc='Add this rule immediately after <after-rule>' )

roleKwMatcher = CliMatcher.KeywordMatcher(
   'role',
   helpdesc='specify role' )

groupNameMatcher = CliMatcher.QuotedStringMatcher(
   helpname="Quoted String",
   helpdesc="Group name" )

roleNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: gv.localConfig.role, "Role name",
   pattern=roleNameRe, helpname="WORD" )

class GroupRole( CliCommand.CliCommandClass ):
   syntax = 'group <GROUPNAME> role <ROLENAME> [ privilege <PRIVLEVEL> ]' \
            '[ ( before <BGROUPNAME> ) | ( after <AGROUPNAME> ) ]'
   noOrDefaultSyntax = 'group <GROUPNAME> [ role <ROLENAME>' \
                       '[ privilege <PRIVLEVEL> ] ] ...'
   data = { 'group': groupKwMatcher,
            '<GROUPNAME>': groupNameMatcher,
            'role': roleKwMatcher,
            '<ROLENAME>': roleNameMatcher,
            'privilege': privilegeKwMatcher,
            '<PRIVLEVEL>': CliMatcher.IntegerMatcher( 0, 15,
                                                      helpdesc="Privilege level" ),
            'before': 'Add this rule immediately before <before-rule>',
            '<BGROUPNAME>': groupNameMatcher,
            'after': 'Add this rule immediately after <after-rule>',
            '<AGROUPNAME>': groupNameMatcher,
            }
   handler = "LdapHandler.enterGroupRoleMode"
   noOrDefaultHandler = "LdapHandler.noGroupRoleMode"

GroupPolicyConfigMode.addCommandClass( GroupRole )

# -----------------------------------------------------------------------------------
# (config-mgmt-ldap-group-policy)# [ no | default ] search filter objectclass
# <GROUPKWD> attribute <MEMBERKWD>
# -----------------------------------------------------------------------------------
searchKwMatcher = CliMatcher.KeywordMatcher(
   'search',
   helpdesc='Configure search options' )

class SearchFilterObjectclass( CliCommand.CliCommandClass ):
   syntax = 'search filter objectclass <GROUPKWD> attribute <MEMBERKWD>'
   noOrDefaultSyntax = 'search filter objectclass ...'
   data = {
      "search": searchKwMatcher,
      "filter": "Configure search filter options",
      "objectclass": "Configure objectclass parameters",
      "<GROUPKWD>": CliMatcher.PatternMatcher(
         pattern=r'[A-Za-z0-9_:{}\[\]-]+',
         helpname="WORD",
         helpdesc="objectclass value" ),
      "attribute": CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'attribute',
         helpdesc="Configure search attribute for the objectclass" ) ),
      "<MEMBERKWD>": CliMatcher.PatternMatcher(
         pattern=r'[A-Za-z0-9_:{}\[\]-]+',
         helpname="WORD",
         helpdesc="attribute for objectclass" )
      }

   @staticmethod
   def handler( mode, args ):
      groupPolicy = gv.ldapConfig.groupPolicy.get( mode.pName )
      if groupPolicy:
         groupPolicy.searchFilter = Tac.Value( "Ldap::ObjectClassOptions",
                                               args[ "<GROUPKWD>" ],
                                               args[ "<MEMBERKWD>" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      groupPolicy = gv.ldapConfig.groupPolicy.get( mode.pName )
      if groupPolicy:
         groupPolicy.searchFilter = Tac.Value( "Ldap::ObjectClassOptions", "", "" )

GroupPolicyConfigMode.addCommandClass( SearchFilterObjectclass )

# -----------------------------------------------------------------------------------
# (config-mgmt-ldap)# [ no | default ] server defaults
# and
# (config-mgmt-ldap)# [ no | default ] server host <HOSTNAME> [ ( vrf <VRFNAME> ) ]
#   [ ( port <PORT> ) ]
# -----------------------------------------------------------------------------------
serverKwMatcher = CliMatcher.KeywordMatcher(
   'server',
   helpdesc='Configure LDAP options for the server' )

defaultsKwMatcher = CliMatcher.KeywordMatcher(
   'defaults',
   helpdesc='Configure default LDAP options for all servers' )

class ServerConfigMode( CliMode.Ldap.ServerConfigModeBase,
                        BasicCli.ConfigModeBase ):
   name = "Server configuration mode"

   def __init__( self, parent, session, label=None ):
      CliMode.Ldap.ServerConfigModeBase.__init__( self, label )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.label = label

   def onExit( self ):
      BasicCli.ConfigModeBase.onExit( self )

class EnterServerConfig( CliCommand.CliCommandClass ):
   syntax = 'server ( defaults | ( host <HOSTNAME> ' \
            '[ { ( VRF ) | ( port <PORT> ) } ] ) )'
   noOrDefaultSyntax = 'server ( defaults | ( host ' \
                       '[ <HOSTNAME> [ VRF ] ' \
                       '[ port <PORT> ] ] ) ) ...'

   data = { 'server': serverKwMatcher,
            'defaults': defaultsKwMatcher,
            'host': 'Specify a LDAP server',
            '<HOSTNAME>': HostnameCli.IpAddrOrHostnameMatcher(
               helpname='WORD',
               helpdesc='Hostname or IP address of LDAP server',
               ipv6=True ),
            'VRF': VrfExprFactory( helpdesc='VRF for this LDAP server',
                                   maxMatches=1 ),
            'port':
            CliCommand.singleKeyword(
                  'port', ( f'port of LDAP server (default '
                            f'{LdapGroup.DEFAULT_LDAP_PORT})' ) ),
            '<PORT>': CliMatcher.IntegerMatcher( 0, 65535,
               helpdesc='Number of the port to use' )
          }

   handler = "LdapHandler.enterServerMode"
   noOrDefaultHandler = "LdapHandler.noServerMode"

LdapConfigMode.addCommandClass( EnterServerConfig )

def ldapConfigure( label, attribute, value ):
   if label == 'defaults':
      setattr( gv.ldapConfig.defaultConfig, attribute, value )
   elif label in gv.ldapConfig.host:
      host = gv.ldapConfig.host[ label ]
      setattr( host.serverConfig, attribute, value )

# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] ssl-profile <profile-name>
# -------------------------------------------------------------------------------
class SslProfile( CliCommand.CliCommandClass ):
   syntax = """ssl-profile <profile-name>"""
   noOrDefaultSyntax = "ssl-profile ..."
   data = {
      "ssl-profile": "Configure SSL profile to use",
      "<profile-name>": CliMatcher.DynamicNameMatcher(
         lambda mode: ( name for name in gv.sslConfig.profileConfig ),
         helpname="WORD",
         helpdesc="Profile name",
         priority=CliParser.PRIO_LOW )
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ '<profile-name>' ]
      ldapConfigure( mode.label, 'sslProfile', profileName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ldapConfigure( mode.label, 'sslProfile', "" )

ServerConfigMode.addCommandClass( SslProfile )

# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] base-dn <WORD>
# -------------------------------------------------------------------------------
class BaseDN( CliCommand.CliCommandClass ):
   syntax = """base-dn <base-dn>"""
   noOrDefaultSyntax = "base-dn ..."
   data = {
      "base-dn": "Configure base Distinguished Name to use",
      "<base-dn>": CliMatcher.QuotedStringMatcher(
         helpdesc="base Distinguised Name",
         helpname="WORD" )
   }

   @staticmethod
   def handler( mode, args ):
      baseDn = args[ '<base-dn>' ]
      ldapConfigure( mode.label, 'baseDn', baseDn )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ldapConfigure( mode.label, 'baseDn', "" )

ServerConfigMode.addCommandClass( BaseDN )

# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] timeout <1-1000>
# -------------------------------------------------------------------------------
class Timeout( CliCommand.CliCommandClass ):
   syntax = """timeout <TIMEOUT>"""
   noOrDefaultSyntax = "timeout ..."
   data = {
      'timeout': 'Time to wait for a Ldap server to respond',
      '<TIMEOUT>': CliMatcher.IntegerMatcher(
         _timeoutMin, _timeoutMax,
         helpdesc='Number of seconds' )
   }

   @staticmethod
   def handler( mode, args ):
      timeout = args[ '<TIMEOUT>' ]
      ldapConfigure( mode.label, 'ldapTimeout', timeout )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ldapConfigure( mode.label, 'ldapTimeout',
                     gv.ldapConfig.defaultConfig.defaultLdapTimeout )

ServerConfigMode.addCommandClass( Timeout )

# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] rdn attribute user <WORD>
# -------------------------------------------------------------------------------
class RdnUser( CliCommand.CliCommandClass ):
   syntax = """rdn attribute user <rdn-user>"""
   noOrDefaultSyntax = "rdn ..."
   data = {
      "rdn": "Configure Relative Distinguished Name to use",
      "attribute": "Configure attribute of RDN",
      "user": "Configure attribute user of RDN",
      "<rdn-user>": CliMatcher.StringMatcher( pattern=r'\w*',
                                              helpdesc="user RDN attribute",
                                              helpname="WORD" )
   }

   @staticmethod
   def handler( mode, args ):
      rdnUser = args[ '<rdn-user>' ]
      ldapConfigure( mode.label, 'userRdnAttribute', rdnUser )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ldapConfigure( mode.label, 'userRdnAttribute', "" )

ServerConfigMode.addCommandClass( RdnUser )

# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] authorization group policy
# <POLICYNAME>
# -------------------------------------------------------------------------------
authzKwMatcher = CliMatcher.KeywordMatcher(
   'authorization',
   helpdesc='Configure LDAP options for user authorization' )

class AuthzGroupPolicy( CliCommand.CliCommandClass ):
   syntax = """authorization group policy <POLICYNAME>"""
   noOrDefaultSyntax = "authorization group policy ..."
   data = {
      "authorization": authzKwMatcher,
      "group": groupKwMatcher,
      "policy": policyKwMatcher,
      "<POLICYNAME>": policyNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ "<POLICYNAME>" ]
      ldapConfigure( mode.label, 'activeGroupPolicy', policyName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ldapConfigure( mode.label, 'activeGroupPolicy', "" )

ServerConfigMode.addCommandClass( AuthzGroupPolicy )
# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] search username <USERNAME>
# password ( [ 0 <PASSWORD> ] | [ 7 <ENCRPASSWORD> ] | <PASSWORD> )
# -------------------------------------------------------------------------------
usernameMatcher = CliMatcher.QuotedStringMatcher( usernameRe,
                                                  helpname='<USERNAME>',
                                                  helpdesc='LDAP user name' )

class SearchUsername( CliCommand.CliCommandClass ):
   syntax = """search username <USERNAME> password <PASSWORD>"""
   noOrDefaultSyntax = "search username ..."
   data = {
      "search": searchKwMatcher,
      "username": "LDAP user name for search",
      "<USERNAME>": usernameMatcher,
      "password": "Configure password for the user",
      "<PASSWORD>": ReversibleSecretCli.defaultReversiblePwdCliExpr,
   }
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      uname = args[ "<USERNAME>" ]
      assert uname
      pswd = args[ "<PASSWORD>" ]
      ldapConfigure( mode.label, 'searchUsernamePassword',
                     Tac.Value( "Ldap::UsernamePassword", uname, pswd ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ldapConfigure( mode.label, 'searchUsernamePassword',
                     Tac.Value( "Ldap::UsernamePassword", "",
                                ReversibleSecretCli.getDefaultSecret() )
                                )

ServerConfigMode.addCommandClass( SearchUsername )

def Plugin( entityManager ):
   gv.ldapConfig = ConfigMount.mount( entityManager,
                                      'security/aaa/ldap/config',
                                      'Ldap::Config', 'w' )
   gv.sslConfig = LazyMount.mount( entityManager,
                                   'mgmt/security/ssl/config',
                                   'Mgmt::Security::Ssl::Config', 'r' )
   gv.localConfig = LazyMount.mount( entityManager,
                                     'security/aaa/local/config',
                                     'LocalUser::Config', 'r' )
