#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AaaCliLib
from AaaPluginLib import hostProtocol
import Ark
import Cell
import CliDynamicSymbol
import CliGlobal
from CliPlugin.VrfCli import DEFAULT_VRF
import ConfigMount
import HostnameCli
import Intf.Log
import LazyMount
import LdapGroup
import ReversibleSecretCli
import Tac
from Tracing import Handle, t0

LdapConfigCli = CliDynamicSymbol.CliDynamicPlugin( "LdapConfigCli" )
LdapModel = CliDynamicSymbol.CliDynamicPlugin( "LdapModel" )

__defaultTraceHandle__ = Handle( 'LdapHandler' )

DEFAULT_LDAP_TIMEOUT = 5
DEFAULT_ROLE_PRIVLEVEL = 1

gv = CliGlobal.CliGlobal( aaaConfig=None,
                          ldapConfig=None,
                          ldapCounterConfig=None,
                          ldapStatus=None,
                          sslConfig=None,
                          localConfig=None )

# util function to access or create an ldap host entry ( of type AaaPlugin::Host )
def ldapHost( label, hostnameOrIp, port, vrf, create=False ):
   hosts = gv.ldapConfig.host
   assert vrf and vrf != ''
   spec = Tac.Value( "Aaa::HostSpec", hostname=hostnameOrIp, port=port,
                     acctPort=0, vrf=vrf, protocol=hostProtocol.protoLdap )
   if spec in hosts:
      host = hosts[ spec ]
   elif create:
      t0( "Creating ldap host:", spec.hostname, ":", spec.port )
      host = hosts.newMember( spec )
      if host is None:
         t0( "Unable to create ldap host:", spec.hostname, ":", spec.port )
      else:
         host.index = AaaCliLib.getHostIndex( hosts )
         t0( spec.vrf, spec.hostname, "Host index", host.index )
   else:
      host = None

   if host is not None:
      assert host.hostname == hostnameOrIp
      assert host.port == port
   return host

def defaultServerDefaults( config ):
   config.sslProfile = ""
   config.baseDn = ""
   config.userRdnAttribute = ""
   config.activeGroupPolicy = ""
   config.searchUsernamePassword = Tac.Value(
         "Ldap::UsernamePassword", "", ReversibleSecretCli.getDefaultSecret() )
   config.ldapTimeout = config.defaultLdapTimeout

# -------------------------------------------
#  (config)#[ no | default ] management ldap
# -------------------------------------------
def enterMgmtLdapMode( mode, args ):
   gv.ldapConfig.defaultConfig = ( 'defaults', )
   childMode = mode.childMode( LdapConfigCli.LdapConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMgmtLdapMode( mode, args ):
   gv.ldapConfig.host.clear()
   gv.ldapConfig.groupPolicy.clear()
   if gv.ldapConfig.defaultConfig:
      defaultServerDefaults( gv.ldapConfig.defaultConfig )

# ---------------------------------------------------------------
# (config-mgmt-ldap)# [ no | default ] group policy <POLICYNAME>
# ---------------------------------------------------------------
# Util class to manipulate sequences of group to rule mappings.
# fromSysdb() reads in config from Sysdb to object local list and dict.
# insert() / delete() modifies object local list and dict.
# toSysdb() writes out object local list and dict to Sysdb.
class GroupPrivilegeSeq:
   def __init__( self, name, sysdbGroupPolicy ):
      self.groupPolicyName = name
      self.groupNamesInSeq = []
      self.groupToRolePrivilege = {}
      self.sysdbGroupPolicy = sysdbGroupPolicy
      # read in from Sysdb
      self.fromSysdb()

   def exists( self, groupName, role, privilege ):
      if self.groupToRolePrivilege.get( groupName ) == ( role, privilege ):
         return True
      return False
   # bgroupName = groupname to insert before
   # agroupName = groupname to insert after

   def insert( self, groupName, role, privilege, bgroupName, agroupName ):
      t0( "insert", groupName, role, privilege, bgroupName, agroupName )
      if bgroupName and bgroupName != groupName:
         try:
            self.groupNamesInSeq.insert( self.groupNamesInSeq.index( bgroupName ),
                                         groupName )
         except ValueError:  # bgroupName not found
            self.groupNamesInSeq.append( groupName )
      elif agroupName and agroupName != groupName:
         try:
            self.groupNamesInSeq.insert( self.groupNamesInSeq.index(
               agroupName ) + 1, groupName )
         except ValueError:  # agroupName not found
            self.groupNamesInSeq.append( groupName )
      else:
         #   ( not bgroupName and not agroupName ) or
         #   ( bgroupName == groupName and not agroupName ) or
         #   ( agroupName == groupName and not bgroupName )
         self.groupNamesInSeq.append( groupName )

      self.groupToRolePrivilege[ groupName ] = (
         role, privilege )
      # write out to Sysdb configuration
      self.toSysdb()

   def delete( self, groupName, role, privilege ):
      if role:
         if ( role, privilege ) != self.groupToRolePrivilege.get( groupName,
                                                                  ( None, None ) ):
            t0( "Given role and privilege, do not match any existing entry" )
            return
      if ( groupName not in self.groupNamesInSeq and
           groupName not in self.groupToRolePrivilege ):
         return
      if groupName in self.groupNamesInSeq:
         del self.groupNamesInSeq[ self.groupNamesInSeq.index( groupName ) ]
      if groupName in self.groupToRolePrivilege:
         del self.groupToRolePrivilege[ groupName ]
      # write out to Sysdb configurtion
      self.toSysdb()

   def fromSysdb( self ):
      for v in self.sysdbGroupPolicy.values():
         self.groupNamesInSeq.append( v.group )
         self.groupToRolePrivilege[ v.group ] = ( v.role, v.privilege )

   def toSysdb( self ):
      self.sysdbGroupPolicy.clear()
      for groupName in self.groupNamesInSeq:
         val = self.groupToRolePrivilege.get( groupName )
         if not val:
            continue
         rP = Tac.Value( "Ldap::GroupRolePrivilege", group=groupName,
                         role=val[ 0 ], privilege=val[ 1 ] )
         self.sysdbGroupPolicy.enq( rP )

def enterGroupPolicyMode( mode, args ):
   pName = args[ "<POLICYNAME>" ]
   childMode = mode.childMode( LdapConfigCli.GroupPolicyConfigMode,
                               pName=pName )
   mode.session_.gotoChildMode( childMode )

def noGroupPolicyMode( mode, args ):
   pName = args[ "<POLICYNAME>" ]
   del gv.ldapConfig.groupPolicy[ pName ]
   mode.groupPolicy_ = None

# -----------------------------------------------------------------------------------
# (config-mgmt-ldap-group-policy)# [ no | default ] group <GROUPNAME> role <ROLENAME>
# [ privilege <PRIVLEVEL> ]
# -----------------------------------------------------------------------------------
def enterGroupRoleMode( mode, args ):
   groupName = args[ '<GROUPNAME>' ]
   role = args[ '<ROLENAME>' ]
   privilege = args.get( '<PRIVLEVEL>', DEFAULT_ROLE_PRIVLEVEL )
   groupPolicy = gv.ldapConfig.groupPolicy.get( mode.pName, None )
   bgroupName = args.get( '<BGROUPNAME>', None )
   agroupName = args.get( '<AGROUPNAME>', None )
   if not groupPolicy:
      t0( "Did not find group policy for ", mode.pName )
      return
   else:
      t0( groupName, role, privilege, bgroupName, agroupName )
      if mode.groupPolicy_.exists( groupName, role, privilege ):
         t0( "Matching entry already exists" )
         return
      else:
         mode.groupPolicy_.delete( groupName, None, None )
         mode.groupPolicy_.insert( groupName, role, privilege, bgroupName,
                                   agroupName )

def noGroupRoleMode( mode, args ):
   groupName = args[ '<GROUPNAME>' ]
   role = args.get( '<ROLENAME>', None )
   privilege = args.get( '<PRIVLEVEL>', DEFAULT_ROLE_PRIVLEVEL )
   mode.groupPolicy_.delete( groupName, role, privilege )

# -----------------------------------------------------------------------------------
# (config-mgmt-ldap)# [ no | default ] server defaults
# and
# (config-mgmt-ldap)# [ no | default ] server host <HOSTNAME> [ ( vrf <VRFNAME> ) ]
#   [ ( port <PORT> ) ]
# -----------------------------------------------------------------------------------
def enterServerMode( mode, args ):
   hostname = args.get( '<HOSTNAME>' )
   label = 'defaults'
   if hostname:
      vrf = args.get( 'VRF', DEFAULT_VRF )
      port = args.get( '<PORT>', LdapGroup.DEFAULT_LDAP_PORT )
      t0( 'setHost hostname:', hostname, "port:", port, 'vrf:', vrf )
      HostnameCli.resolveHostname( mode, hostname, doWarn=True )
      if isinstance( vrf, list ):
         assert len( vrf ) == 1
         vrf = vrf[ 0 ]
      assert vrf != ''
      if isinstance( port, list ):
         assert len( port ) == 1
         port = port[ 0 ]

      host = ldapHost( mode, hostname, port,
                       Tac.Value( "L3::VrfName", vrf ), create=True )

      label = host.spec
      host.serverConfig = ( str( label ), )
   else:
      gv.ldapConfig.defaultConfig = ( label, )
   childMode = mode.childMode( LdapConfigCli.ServerConfigMode,
                               label=label )
   mode.session_.gotoChildMode( childMode )

def noServerMode( mode, args ):
   t0( "Set all attributes to default values" )
   hostname = args.get( '<HOSTNAME>' )
   hosts = gv.ldapConfig.host
   defaults = args.get( 'defaults' )
   if defaults:
      if gv.ldapConfig.defaultConfig:
         defaultServerDefaults( gv.ldapConfig.defaultConfig )
   elif hostname:
      vrf = args.get( 'VRF', DEFAULT_VRF )
      port = args.get( '<PORT>', LdapGroup.DEFAULT_LDAP_PORT )
      if isinstance( vrf, list ):
         assert len( vrf ) == 1
         vrf = vrf[ 0 ]
      assert vrf != ''
      if isinstance( port, list ):
         assert len( port ) == 1
         port = port[ 0 ]

      t0( 'noHost hostname:', hostname, "vrf: ", vrf, "port:", port )
      spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                        acctPort=0, vrf=vrf, protocol=hostProtocol.protoLdap )
      if spec in hosts:
         del hosts[ spec ]
      else:
         if mode.session_.interactive_:
            warningMessage = f"LDAP host {hostname} with port {port} not found"
            mode.addWarning( warningMessage )
   else:
      # Delete all hosts since no hostname was specified
      hosts.clear()

def hostStr( spec ):
   name = f'{spec.hostname}/{spec.port}'
   if spec.vrf != DEFAULT_VRF:
      name += f' vrf {spec.vrf}'
   return name

def showLdap( mode, args ):
   ldapModel = LdapModel.ShowLdapModel()
   for host in gv.ldapConfig.host.values():
      stat = gv.ldapStatus.counter.get( host.spec )
      if stat:
         s = LdapModel.LdapStatsModel()
         s._index = host.index  # pylint: disable=protected-access
         s.bindRequests = stat.bindRequests
         s.bindFails = stat.bindFails
         s.bindSuccesses = stat.bindSuccesses
         s.bindTimeouts = stat.bindTimeouts
         s.searchRequests = stat.searchRequests
         s.searchFails = stat.searchFails
         s.searchSuccesses = stat.searchSuccesses
         s.searchTimeouts = stat.searchTimeouts
         ldapModel.ldapServers[ hostStr( host.spec ) ] = s
      ldapStatusModel = LdapModel.LdapStatusModel()
      ldapStatusModel._index = host.index  # pylint: disable=protected-access
      ldapStatusModel.fipsStatus = gv.ldapStatus.fips
      ldapModel.ldapStatus[ hostStr( host.spec ) ] = ldapStatusModel

   for k in sorted( gv.aaaConfig.hostgroup ):
      g = gv.aaaConfig.hostgroup[ k ]
      if g.groupType == 'ldap':
         serverGroupDisplay = AaaCliLib.getCliDisplayFromGroup( g.groupType )
         serverGroupName = g.name
         sgModel = LdapModel.ServerGroupModel()
         sgModel.serverGroup = serverGroupDisplay
         for m in g.member.values():
            si = LdapModel.ServerInfoModel()
            si.hostname = m.spec.hostname
            si.port = m.spec.port
            sgModel.members.append( si )
         ldapModel.groups[ serverGroupName ] = sgModel

   ldapModel.lastCounterClearTime = \
      Ark.switchTimeToUtc( gv.ldapStatus.lastClearTime )
   return ldapModel

def clearCounters( mode, args ):
   Intf.Log.logClearCounters( "ldap" )
   gv.ldapCounterConfig.clearCounterRequestTime = Tac.now()
   try:
      Tac.waitFor( lambda: gv.ldapStatus.lastClearTime >=
                   gv.ldapCounterConfig.clearCounterRequestTime,
                   description='LDAP clear counter request to complete',
                   warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      mode.addWarning(
         "LDAP counters may not have been reset yet" )

def Plugin( entityManager ):
   gv.aaaConfig = LazyMount.mount( entityManager, "security/aaa/config",
                                   "Aaa::Config", "r" )
   gv.ldapConfig = ConfigMount.mount( entityManager,
                                      'security/aaa/ldap/config',
                                      'Ldap::Config', 'w' )
   gv.ldapStatus = LazyMount.mount( entityManager,
                                    Cell.path( 'security/aaa/ldap/status' ),
                                    'Ldap::Status', 'r' )
   gv.ldapCounterConfig = LazyMount.mount( entityManager,
                                           "security/aaa/ldap/counterConfig",
                                           "AaaPlugin::CounterConfig", "w" )
   gv.sslConfig = LazyMount.mount( entityManager,
                                   'mgmt/security/ssl/config',
                                   'Mgmt::Security::Ssl::Config', 'r' )
   gv.localConfig = LazyMount.mount( entityManager,
                                     'security/aaa/local/config',
                                     'LocalUser::Config', 'r' )
