# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
from typing import Optional

import CliGlobal
import Intf.IntfRange
import IntfRangePlugin.EthIntf
import IntfRangePlugin.XcvrSlot
import MultiRangeRule
from TypeFuture import TacLazyType
from XcvrLib import getXcvrSlotName

import CliPlugin.XcvrAllStatusDir
from CliPlugin.EthIntfCli import EthPhyIntf
from CliPlugin.XcvrCliLib import ( getAllIntfsWrapper, getAuxiliaryRenamedAsEthernet,
                                   IntfMapping )

gv = CliGlobal.CliGlobal( xcvrStatusDir=None )

# -------------------------------------------------------------------------------
# TAC Types used in commands
# -------------------------------------------------------------------------------
MediaType = TacLazyType( 'Xcvr::MediaType' )

# -------------------------------------------------------------------------------
# Constants used in the CLI commands
# -------------------------------------------------------------------------------
BOOSTER_CHANNEL = 1
PRE_AMP_CHANNEL = 2

# -------------------------------------------------------------------------------
# Helper methods used in the CLI commands
# -------------------------------------------------------------------------------

class mimicIntfMapping( IntfMapping ):
   """
   Imitation intfMapping class with support for auxiliary slots.
   Generates necessary interface mappings for line system CLI to use xcvrStatus CLI
   handler. Renames channels /1 and /2 to booster and pre-amp
   """
   # BUG1003829 - We should not juggle between "Ethernet" and "Auxiliary"
   # nomenclature for auxiliary slots.

   def __init__( self, slotNames: set[ str ] ) -> None:
      self.laneMap = {}
      self.intfMap = {}
      self.intfNameOverride = {}
      for slot in slotNames:
         self.intfNameOverride[ f"{slot}/1" ] = "Booster"
         self.intfNameOverride[ f"{slot}/2" ] = "Pre-amp"
         self.intfMap[ f"{slot}/1" ] = { f"{slot}/1": ( 0, ) }
         self.intfMap[ f"{slot}/2" ] = { f"{slot}/2": ( 1, ) }
      # intfRange does not provide information about which interfaces are Auxiliary
      # or not. Cross-check with xcvr status to figure out and change all names
      # which should say "Auxiliary"
      for xcvr in gv.xcvrStatusDir.xcvrStatus.values():
         if "Auxiliary" in xcvr.name:
            etName1 = xcvr.name.replace( "Auxiliary", "Ethernet" ) + "/1"
            etName2 = xcvr.name.replace( "Auxiliary", "Ethernet" ) + "/2"
            if etName1 in self.intfMap and etName2 in self.intfMap:
               self.intfMap[ f"{xcvr.name}/1" ] = self.intfMap[ etName1 ]
               self.intfMap[ f"{xcvr.name}/2" ] = self.intfMap[ etName2 ]
               del self.intfMap[ etName1 ]
               del self.intfMap[ etName2 ]
      self.laneMap = self.intfMap

def lineSystemIntfsInPortRange( mode,
                                portRange: Optional[ MultiRangeRule.IntfList ] ) \
      -> MultiRangeRule.IntfList:
   """
   Finds the range of 'interfaces' with line systems inserted within the port range
   (including auxiliary). If portRange is None, returns all ports with line systems
   """
   intfList = []
   if not portRange:
      # If no `port X` token is supplied, use all interfaces as port range
      _, intfList = getAllIntfsWrapper( mode, None, None )
      intfList = { getXcvrSlotName( intfName ) for intfName in intfList }
      intfList |= set( getAuxiliaryRenamedAsEthernet() )
   else:
      intfList = Intf.IntfRange.intfListFromCanonical( [ portRange ] )
   interfacesWithLineSystems = getPortsWithLineSystem( intfList )
   # BUG1003829 - we should not be juggling between "ethernet" and "auxiliary".
   # IntfRange items are in form "EthernetX". This is also the case for the
   # auxiliary slots included within!
   applicablePorts = getIntfRangeFromPortRange( interfacesWithLineSystems )
   return applicablePorts

def getPortsWithLineSystem( portList: list[ str ] ) -> set[ str ]:
   ''' CLI helper method to get all port names that have AMP-ZR modules in them
   Params:
   --------
   portRange : list[ str ]
      The user requested port names

   Returns:
   ---------
   intfWithLineSystem : [str]
      A list of all the intf names with AMP-ZR in them.
   '''
   intfWithLineSystem = set()
   # Iterate over all port names requested and check if the xcvr slots have
   # AMP-ZR modules in them.
   for port in portList:
      # Get corresponding xcvr name.
      ethSlotName = port.replace( "port", "Ethernet" )
      auxSlotName = ethSlotName.replace( "Ethernet", "Auxiliary" )
      # Get the xcvrStatus for the xcvr.
      xcvrStatus = gv.xcvrStatusDir.xcvrStatus.get( ethSlotName )
      if not xcvrStatus:
         # If no status returned, try for auxilary slot
         xcvrStatus = gv.xcvrStatusDir.xcvrStatus.get( auxSlotName )
      if xcvrStatus and xcvrStatus.mediaType == MediaType.xcvrAmpZr:
         intfWithLineSystem.add( ethSlotName )
   return intfWithLineSystem

def endpointFunc( partial, hasSub=False ):
   ''' Not clear what this method does, but basing this on
   CliParser/MultiRangeRule.
   '''
   genericRange = MultiRangeRule.GenericRangeType( lambda: ( 0, 0xFFFFFFFF ), "" )
   return genericRange.endpoints( partial, hasSub )

def getIntfRangeFromPortRange( portRange: tuple[ str, ... ] ) \
      -> Optional[ MultiRangeRule.IntfList ]:
   ''' CLI helper to get interface names list from
   POLS ports.
   Params:
   --------
   portRange : [str]
      A list of all the port names with AMP-ZR in them.

   Returns:
   ---------
   intfNamesRange : MultiRangeRule.IntfList | None
      All interfaces with AMP-ZR in them.
   '''
   # We need to get the interface names of these ports.
   ethPhyRangeIntf = IntfRangePlugin.EthIntf.EthPhyRangeIntfType()
   # Register it as type EthPhyIntf (as in the type returned for interfaces)
   ethPhyRangeIntf.registerEthPhyIntfClass( EthPhyIntf )
   # IrPathSet should be created for every interface on the port.
   irPathSet = MultiRangeRule.IrPathSet()
   if not portRange:
      # MultiRangeRule crashes if empty, best to avoid.
      return None
   for port in portRange:
      # Find the linecard and slot number, this will be used as
      # an input to irPathList.
      portNums = [ int( s ) for s in re.findall( r'\d+', port ) ]
      # We know that line-system slots have 2 channels; booster and a pre-amp.
      for lineSystemInterface in [ BOOSTER_CHANNEL, PRE_AMP_CHANNEL ]:
         # Create a new list.
         irPathList = portNums[ : ]
         # We need to create irPathList for both the channels.
         irPathList.append( lineSystemInterface )
         # Finally add it to irPathSet
         irPathSet.add( MultiRangeRule.IrPath( endpointFunc, irPathList ) )
   return MultiRangeRule.IntfList( ethPhyRangeIntf, irPathSet )

def subPrefixToPort( portName: str ) -> str:
   """
   Changes an "Ethernet" or "Auxiliary" prefix to "Port"
   """
   return re.sub( r"Ethernet|Auxiliary", "Port ", getXcvrSlotName( portName ) )

# -------------------------------------------------------------------------------
# Plugin
# -------------------------------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
