# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Arnet
import CliGlobal
from CliDynamicSymbol import CliDynamicPlugin
from XcvrLib import getLaneId, getXcvrSlotName

import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrCliLib import getAllIntfsWrapper
from CliPlugin.XcvrShowDomHandler import _getIntfXcvrDomInfo
from CliPlugin.XcvrShowDomModel import InterfacesTransceiverDom

LSCL = CliDynamicPlugin( 'LineSystemCliLib' )
gv = CliGlobal.CliGlobal( xcvrStatusDir=None )

# -------------------------------------------------------------------------------
# show line system [PORT X] dom [thresholds]
# -------------------------------------------------------------------------------

def showLineSystemDomHelper( mode, intf=None, mod=None, outputFmt=None,
                             thresholds: bool = False ) -> InterfacesTransceiverDom:
   model = InterfacesTransceiverDom( _printFmt=outputFmt, _detailed=thresholds )
   ( _, intfNames ) = getAllIntfsWrapper( mode, intf, mod, includeAuxiliary=True )

   if not intfNames:
      return model

   slotNames = { getXcvrSlotName( intf ) for intf in intfNames }
   # Get the laneMappings for all the active interfaces
   intfMapping = LSCL.mimicIntfMapping( slotNames )
   intfLaneMapping = intfMapping.intfMap

   for intfToDisplay in sorted( intfMapping.laneMap, key=LSCL.subPrefixToPort ):
      for intfName in Arnet.sortIntf( intfMapping.laneMap[ intfToDisplay ] ):
         channels = intfLaneMapping[ intfToDisplay ][ intfName ]
         xS = gv.xcvrStatusDir.xcvrStatus.get( getXcvrSlotName( intfToDisplay ) )

         if not xS.xcvrMgrCapabilities.domDynamicValues:
            continue

         if xS.presence != "xcvrPresent":
            # Skip transceivers that aren't present
            continue

         if intfName in model.interfaces or not channels:
            continue

         laneId = getLaneId( intfName )
         laneId = laneId - 1 if laneId else laneId

         portModel = _getIntfXcvrDomInfo( xS, laneId, channels, thresholds )
         portModel.displayName = intfName

         # pylint:disable-msg=protected-access
         model.interfaces[ intfName ] = portModel
         model._interfacesOrder.append( intfName )

   return model

def showLineSystemDom( mode, args ) -> InterfacesTransceiverDom:
   # Get the ports requested by the user.
   portRange = args.get( 'PORTRANGE' )
   # Check if the user requested threshold data
   thresholds = 'thresholds' in args
   # Interface range that will be passed into the _showInterfacesXcvrDom method.
   intfRangeLineSys = LSCL.lineSystemIntfsInPortRange( mode, portRange )
   if not intfRangeLineSys:
      return InterfacesTransceiverDom( _printFmt='default', _detailed=thresholds )

   return showLineSystemDomHelper( mode, intfRangeLineSys, outputFmt='default',
                                   thresholds=thresholds )

# -------------------------------------------------------------------------------
# Plugin
# -------------------------------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
