# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import sortIntf
from CliModel import Model, Int, List, Dict
from Intf.IntfRange import intfListToCanonical
from IntfModels import Interface
import TableOutput

class PoolStatus( Model ):
   maximum = Int( help="Maximum logical ports in the pool" )
   free = Int( help="Number of free logical ports in the pool" )
   configured = Int( help="Number of configured ports in the pool" )
   interfaces = List( valueType=Interface, help="Member interfaces" )

   def toModel( self, poolStatus ):
      self.maximum = poolStatus.numTotalPorts
      self.free = poolStatus.numFreePorts
      self.configured = poolStatus.numConfiguredPorts
      self.interfaces = list( poolStatus.memberIntf )

      return self

class PoolStatuses( Model ):
   pools = Dict( keyType=str, valueType=PoolStatus,
                 help="Mapping between a logical port pool's name and its status" )

   def render( self ):
      if not self.pools:
         return

      table = TableOutput.createTable(
            ( 'Pool', 'Max', 'Free', 'Configured', 'Interfaces' ),
            tableWidth=100 )
      numberFmt = TableOutput.Format()
      intfFmt = TableOutput.Format( justify='left', wrap=True,
                                    minWidth=74, maxWidth=74 )
      table.formatColumns( numberFmt, numberFmt, numberFmt, numberFmt, intfFmt )

      for poolId in sortIntf( self.pools ):
         poolStatus = self.pools[ poolId ]
         table.newRow( poolId,
                       poolStatus.maximum,
                       poolStatus.free,
                       poolStatus.configured,
                       ','.join( intfListToCanonical(
                                    poolStatus.interfaces,
                                    noHoleRange=True,
                                    verifyIntfHoles=True ) ) )
      print( table.output().rstrip( '\n' ) )
