# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliUtil
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.MacsecCli import (
   config,
   hwMacsecStatus,
   trapConfig
)
import Tac
from TypeFuture import TacLazyType

PostStage = TacLazyType( "Macsec::POSTStage" )
PtpBypass = TacLazyType( "Macsec::PtpBypass" )
TrapFeatureName = TacLazyType( 'Arnet::TrapFeatureName' )
KeyDerivationPadding = TacLazyType( "Macsec::KeyDerivationPadding" )

macsecDynamicSubmode = CliDynamicPlugin( "MacsecCli" )

# Handler to delete a mac security profile from an interface
#--------------------------------------------------------------------------------
# [ no | default ] mac security profile PROFILE_NAME
#--------------------------------------------------------------------------------
def delMacsecIntf( intfId ):
   # Get the intfConfig object.
   if intfId in config.intfConfig:
      intfConfig = config.intfConfig[ intfId ]
   else:
      # Nothing to do.
      return
   profileName = intfConfig.profileName
   if not profileName:
      # Nothing to do.
      return
   if ( profileName in config.profile ) and \
         ( intfId in config.profile[ profileName ].intf ):
      del config.profile[ profileName ].intf[ intfId ]
   # Delete the interface config.
   del config.intfConfig[ intfId ]

# Handler to configure key derivation padding in a macsec profile
#--------------------------------------------------------------------------------
# "( no | default ) key derivation padding ( prepend | append )"
#--------------------------------------------------------------------------------
def addKeyDerivationPadding( mode, args ):
   paddingType = None
   if 'append' in args:
      paddingType = KeyDerivationPadding.append
   else:
      paddingType = KeyDerivationPadding.prepend

   mode.profile().keyDerivationPadding = paddingType

# Handler to associate a mac security profile for an interface
#--------------------------------------------------------------------------------
# mac security profile PROFILE_NAME
#--------------------------------------------------------------------------------
def configMacsecIntfCmd( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   # If an intfConfig does not exist yet, create it now.
   intfId = mode.intf.name
   newProfileExists = True
   intfConfig = config.intfConfig.newMember( intfId )
   # Get the associated profile.
   if profileName not in config.profile:
      mode.addWarning( f'MAC security profile {profileName} does not exist.' )
      newProfileExists = False
   # If profile associated with the interface has not changed then there is
   # nothing to do.
   if intfConfig.profileName != profileName:
      
      # Disassociate the interface from its old profile.
      if intfConfig.profileName in config.profile:
         del config.profile[ intfConfig.profileName ].intf[ intfId ]
      
      # Associate the interface with its new profile.
      if newProfileExists:
         config.profile[ profileName ].intf[ intfId ] = True
      # Update the interface.
      intfConfig.profileName = profileName

def noMacsecIntfCmd( mode, args ):
   # Disassociate the interface from its current profile.
   intfId = mode.intf.name
   delMacsecIntf( intfId )

# Handler to reset MKA counters
#--------------------------------------------------------------------------------
# clear mac security mka counters
#--------------------------------------------------------------------------------
def doClearMkaCounters( mode, args ):
   config.clearMkaCounters += 1

#--------------------------------------------------------------------------------
# Handler for "license LICENSEENAME AUTHKEYSTR", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def setMacsecLicense( mode, args ):
   authKey = int( args[ 'AUTHKEYSTR' ], 16 )
   config.licenseConfig = Tac.Value( "Macsec::LicenseConfig",
                                     licenseeName=args[ 'LICENSEENAME' ],
                                     authKey=authKey )

#--------------------------------------------------------------------------------
# Handler for "( no | default ) license ", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def noMacsecLicense( mode, args=None ):
   config.licenseConfig = Tac.Value( "Macsec::LicenseConfig" )

#-------------------------------------------------------------------------
# Handler for "delay protection" command, in "config-mac-sec" mode
#-------------------------------------------------------------------------
def handleDelayProtection( mode, args ):
   config.delayProtection = True

#--------------------------------------------------------------------------------
# Handler for "[no | default] delay protection" command, in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def handleNoDelayProtection( mode, args=None ):
   config.delayProtection = config.delayProtectionDefault

#--------------------------------------------------------------------------------
# Handler for "fips restrictions", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def handleFipsRestrictions( mode, args ):
   def checkInternalPostSliceResetRequired( statusDir ):
      for hwSlice in statusDir:
         sliceStatus = statusDir[ hwSlice ].status
         hwCapabilities = statusDir[ hwSlice ].hwCapabilities
         for intfName in sliceStatus:
            if sliceStatus[ intfName ].hwPostStatus == PostStage.skipped and \
               hwCapabilities[ intfName ].internalPostSupported and \
               hwCapabilities[ intfName ].postSliceResetRequired:
               return True
      return False

   if checkInternalPostSliceResetRequired( hwMacsecStatus ):
      prompt = (
         "WARNING\r\n"
         "This command will reset the data plane in order to become FIPS compliant "
         "and will impact traffic forwarding.\r\n"
         "Proceed (y/[N])? " )
      if not BasicCliUtil.confirm( mode, prompt, answerForReturn=False ):
         return
   config.fipsRestrictions = True

#--------------------------------------------------------------------------------
# Handler for "[ no | default ] fips restrictions", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def handleNoFipsRestrictions( mode, args=None ):
   config.fipsRestrictions = config.fipsRestrictionsDefault

#--------------------------------------------------------------------------------
# Handler for "l2-protocol ethernet-flow-control ( encrypt | bypass )",
# in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def handleGlobalEthFlowControlEncrypt( mode, args ):
   if 'bypass' in args:
      # Transmit Ethernet Flow Control frames without protection
      mode = Tac.Type( "Macsec::EthFlowControl" ).bypass
   else:
      # Transmit/Receive Ethernet Flow Control frames with protection
      assert 'encrypt' in args
      mode = Tac.Type( "Macsec::EthFlowControl" ).encrypt
   config.ethFlowControl = mode

#-----------------------------------------------------------------------------------
# Handler for "[ no | default ] l2-protocol ethernet-flow-control
#                                                             ( encrypt | bypass )",
# in "config-mac-sec" mode
#-----------------------------------------------------------------------------------
def handleGlobalDefaultEthFlowControlEncrypt( mode, args ):
   config.ethFlowControl = Tac.Type( "Macsec::EthFlowControl" ).platformDefault

def setEapolAttributes( dmac=None, ethType=None ):
   eapolAttr = Tac.Value( "Macsec::EapolAttributes",
                          destinationMac=config.eapolAttr.destinationMac,
                          etherType=config.eapolAttr.etherType )
   if dmac:
      eapolAttr.destinationMac = dmac
   if ethType:
      eapolAttr.etherType = ethType
   config.eapolAttr = eapolAttr

#-----------------------------------------------------------------------------------
# Handler for "[ no | default ] eapol mac destination MAC", in "config-mac-sec" mode
#-----------------------------------------------------------------------------------
def setMacsecEapolDestinationMac( mode, args ):
   dmac = args.get( 'MAC', config.eapolAttr.destinationMacDefault )
   if dmac == config.eapolAttr.destinationMacDefault:
      trapConfig.features.remove( TrapFeatureName.eapol )
   else:
      trapConfig.features.add( TrapFeatureName.eapol )
   setEapolAttributes( dmac=dmac )

#--------------------------------------------------------------------------------
# Handler for "[ no | default ] eapol ethertype TYPE", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def setMacsecEapolEtherType( mode, args ):
   ethType = \
         int( args.get( 'TYPE', config.eapolAttr.etherTypeDefault ) )
   setEapolAttributes( ethType=ethType )

#--------------------------------------------------------------------------------
# Handler for "ptp bypass", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def handlePtpBypass( mode, args ):
   config.bypassPtp = PtpBypass.ptpBypass

#--------------------------------------------------------------------------------
# Handler for "no ptp bypass", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def handleNoPtpBypass( mode, args ):
   config.bypassPtp = PtpBypass.ptpEncrypt

#--------------------------------------------------------------------------------
# Handler for "default ptp bypass", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def handleDefaultPtpBypass( mode, args ):
   config.bypassPtp = PtpBypass.ptpPlatformDefault

#--------------------------------------------------------------------------------
# Handler for "shutdown", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def shutDownMacsec( mode, args ):
   config.shutdown = True

#--------------------------------------------------------------------------------
# Handler for "[ no | default ] shutdown", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def noShutDownMacsec( mode, args=None ):
   config.shutdown = config.shutdownDefault

def gotoMacsecMode( mode, args ):
   childMode = mode.childMode( macsecDynamicSubmode.MacsecMode )
   mode.session_.gotoChildMode( childMode )

def noMacsecMode( mode, args ):
   # Reset all the commands at mac security mode.
   # Resetting license first will make sure no ConfigSm reactors are fired further.
   noMacsecLicense( mode )
   handleNoDelayProtection( mode )
   handleNoFipsRestrictions( mode )
   handleGlobalDefaultEthFlowControlEncrypt( mode, args=None )
   setMacsecEapolDestinationMac( mode, args )
   setMacsecEapolEtherType( mode, args )
   handleDefaultPtpBypass( mode, args=None )
   noShutDownMacsec( mode )
   # Delete all configured macsec profiles.
   config.profile.clear()

#--------------------------------------------------------------------------------
# Handler for "[ no | default ] profile PROFILENAME", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
def noMacsecProfileMode( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   del config.profile[ profileName ]
