#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.FlowTrackingCliLib import isMirrorOnDropAgentRunning
from CliPlugin.MirrorOnDropCliLib import (
   ReportKey,
   getDropReasonStr,
   getInterfaces,
)
from CliPlugin.MirrorOnDropCliLib import DropReasonCliMap
import LazyMount
import Tac
import SmashLazyMount
from TypeFuture import TacLazyType

ModModel = CliDynamicPlugin( 'MirrorOnDropModel' )
AddressFamily = TacLazyType( 'Arnet::AddressFamily' )

USEC_IN_SEC = 1000000

entityManager = None
activeAgentDir = None
modHwConfig = None
ethIntfStatusDir = None
emtStatus = None
reportStatus = None
collectorStatus = None
dropExportStatus = None
trackerPacketStatistics = {}

def getTrackerPacketStatistics( trackerName ):
   if trackerName not in trackerPacketStatistics:
      packetStatistics = SmashLazyMount.mount( entityManager,
                                               f"dropExport/counters/{trackerName}",
                                               "DropExport::PacketStatistics",
                                               SmashLazyMount.mountInfo( 'reader' ) )
      trackerPacketStatistics[ trackerName ] = packetStatistics
   return trackerPacketStatistics[ trackerName ]

def showFlowEntry( groupModel, emtEntry, reportEntry, utcTime,
                   intfName, isInstalled ):
   flowModel = ModModel.FlowModel()
   flowModel.srcAddr = reportEntry.key.srcIp
   flowModel.dstAddr = reportEntry.key.dstIp
   flowModel.srcPort = reportEntry.key.srcPort
   flowModel.dstPort = reportEntry.key.dstPort
   flowModel.ipProtocolNumber = reportEntry.key.ipProto
   flowModel.dropReason = getDropReasonStr( reportEntry.report.reason )
   try:
      flowModel.ipProtocol = Tac.typeNode( "Arnet::IpProtocolNumber" ).\
                                attr( reportEntry.key.ipProto ).name
   except AttributeError:
      # for unknown protocols, we will use ipProtocolNumber so its safe to skip.
      pass

   flowModel.age = int( round( utcTime - reportEntry.report.timestamp /
                               USEC_IN_SEC ) )
   flowModel.ingressIntf = intfName
   flowModel.emtStatus = isInstalled
   groupModel.flows.append( flowModel )

def showFlowTable( mode, args ):
   flowTableModel = ModModel.ModShowModel()
   flowTableModel.modelTypeIs( 'flowTable' )
   flowTableModel.running = isMirrorOnDropAgentRunning( entityManager,
                                                        activeAgentDir )
   if not flowTableModel.running:
      return flowTableModel

   ipv4Flows = ModModel.GroupModel()
   ipv6Flows = ModModel.GroupModel()

   trackerFilter = args.get( 'TRACKER_NAME' )
   for trName in modHwConfig.hwFtConfig:
      if trackerFilter and trackerFilter != trName:
         continue

      trackerModel = ModModel.TrackerModel()
      flowTableModel.trackers[ trName ] = trackerModel
      trackerModel.groups[ 'IPv4' ] = ipv4Flows
      trackerModel.groups[ 'IPv6' ] = ipv6Flows

      entryColl = emtStatus if modHwConfig.trapSuppression else reportStatus
      for key, emtEntry in entryColl.entry.items():
         reportKey = ReportKey( key.ipProto, key.srcIp, key.dstIp, key.srcPort,
                                key.dstPort )
         report = reportStatus.entry.get( reportKey )
         if not report:
            # if entry is not found in reportStatus, it was aged out and we ignore
            # this EMT entry
            continue

         if key.srcIp.af == AddressFamily.ipv4:
            group = ipv4Flows
         else:
            group = ipv6Flows

         if hasattr( emtEntry.key, "intfId" ):
            isInstalled = "installed" if emtEntry.isInstalled else "not installed"
            showFlowEntry( group, emtEntry, report, Tac.utcNow(),
                           emtEntry.key.intfId, isInstalled )
         else:
            isInstalled = "not installed"
            for intf in emtEntry.intfs.values():
               showFlowEntry( group, emtEntry, report, Tac.utcNow(),
                              intf, isInstalled )

      trackerModel.numFlows = len( ipv4Flows.flows ) + len( ipv6Flows.flows )
      # MOD only supports one tracker. All flows are global so display once only
      break

   return flowTableModel

def showInterfaceCounters( mode, args ):
   intfModel = ModModel.ModShowModel()
   intfModel.modelTypeIs( 'interface' )
   intfModel.running = isMirrorOnDropAgentRunning( entityManager,
                                                   activeAgentDir )
   if not intfModel.running:
      return intfModel

   trackerFilter = args.get( 'TRACKER_NAME' )
   for trName in modHwConfig.hwFtConfig:
      if trackerFilter and trackerFilter != trName:
         continue

      trackerModel = ModModel.TrackerModel()
      intfModel.trackers[ trName ] = trackerModel
      trackerModel.numFlows = len( reportStatus.entry )

      interfaces = getInterfaces( ethIntfStatusDir )
      intfCount = { intf: 0 for intf in interfaces }

      entryColl = emtStatus if modHwConfig.trapSuppression else reportStatus
      for emtEntry in entryColl.entry.values():
         if hasattr( emtEntry, "intfs" ):
            for intfId in emtEntry.intfs.values():
               intfCount[ intfId ] += 1
         else:
            intfCount[ emtEntry.key.intfId ] += 1

      for intf in interfaces:
         if intfCount[ intf ]:
            trackerModel.intfs[ intf ] = intfCount[ intf ]

      # MOD only supports one tracker. All flows are global so display once only
      break

   return intfModel

def showCollector( exporterModel, expName, colName, collector ):
   CollectorKey = TacLazyType( 'DropExport::Collector::Key' )
   key = CollectorKey( expName, colName )
   entry = collectorStatus.entry.get( key )
   if entry:
      collectorModel = ModModel.CollectorModel()
      collectorModel.addrAndPort = ModModel.IpGenericAddrAndPort()
      collectorModel.addrAndPort.ip = IpGenAddr( str( collector.ip ) )
      collectorModel.addrAndPort.port = collector.port
      collectorModel.byteCount = entry.bytes
      collectorModel.packetCount = entry.stop + entry.start
      collectorModel.stopCount = entry.stop
      collectorModel.startCount = entry.start
      exporterModel.collectors[ colName ] = collectorModel

def showCollectors( exporterModel, expName, exporter ):
   # Ipv4 Collectors
   for colName, collector in exporter.collectorIpAndPort.items():
      showCollector( exporterModel, expName, colName, collector )

   # Ipv6 Collectors
   for colName, collector in exporter.collectorIp6AndPort.items():
      showCollector( exporterModel, expName, colName, collector )

def showExporterCounters( mode, args ):
   expModel = ModModel.ModShowModel()
   expModel.modelTypeIs( 'collector' )
   expModel.running = isMirrorOnDropAgentRunning( entityManager, activeAgentDir )
   if not expModel.running:
      return expModel

   trackerFilter = args.get( 'TRACKER_NAME' )
   exporterFilter = args.get( 'EXPORTER_NAME' )
   for trName, tracker in modHwConfig.hwFtConfig.items():
      if trackerFilter and trackerFilter != trName:
         continue

      trackerModel = ModModel.TrackerModel()
      expModel.trackers[ trName ] = trackerModel
      trackerModel.numFlows = len( reportStatus.entry )

      for expName, exporter in tracker.expConfig.items():
         if exporterFilter and exporterFilter != expName:
            continue

         exporterModel = ModModel.ExporterModel()
         trackerModel.exporters[ expName ] = exporterModel

         showCollectors( exporterModel, expName, exporter )

         if exporterFilter:
            break

      if trackerFilter:
         break

   return expModel

def showDropReasonCounters( mode, args ):
   countersModel = ModModel.ModShowModel()
   countersModel.modelTypeIs( 'dropReason' )
   countersModel.running = \
      isMirrorOnDropAgentRunning( entityManager, activeAgentDir )
   if not countersModel.running:
      return countersModel

   trackerFilter = args.get( 'TRACKER_NAME', None )
   interfaceFilter = args.get( 'INTERFACE_NAME', None )
   if interfaceFilter:
      interfaceFilter = interfaceFilter.name

   for trName in modHwConfig.hwFtConfig.keys():
      if trackerFilter and trackerFilter != trName:
         continue

      # check that a status object for this tracker exists in DropExport::Status.
      # If it does not exist, then the DropExport  would not have created
      # a statistics object for this tracker at path 'dropExport/counters/<tracker>'
      if trName not in dropExportStatus.trackerStatus.keys():
         continue

      packetStatistics = getTrackerPacketStatistics( trName )
      trackerModel = ModModel.TrackerModel()
      trackerModel.numFlows = 0
      countersModel.trackers[ trName ] = trackerModel

      for intf, intfCounters in packetStatistics.droppedPktCounters.items():
         if interfaceFilter and interfaceFilter != intf:
            continue
         counters = ModModel.DropReasonModel()
         trackerModel.dropReasonCounters[ intf ] = counters
         for reason, reasonInfo in DropReasonCliMap.items():
            dropCount = intfCounters.droppedCount[ reason ]
            if dropCount:
               cliModelKey = reasonInfo.dropReasonCliKey
               counters.__setattr__( cliModelKey, dropCount )
   return countersModel

def showCounters( mode, args ):
   if 'interface' in args:
      return showInterfaceCounters( mode, args )
   else:
      return showExporterCounters( mode, args )

def Plugin( em ):
   global entityManager
   global modHwConfig
   global activeAgentDir
   global ethIntfStatusDir
   global emtStatus
   global reportStatus
   global collectorStatus
   global dropExportStatus

   entityManager = em
   modHwConfig = LazyMount.mount( em, 'hardware/flowtracking/config/mirrorOnDrop',
                                  'HwFlowTracking::Config', 'r' )
   dropExportStatus = LazyMount.mount( em, 'dropExport/status',
                                       'DropExport::Status', 'r' )
   activeAgentDir = LazyMount.mount( em, 'flowtracking/activeAgent',
                                     'Tac::Dir', 'ri' )
   ethIntfStatusDir = LazyMount.mount( em, 'interface/status/eth/intf',
                                       'Interface::EthIntfStatusDir', 'r' )
   emtStatus = SmashLazyMount.mount( em, 'strataMirror/mirrorOnDrop/emtStatus',
                                     'DropExport::TrapSuppression::Status',
                                     SmashLazyMount.mountInfo( 'reader' ) )

   reportStatusPath = 'dropExport/reportStatus'
   collectorStatusPath = 'dropExport/collectorStatus'
   collectorStatusType = 'DropExport::Collector::Status'
   reportStatusType = 'DropExport::Report::Status'
   reportStatus = SmashLazyMount.mount( em, reportStatusPath,
                                        reportStatusType,
                                        SmashLazyMount.mountInfo( 'reader' ) )
   collectorStatus = SmashLazyMount.mount( em,
                                           collectorStatusPath,
                                           collectorStatusType,
                                           SmashLazyMount.mountInfo( 'reader' ) )
