#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from BgpLib import vpnAfTypeMapInv
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.RoutingBgpCli import (
   afModeExtensionHook,
   bgpNeighborConfig,
   configForVrf,
   BgpCmdBaseClass,
)
from CliPlugin.RoutingBgpInstanceCli import (
   nhResRibsConfigHelper,
   removeNexthopSelfReceivedVpnRoutesConfig,
   resetBgpAfModeConfig,
)
from CliPlugin.RoutingBgpNeighborCli import (
   neighborAfActivateHelper,
   PeerAfStateEnum,
)
from IpLibTypes import ProtocolAgentModelType

RoutingBgpCliHandler = CliDynamicPlugin( "RoutingBgpCliHandler" )

def setMplsNexthopSelfSourceInterface( mode, args ):
   peer = args.get( 'PEER' )
   if peer:
      config = bgpNeighborConfig( peer, mode.vrfName )
   else:
      config = configForVrf( mode.vrfName )

   tacEncapType = 'encapMpls'
   if mode.addrFamily == 'vpn-ipv4':
      config.afVpnV4Encap = tacEncapType
      config.nexthopSelfSrcIntfVpnV4 = args[ 'INTF' ].name
      if not peer:
         config.afVpnV4Srv6Locator = config.afVpnV4Srv6LocatorDefault
      else:
         config.afVpnV4EncapPresent = True
   elif mode.addrFamily == 'vpn-ipv6':
      config.afVpnV6Encap = tacEncapType
      config.nexthopSelfSrcIntfVpnV6 = args[ 'INTF' ].name
      if not peer:
         config.afVpnV6Srv6Locator = config.afVpnV6Srv6LocatorDefault
      else:
         config.afVpnV6EncapPresent = True
   else:
      raise ValueError()

def setSrv6EncapLocator( mode, args ):
   config = configForVrf( mode.vrfName )

   tacEncapType = 'encapSrv6'
   if mode.addrFamily == 'vpn-ipv4':
      config.afVpnV4Encap = tacEncapType
      config.nexthopSelfSrcIntfVpnV4 = config.nexthopSelfSrcIntfVpnV4Default
      config.afVpnV4Srv6Locator = args[ 'LOCATOR' ]
   elif mode.addrFamily == 'vpn-ipv6':
      config.afVpnV6Encap = tacEncapType
      config.nexthopSelfSrcIntfVpnV6 = config.nexthopSelfSrcIntfVpnV6Default
      config.afVpnV6Srv6Locator = args[ 'LOCATOR' ]
   else:
      raise ValueError()

def setEncapsulation( mode, args ):
   if 'mpls' in args:
      setMplsNexthopSelfSourceInterface( mode, args )
   elif 'ipv6' in args:
      setSrv6EncapLocator( mode, args )
   else:
      raise ValueError()

def setDefaultEncapsulation( mode, args ):
   peer = args.get( 'PEER' )
   if peer:
      config = bgpNeighborConfig( peer, mode.vrfName, create=False )
   else:
      config = configForVrf( mode.vrfName )

   if not config:
      return

   if mode.addrFamily == 'vpn-ipv4':
      config.afVpnV4Encap = config.afVpnV4EncapDefault
      config.nexthopSelfSrcIntfVpnV4 = config.nexthopSelfSrcIntfVpnV4Default
      if not peer:
         config.afVpnV4Srv6Locator = config.afVpnV4Srv6LocatorDefault
      else:
         config.afVpnV4EncapPresent = False
   elif mode.addrFamily == 'vpn-ipv6':
      config.afVpnV6Encap = config.afVpnV6EncapDefault
      config.nexthopSelfSrcIntfVpnV6 = config.nexthopSelfSrcIntfVpnV6Default
      if not peer:
         config.afVpnV6Srv6Locator = config.afVpnV6Srv6LocatorDefault
      else:
         config.afVpnV6EncapPresent = False
   else:
      raise ValueError()

   if peer:
      RoutingBgpCliHandler.delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

# class MplsVpnNeighborActDeact( CliCommand.CliCommandClass ):
def MplsVpnNeighborActDeact_handler( mode, args ):
   if 'activate' in args:
      neighborAfActivateHelper( mode, args[ 'PEER' ], PeerAfStateEnum.afActive )
   else:
      neighborAfActivateHelper( mode, args[ 'PEER' ], PeerAfStateEnum.afInactive )

def MplsVpnNeighborActDeact_noHandler( mode, args ):
   neighborAfActivateHelper( mode, args[ 'PEER' ], PeerAfStateEnum.afInactive )

def MplsVpnNeighborActDeact_defaultHandler( mode, args ):
   neighborAfActivateHelper( mode, args[ 'PEER' ], PeerAfStateEnum.afDefault )

# class MplsVpnAfModeCmd( CliCommand.CliCommandClass ):
def MplsVpnAfModeCmd_handler( mode, args ):
   vpnAddrFamily = args[ 'AF' ]
   if getEffectiveProtocolModel( mode ) == ProtocolAgentModelType.ribd:
      mode.addWarning( "Routing protocols model multi-agent must be "
                        "configured for MPLS-VPN address-family" )
   assert vpnAddrFamily in afModeExtensionHook.afModeExtensions()
   childMode = mode.childMode(
      afModeExtensionHook.afModeExtension[ vpnAddrFamily ] )
   mode.session_.gotoChildMode( childMode )

def MplsVpnAfModeCmd_noOrDefaultHandler( mode, args ):
   bgpConfig = configForVrf( mode.vrfName )
   resetBgpAfModeConfig( bgpConfig, args[ 'AF' ], mode.vrfName )

# class NeighborDefaultEncapMplsNhselfSourceInterfaceCmd( BgpCmdBaseClass ):
def NeighborDefaultEncapMplsNhselfSourceInterfaceCmd_handleNormal( mode, args ):
   setEncapsulation( mode, args )

def NeighborDefaultEncapMplsNhselfSourceInterfaceCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   setDefaultEncapsulation( mode, args )

def NeighborDefaultEncapMplsNhselfSourceInterfaceCmd_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      NeighborDefaultEncapMplsNhselfSourceInterfaceCmd_handleNormal,
      NeighborDefaultEncapMplsNhselfSourceInterfaceCmd_handleNoOrDefault,
      mode,
      args )

def NeighborDefaultEncapMplsNhselfSourceInterfaceCmd_noOrDefaultHandler(
      mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      NeighborDefaultEncapMplsNhselfSourceInterfaceCmd_handleNoOrDefault,
      mode,
      args )

# class NeighborDefaultEncapSegmentRoutingIpv6Cmd( BgpCmdBaseClass ):
def NeighborDefaultEncapSegmentRoutingIpv6Cmd_handleNormal( mode, args ):
   setEncapsulation( mode, args )

def NeighborDefaultEncapSegmentRoutingIpv6Cmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   setDefaultEncapsulation( mode, args )

def NeighborDefaultEncapSegmentRoutingIpv6Cmd_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      NeighborDefaultEncapSegmentRoutingIpv6Cmd_handleNormal,
      NeighborDefaultEncapSegmentRoutingIpv6Cmd_handleNoOrDefault,
      mode,
      args )

def NeighborDefaultEncapSegmentRoutingIpv6Cmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      NeighborDefaultEncapSegmentRoutingIpv6Cmd_handleNoOrDefault,
      mode,
      args )

# class NeighborPeerEncapMplsNhselfSourceInterfaceCmd( BgpCmdBaseClass ):
def NeighborPeerEncapMplsNhselfSourceInterfaceCmd_handleNormal( mode, args ):
   setEncapsulation( mode, args )

def NeighborPeerEncapMplsNhselfSourceInterfaceCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   setDefaultEncapsulation( mode, args )

def NeighborPeerEncapMplsNhselfSourceInterfaceCmd_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      NeighborPeerEncapMplsNhselfSourceInterfaceCmd_handleNormal,
      NeighborPeerEncapMplsNhselfSourceInterfaceCmd_handleNoOrDefault,
      mode,
      args )

def NeighborPeerEncapMplsNhselfSourceInterfaceCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      NeighborPeerEncapMplsNhselfSourceInterfaceCmd_handleNoOrDefault,
      mode,
      args )

# class MplsLabelAllocationDisabledCmd( CliCommand.CliCommandClass ):
def setLabelAllocation( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.labelAllocationDisabledAfVpnV4 = 'isTrue'
   elif mode.addrFamily == 'vpn-ipv6':
      config.labelAllocationDisabledAfVpnV6 = 'isTrue'
   else:
      raise ValueError()

def noLabelAllocation( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.labelAllocationDisabledAfVpnV4 = 'isInvalid'
   elif mode.addrFamily == 'vpn-ipv6':
      config.labelAllocationDisabledAfVpnV6 = 'isInvalid'
   else:
      raise ValueError()

def MplsLabelAllocationDisabledCmd_handler( mode, args ):
   setLabelAllocation( mode, args )

def MplsLabelAllocationDisabledCmd_noOrDefaultHandler( mode, args ):
   noLabelAllocation( mode, args )

# class ResolutionVpnRibCmd( CliCommand.CliCommandClass ):
def setNexthopResolutionInVrfUnicastRib( mode, args, value ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.nexthopResolutionVrfUniRibAfVpnV4 = value
   elif mode.addrFamily == 'vpn-ipv6':
      config.nexthopResolutionVrfUniRibAfVpnV6 = value
   else:
      raise ValueError()

def setNexthopResolutionInUdpTunnel( mode, args, value ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.nexthopResolutionUdpTunnelAfVpnV4 = value
   elif mode.addrFamily == 'vpn-ipv6':
      config.nexthopResolutionUdpTunnelAfVpnV6 = value
   else:
      raise ValueError()

def ResolutionVpnRibCmd_handler( mode, args ):
   if 'vrf-unicast-rib' in args:
      setNexthopResolutionInUdpTunnel( mode, args, 'isInvalid' )
      setNexthopResolutionInVrfUnicastRib( mode, args, 'isTrue' )
   elif 'udp-tunnel' in args:
      setNexthopResolutionInVrfUnicastRib( mode, args, 'isInvalid' )
      setNexthopResolutionInUdpTunnel( mode, args, 'isTrue' )
   else:
      nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )

def ResolutionVpnRibCmd_noOrDefaultHandler( mode, args ):
   if 'vrf-unicast-rib' in args:
      setNexthopResolutionInVrfUnicastRib( mode, args, 'isInvalid' )
   elif 'udp-tunnel' in args:
      setNexthopResolutionInUdpTunnel( mode, args, 'isInvalid' )
   else:
      nhResRibsConfigHelper( mode.addrFamily, args, no=True, vrfName=mode.vrfName )

# class SetMplsVpnDomainIdCmd( CliCommand.CliCommandClass ):
def setVpnDomainId( mode, vpnDomainId ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.domainIdVpnV4 = vpnDomainId
   elif mode.addrFamily == 'vpn-ipv6':
      config.domainIdVpnV6 = vpnDomainId
   else:
      raise ValueError()

def noVpnDomainId( mode ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      defaultVal = config.domainIdVpnV4Default
      config.domainIdVpnV4 = defaultVal
   elif mode.addrFamily == 'vpn-ipv6':
      defaultVal = config.domainIdVpnV6Default
      config.domainIdVpnV6 = defaultVal
   else:
      raise ValueError()

def SetMplsVpnDomainIdCmd_handler( mode, args ):
   setVpnDomainId( mode, args[ 'DOMAIN_ID' ] )

def SetMplsVpnDomainIdCmd_noOrDefaultHandler( mode, args ):
   noVpnDomainId( mode )

# class NeighborDefaultEncapMplsNhselfReceivedVpnv4RoutesCmd( CliCommandClass ):
def setMplsNexthopSelfReceivedVpnRoutes( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily in [ 'vpn-ipv4', 'vpn-ipv6' ]:
      afiSafi = vpnAfTypeMapInv[ mode.addrFamily ]
      config.nexthopSelfLocalLabelAlloc[ afiSafi ] = True
   else:
      raise ValueError()

def noMplsNexthopSelfReceivedVpnRoutes( mode, args ):
   config = configForVrf( mode.vrfName )
   removeNexthopSelfReceivedVpnRoutesConfig(
      config, mode.addrFamily, vrfName=mode.vrfName )

def NeighborDefaultEncapMplsNhselfReceivedVpnv4RoutesCmd_handler( mode, args ):
   setMplsNexthopSelfReceivedVpnRoutes( mode, args )

def NeighborDefaultEncapMplsNhselfReceivedVpnv4RoutesCmd_noOrDefaultHandler(
      mode, args ):
   noMplsNexthopSelfReceivedVpnRoutes( mode, args )

# class NeighborDefaultEncapMplsNhselfReceivedVpnv6RoutesCmd( CliCommandClass ):
def NeighborDefaultEncapMplsNhselfReceivedVpnv6RoutesCmd_handler( mode, args ):
   setMplsNexthopSelfReceivedVpnRoutes( mode, args )

def NeighborDefaultEncapMplsNhselfReceivedVpnv6RoutesCmd_noOrDefaultHandler(
      mode, args ):
   noMplsNexthopSelfReceivedVpnRoutes( mode, args )
