# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import CliDynamicPlugin
from CliModel import ( DeferredModel, Submodel, Str, Int, Float, List, Enum,
                       GeneratorDict, Bool )
from ArnetModel import Ip4Address, IpGenericPrefix
from CliPlugin.BgpCliModels import (
   EncapLabel,
   REASON_NOT_BEST_LIST,
)
from CliPlugin.VrfCli import generateVrfCliModel

BgpVpnModels = CliDynamicPlugin( "BgpVpnModels" )
BgpVpnRouteDetailEntry = BgpVpnModels.BgpVpnRouteDetailEntry
BgpVpnRouteTypeEntry = BgpVpnModels.BgpVpnRouteTypeEntry

class MplsVpnRouteASPathEntry( DeferredModel ):
   asPath = Str( optional=True, help='AS path string (if absent,  '
                 'then the route was originated locally)' )
   asPathType = Enum( values=( 'Internal', 'External', 'Confed-External', 'Local',
                               'Invalid' ),
                      help='AS path type: '
                            'Internal - originated by I-BGP '
                            'External - originated by E-BGP '
                            'Confed-External - originated by a E-BGP confederation '
                            'Local - originated locally '
                            'Invalid - AS path is invalid' )

class MplsVpnRouteDetailEntry( BgpVpnRouteDetailEntry ):
   aggregator = Str( optional=True, help='Aggregator of the route' )
   rxPathId = Int( optional=True, help='Received path ID of this route' )
   txPathId = Int( optional=True, help='Advertised path ID for this route' )
   seqNumber = Int( optional=True, help='Route sequence number' )
   pendingTimeToAdv = Float( optional=True, help='Timestamp of route advertisement' )
   # the redistibution protocol strings here need to be kept in sync
   # with the strings returned by
   # gated/gated-ctk/src/bgp/bgp_dget_route.c:redist_proto_str()
   # note these are usually transformed by maybeCamelize()
   redistributionProtocol = Enum( values=( 'Connected', 'Static', 'Ospf3', 'Ospf',
                                           'Rip', 'Is-Is', 'unknown' ),
                                  optional=True,
                                  help='Protocol from which route got '
                                     'redistributed into BGP' )
   labelStack = List( optional=True, valueType=int,
                      help='MPLS label stack information' )
   remoteLabel = Submodel( optional=True, valueType=EncapLabel,
                           help='Remote MPLS label' )
   localLabel = Submodel( optional=True, valueType=EncapLabel,
                          help='Local MPLS label' )
   remoteSrv6Sid = Submodel( optional=True, valueType=EncapLabel,
                             help='Remote SRv6 SID' )
   localSrv6Sid = Submodel( optional=True, valueType=EncapLabel,
                            help='Local SRv6 SID' )
   domainPath = List( optional=True, valueType=str, help="Domain path attribute" )

class MplsVpnRoutePath( DeferredModel ):
   nextHop = Str( optional=True, help='Route next hop address' )
   asPathEntry = Submodel( valueType=MplsVpnRouteASPathEntry,
                           help='AS path information' )
   importedMplsVpnPathRd = Str( optional=True,
                                help='Imported MPLS VPN path RouteDistinguisher' )
   med = Int( optional=True, help='Multi Exit Discriminator for the route' )
   localPreference = Int( optional=True, help='I-BGP Local preference indicator' )
   routeType = Submodel( valueType=BgpVpnRouteTypeEntry, help='Route type' )
   weight = Int( optional=True, help='Weight for the route' )
   tag = Int( optional=True, help='Tag for the route' )
   timestamp = Int( optional=True,
                    help="UTC seconds since epoch when the route was received."
                          "Only returned with detail output" )
   routeDetail = Submodel( valueType=MplsVpnRouteDetailEntry, optional=True,
                           help='Route details' )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )

class MplsVpnRouteKeyDetail( DeferredModel ):
   rd = Str( optional=True, help='Route distinguisher' )
   nlriType = Enum( values=( "IPv4 prefix", "IPv6 prefix" ), help='NLRI type' )
   ipGenPrefix = IpGenericPrefix( optional=True, help='IPv4 or IPv6 address prefix' )

class MplsVpnRouteEntry( DeferredModel ):
   __revision__ = 2
   routeKeyDetail = Submodel( valueType=MplsVpnRouteKeyDetail, help='NLRI details' )
   totalPaths = Int( optional=True, help='Total number of paths for this route' )
   routePriority = Enum( values=( 'low', 'medium', 'high' ),
                         help="Route processing priority",
                         optional=True )
   mplsVpnRoutePaths = List( valueType=MplsVpnRoutePath,
                             help='List of BGP MPLS VPN route ECMP paths' )

class MplsVpnRoutes( DeferredModel ):
   __revision__ = 2
   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Int( help='Autonomous System Number' )
   lastProcessedSeqNum = Int( optional=True,
                              help='Last route sequence number acknowledged' )
   currentSeqNum = Int( optional=True, help='Current route sequence number' )
   _detail = Bool( optional=True, help='Detailed output is requested' )
   _advRoutes = Bool( optional=True, help='Advertised routes output is requested' )
   mplsVpnRoutes = GeneratorDict( keyType=str, valueType=MplsVpnRouteEntry,
         help='Dictionary of BGP MPLS VPN route entries indexed by the route key' )

# Wrap MplsVpnRoutes model under "vrfs" key
mplsVpnRoutesVrfModel = generateVrfCliModel( MplsVpnRoutes,
                                             "MPLS VPN route summary" )
