#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import CliDynamicPlugin
from CliModel import cliPrinted
from CliPlugin.ArBgpCli import (
      ArBgpAsyncCliCommand
      )
from CliPlugin.RoutingBgpShowCli import (
   ArBgpShowOutput,
   BgpVrfRoutingContextDefaultOnly,
   summaryVrfModel,
)

BgpVpnCliHandler = CliDynamicPlugin( "BgpVpnCliHandler" )
VpnCliHelperCommand = BgpVpnCliHandler.VpnCliHelperCommand
MplsVpnCliModels = CliDynamicPlugin( "MplsVpnCliModels" )
ArBgpCliHandler = CliDynamicPlugin( "ArBgpCliHandler" )

class MplsVpnCliHelperCommand( VpnCliHelperCommand ):
   def __init__( self, mode, command, **kwargs ):
      super().__init__(
         mode, command, None, **kwargs )

   def run( self, **kwargs ):
      super().run( **kwargs )
      return cliPrinted( MplsVpnCliModels.mplsVpnRoutesVrfModel )

@BgpVrfRoutingContextDefaultOnly( cliModel=MplsVpnCliModels.mplsVpnRoutesVrfModel )
def doShowBgpMplsVpn( mode, **kwargs ):
   setKwargs = {}
   VpnCliHelperCommand.flattenArgs( kwargs, setKwargs )

   if setKwargs.get( "prefixValue", None ) is not None:
      # Set a distinct prefix value to not collide with Evpn prefix value
      setKwargs[ "mplsVpnPrefixValue" ] = setKwargs.pop( "prefixValue" )

   if setKwargs.get( "internal", False ):
      prefix = setKwargs.pop( "mplsVpnPrefixValue", None )
      ArBgpAsyncCliCommand( mode, "show bgp <af> internal",
                            prefix=prefix, **setKwargs ).run()
      return cliPrinted( MplsVpnCliModels.mplsVpnRoutesVrfModel )

   return MplsVpnCliHelperCommand( mode, 'show bgp vpn-ipvx', **setKwargs ).run()

#-------------------------------------------------------------------------------
# Helper Methods
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpMplsVpn', arBgpModeOnly=True )
@BgpVrfRoutingContextDefaultOnly( cliModel=MplsVpnCliModels.mplsVpnRoutesVrfModel )
def _doShowBgpMplsVpn( mode, nlriType=None, peerAddrValue=None,
                      bgpRouteTypeValue=None,
                      commValuesAndExact=None,
                      extCommValuesAndExact=None,
                      largeCommValuesAndExact=None,
                      prefixValue=None, longerPrefixes=False,
                      rdValue=None, nexthopValue=None,
                      detail=None, internal=None ):
   assert False, "This method is abstract. " + \
      "If this was reached then something went wrong with MplsVpnHelperCli loading"


# For 'show bgp vpn-ipv4/vpn-ipv6 summary' cmd, we are using ArBgp
# implementation of 'show ip[v6] bgp summary' cmd.
@ArBgpShowOutput( 'doShowBgpMplsVpnV4Summary', arBgpModeOnly=True )
@BgpVrfRoutingContextDefaultOnly( cliModel=summaryVrfModel )
def doShowBgpMplsVpnV4Summary( mode ):
   return ArBgpCliHandler.doShowIpBgpSummaryAcrImpl( mode,
                                                    nlriAfiSafi='ipv4MplsVpn' )

@ArBgpShowOutput( 'doShowBgpMplsVpnV6Summary', arBgpModeOnly=True )
@BgpVrfRoutingContextDefaultOnly( cliModel=summaryVrfModel )
def doShowBgpMplsVpnV6Summary( mode ):
   return ArBgpCliHandler.doShowIpBgpSummaryAcrImpl( mode,
                                                    nlriAfiSafi='ipv6MplsVpn' )

def handlerMplsVpnSummaryCommand( mode, args ):
   if args[ 'AF' ] == 'vpn-ipv4':
      return doShowBgpMplsVpnV4Summary( mode )
   else:
      return doShowBgpMplsVpnV6Summary( mode )

def handlerVpnV4( mode, args ):
   return _doShowBgpMplsVpn( mode,
            nlriType='mplsVpnIpv4Unicast',
            peerAddrValue=args.get( "PEER" ),
            bgpRouteTypeValue=args.get( "route-type" ),
            commValuesAndExact=args.get( "communityValuesAndExact" ),
            extCommValuesAndExact=args.get( "EXT_COMMUNITIES" ),
            largeCommValuesAndExact=args.get( "LARGE_COMMUNITIES" ),
            prefixValue=args.get( "PREFIX" ),
            longerPrefixes=args.get( "longer-prefixes" ),
            rdValue=args.get( "RD_VAL" ),
            nexthopValue=args.get( "nexthop-addr" ),
            detail=args.get( "detail" ),
            internal=args.get( "internal" ) )

def handlerVpnV6( mode, args ):
   return _doShowBgpMplsVpn( mode,
            nlriType='mplsVpnIpv6Unicast',
            peerAddrValue=args.get( "PEER" ),
            bgpRouteTypeValue=args.get( "route-type" ),
            commValuesAndExact=args.get( "communityValuesAndExact" ),
            extCommValuesAndExact=args.get( "EXT_COMMUNITIES" ),
            largeCommValuesAndExact=args.get( "LARGE_COMMUNITIES" ),
            prefixValue=args.get( "PREFIX" ),
            longerPrefixes=args.get( "longer-prefixes" ),
            rdValue=args.get( "RD_VAL" ),
            nexthopValue=args.get( "nexthop-addr" ),
            detail=args.get( "detail" ),
            internal=args.get( "internal" ) )

@ArBgpShowOutput( 'doShowBgpDebugPolicy', arBgpModeOnly=True )
def policyDebugMplsVpnHandler( mode, args ):
   nlriAttrs = {}
   # rd for export routes will be populated on Model creation This is because we
   # do not know the rd value of the path prior to evaluation and do not have access
   # to it at this point
   if not args.get( 'export' ):
      assert args.get( 'RD' )
      nlriAttrs = { 'rd' : args.get( 'RD' ) }
   # Get the NLRI type
   # The 'nlriType' value should correspond to the enum values in BgpNlriType.tac
   if 'vpn-ipv4' in args:
      nlriType = 'mplsVpnIpv4Unicast'
      # The IPv4 prefix is stored as a string so processing the default prefix
      # works as intended
      assert args.get( 'PREFIX' )
      nlriAttrs[ 'ipPrefix' ] = args[ 'PREFIX' ]
   elif 'vpn-ipv6' in args:
      nlriType = 'mplsVpnIpv6Unicast'
      # The IPv6 prefix is stored as an IpGenAddr so we need to check if the attr
      # "is not None" to correctly process the default prefix ::/0
      assert args.get( 'PREFIX6' ) is not None
      nlriAttrs[ 'ipPrefix' ] = args[ 'PREFIX6' ].stringValue()
   elif args.get( 'SAFI', '' ) == 'unicast':
      # This section is needed for the unicast main function on the import POA
      nlriType, nlriAttrs = ArBgpCliHandler.getUnicastNlriHelper( args )
   else:
      assert False, "Unexpected afisafi"
   return ArBgpCliHandler.policyDebugHandlerCommon( mode, args, nlriType, nlriAttrs )
