#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import (
   Ip4Address,
   IpGenericAddress,
)
from CliDynamicSymbol import CliDynamicPlugin
from CliModel import (
   DeferredModel,
   Enum,
   GeneratorDict,
   Int,
   List,
   Str,
   Submodel,
   Model,
   Dict,
)
from CliPlugin.BgpCliModels import (
   PmsiTunnel,
   REASON_NOT_BEST_LIST,
)
from CliPlugin.VrfCli import generateVrfCliModel
import TableOutput

BgpVpnModels = CliDynamicPlugin( "BgpVpnModels" )
BgpVpnRouteDetailEntry = BgpVpnModels.BgpVpnRouteDetailEntry
BgpVpnRouteTypeEntry = BgpVpnModels.BgpVpnRouteTypeEntry

class MvpnRouteAsPathEntry( DeferredModel ):
   asPath = Str( optional=True, help='AS path string (if absent,  '
                 'then the route was originated locally)' )
   asPathType = Enum( values=( 'Internal', 'External', 'Confed-External', 'Local',
                               'Invalid' ),
                      help='AS path type: '
                            'Internal: originated by I-BGP, '
                            'External: originated by E-BGP, '
                            'Confed-External: originated by a E-BGP confederation, '
                            'Local: originated locally, '
                            'Invalid: AS path is invalid' )

class MvpnRouteDetailEntry( BgpVpnRouteDetailEntry ):
   pmsiTunnel = Submodel( optional=True, valueType=PmsiTunnel,
                          help="P-Multicast Serivce Interface Tunnel Attribute" )

class MvpnRoutePath( DeferredModel ):
   nextHop = Str( optional=True, help='Route next hop address' )
   asPathEntry = Submodel( valueType=MvpnRouteAsPathEntry,
                           help='AS path information' )
   med = Int( optional=True, help='Multi Exit Discriminator for the route' )
   localPreference = Int( optional=True, help='I-BGP Local preference indicator' )
   routeType = Submodel( valueType=BgpVpnRouteTypeEntry, help='Route type' )
   weight = Int( optional=True, help='Weight for the route' )
   tag = Int( optional=True, help='Tag for the route' )
   timestamp = Int( optional=True,
                    help="UTC seconds since epoch when the route was received."
                          "Only returned with detail output" )
   routeDetail = Submodel( valueType=MvpnRouteDetailEntry, optional=True,
                           help='Route details' )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )

class MvpnRouteKeyDetail( DeferredModel ):
   rd = Str( optional=True, help='Route distinguisher. For locally sourced '
            'source-tree and shared-tree routes, set to the RD of the UMH' )
   nlriType = Enum( values=( 'intraAsIpmsi', 'selectivePmsi', 'sourceActive',
                             'sharedTree', 'sourceTree' ), help='NLRI type' )
   originator = IpGenericAddress( optional=True,
                                  help='The originating routers IP address in '
                                  'intra-as-ipmsi routes' )
   sourceAs = Str( optional=True, help='The originating routers Autonomous System '
                   'in shared-tree and source-tree routes' )
   source = IpGenericAddress( optional=True, help='Multicast source or RP address' )
   group = IpGenericAddress( optional=True, help='Multicast group address' )

class MvpnRouteEntry( DeferredModel ):
   routeKeyDetail = Submodel( valueType=MvpnRouteKeyDetail, help='NLRI details' )
   totalPaths = Int( optional=True, help='Total number of paths for this route' )
   mvpnRoutePaths = List( valueType=MvpnRoutePath,
                          help='List of MVPN route ECMP paths' )

class MvpnRoutes( DeferredModel ):
   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Int( help='Autonomous System Number' )
   mvpnRoutes = GeneratorDict( keyType=str, valueType=MvpnRouteEntry,
         help='Dictionary of MVPN route entries indexed by the route key' )

# Wrap MvpnRoutes model under "vrfs" key
MvpnRoutesVrfModel = generateVrfCliModel( MvpnRoutes, 'MVPN route summary' )

class MvpnRsvpSessionIdLeafSet( Model ):
   leafNodes = List( valueType=IpGenericAddress,
                     help='A set of leaf nodes in a VRF with the same SessionId' )

class MvpnVrfRsvpLeafTunnelIdModel( Model ):
   tunnelIds = Dict( keyType=int, valueType=MvpnRsvpSessionIdLeafSet,
                     help='Dictionary of MVPN RSVP leaf sets indexed by tunnel ID' )

class MvpnVrfRsvpLeafStatus( Model ):
   vrfId = Int( help='VRF ID' )
   extendedTunnel = IpGenericAddress( help='Extended tunnel ID' )
   tunnelProfile = Str( optional=True, help='Tunnel profile name' )
   p2mpIds = Dict( keyType=int, valueType=MvpnVrfRsvpLeafTunnelIdModel,
                   help='Dictionary of MVPN RSVP tunnel models indexed by P2MP ID' )

class MvpnRsvpLeafStatus( Model ):
   vrfs = Dict( keyType=str, valueType=MvpnVrfRsvpLeafStatus,
                help='Dictionary of MVPN RSVP VRF leaf status indexed by VRF ID' )

   def render( self ):
      for vrfName, vrfStatus in self.vrfs.items():
         # Print out VRF information
         print( f'VRF: {vrfName}, VRF ID: {vrfStatus.vrfId}, Tunnel profile: '
                f'{vrfStatus.tunnelProfile}' )

         # Print out table header
         headers = ( 'P2MP ID', 'Tunnel ID', 'Ext Tunnel ID', 'Leaf' )
         table = TableOutput.createTable( headers )

         # Set up table format
         formatRight = TableOutput.Format( justify='right' )
         formatRight.padLimitIs( True )
         formatLeft = TableOutput.Format( justify='left' )
         formatLeft.padLimitIs( True )
         formatLeaf = TableOutput.Format( justify='left', maxWidth=35, wrap=True )
         table.formatColumns( formatRight, formatRight, formatLeft, formatLeaf )

         extTunnelId = vrfStatus.extendedTunnel
         for p2mpId, tunnelIdModel in vrfStatus.p2mpIds.items():
            for tunnelId, leafSet in tunnelIdModel.tunnelIds.items():
               # Print out this leaf set
               leafList = [ l.stringValue for l in leafSet.leafNodes ]
               leaves = ", ".join( leafList )
               table.newRow( p2mpId, tunnelId, extTunnelId, leaves )
               print( table.output() )
