# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import CliMatcher
import CliCommand

# Importing module instead of global variables as importing
# global variables objects can give incorrect values.
# When we import an object from another package, we copy a
# reference to the object into our package
# By setting the value in the other package, we overwrite the
# key in that package with a new value, but we leave our
# reference pointing to the old one. Python recommendation link:
# https://docs.python.org/3/faq/programming.html#how-do-i-share-
# global-variables-across-modules
from CliPlugin import NetworkTopologyService

from CliPlugin.NetworkTopologyService import matcherPhysicalTopology
from CliPlugin.NetworkTopologyService import nodeNetwork
from CliMode.Topology import TopologyMode

class TopologyConfigMode( TopologyMode,
                          BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'cvx-topology'

   def __init__( self, parent, session ):
      TopologyMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

matcherSwitch = CliMatcher.KeywordMatcher( 'switch',
      helpdesc='Network device' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Network device interface' )
matcherNeighbor = CliMatcher.KeywordMatcher( 'neighbor',
      helpdesc='Neighbor device' )
matcherNeighborInterface = CliMatcher.KeywordMatcher( 'neighbor-interface',
      helpdesc='Neighbor interface' )

# --------------------------------------------------------------------------------
# network physical-topology switch SWITCH interface INTERFACE neighbor
# NEIGHBOR-HOST [ neighbor-interface NEIGHBOR-INTERFACE ]
# --------------------------------------------------------------------------------

def getSwitchNames( mode ):
   return [ h.hostname for h in NetworkTopologyService.status.host.values() ] \
         if NetworkTopologyService.status else []

def getNeighborNames( mode, context ):
   if not NetworkTopologyService.status:
      return []
   switch = context.sharedResult[ 'SWITCH' ]
   return [ h.hostname for h in NetworkTopologyService.status.host.values()
         if h.hostname != switch ]

def getSwitchIntf( mode, context ):
   if not NetworkTopologyService.status:
      return []
   switch = context.sharedResult[ 'SWITCH' ]
   hosts = NetworkTopologyService.status.hostsByHostname.get( switch )
   if not hosts or len( hosts.host ) != 1:
      return []
   host = next( iter( hosts.host.values() ) )
   return list( host.port ) if host is not None else []

def getNeighborIntf( mode, context ):
   if not NetworkTopologyService.status:
      return []
   neighbor = context.sharedResult[ 'NEIGHBOR-HOST' ]
   hosts = NetworkTopologyService.status.hostsByHostname.get( neighbor )
   if not hosts or len( hosts.host ) != 1:
      return []
   host = next( iter( hosts.host.values() ) )
   return list( host.port ) if host is not None else []

# Pattern to verify physical interface for fixed and modular switches
switchIntfPattern = r'(^[Et|et][a-zA-Z]+)((\d+)|(\d+\/\d+)|(\d+\/\d+\/\d+))$'

class NetworkPhysicalTopologyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''network physical-topology switch SWITCH interface INTERFACE neighbor 
               NEIGHBOR-HOST [ neighbor-interface NEIGHBOR-INTERFACE ]'''
   noOrDefaultSyntax = syntax

   _matcherSwitchObj = CliMatcher.DynamicNameMatcher( getSwitchNames,
      helpdesc='Switch name', pattern=r'.+' )
   _matcherNeighborObj = CliMatcher.DynamicNameMatcher( getNeighborNames,
      pattern=r'.+', passContext=True, helpdesc='Neighbor name' )
   _matcherIntfObj = CliMatcher.DynamicNameMatcher( getSwitchIntf,
      passContext=True, helpname='Ethernet', helpdesc='Switch interface name',
      pattern=switchIntfPattern )
   _matcherNeighborIntfObj = CliMatcher.DynamicNameMatcher( getNeighborIntf,
      passContext=True, helpdesc='Neighbor interface name' )

   data = {
      'network': nodeNetwork,
      'physical-topology': matcherPhysicalTopology,
      'switch': matcherSwitch,
      'SWITCH': CliCommand.Node( _matcherSwitchObj, storeSharedResult=True ),
      'interface': matcherInterface,
      'INTERFACE': CliCommand.Node( _matcherIntfObj ),
      'neighbor': matcherNeighbor,
      'NEIGHBOR-HOST': CliCommand.Node( _matcherNeighborObj,
                                         storeSharedResult=True ),
      'neighbor-interface': matcherNeighborInterface,
      'NEIGHBOR-INTERFACE': CliCommand.Node( _matcherNeighborIntfObj )
   }

   handler = "NetworkTopologyServiceHandler.\
networkPhysicalTopologyConfig"

   noOrDefaultHandler = "NetworkTopologyServiceHandler.\
networkPhysicalTopologyConfigNoOrDef"

TopologyConfigMode.addCommandClass( NetworkPhysicalTopologyConfigCmd )
