# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements IP DHCP Relay configuration.
# It contains both interface-specific and non-interface-specific commands.
#-------------------------------------------------------------------------------

import Arnet
import Tac
import LazyMount
import ConfigMount
import CliCommand
import CliDynamicSymbol
from IpLibConsts import DEFAULT_VRF

dhcpRelayConfig = None
nativeRelayConfig = None

def configConflict( mode ):
   # new relay is configured, old one isn't
   if not dhcpRelayConfig.intfConfig and nativeRelayConfig.intfConfig:
      msg = ( "To configure DHCP relay first remove incompatible" 
              " configuration of the form 'ip helper-address ...'" 
              " or 'ip dhcp relay information option circuit-id ...'" 
              " or 'ip dhcp relay all-subnets'" )
      mode.addWarning( msg )
      return True
   # old relay doesn't have the alwaysOn feature
   if nativeRelayConfig.alwaysOn:
      msg = ( "To configure DHCP relay first remove incompatible"
              " configuration 'ip dhcp relay always-on'" )
      mode.addWarning( msg )
      return True
   # old relay doesn't have the smartRelay feature
   if nativeRelayConfig.smartRelayGlobal:
      msg = ( "To configure DHCP relay first remove incompatible"
              " configuration 'ip dhcp relay all-subnets'" )
      mode.addWarning( msg )
      return True
   return False

# bypasses checks while calling a function in native relay CLI
def callNativeRelay( func, args ):
   helperCli = CliDynamicSymbol.loadDynamicPlugin( "DhcpRelayHelperHandler" )
   helperCli.fromForwarder = True
   getattr( helperCli, func )( *args )
   helperCli.fromForwarder = False

def deleteIntfConfigIfDefault( mode ):
   c = dhcpRelayConfig
   i = mode.intf.name
   conf = c.intfConfig.get( i, None )
   if conf:
      keep = False
      if conf.clientAllowed:
         keep = True
      if conf.serverAllowed:
         keep = True
      if conf.circuitId != "":
         keep = True
      if not keep:
         del c.intfConfig[ i ]


#-------------------------------------------------------------------------------
# The "[no|default] ip dhcp relay [client] [server]" interface command
#-------------------------------------------------------------------------------
def setIntfDhcpRelay( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   client = 'client' in args
   server = 'server' in args
   c = dhcpRelayConfig
   i = mode.intf.name
   conf = c.intfConfig.get( i, None )
   if not conf and not no:
      if configConflict( mode ):
         return
      conf = c.intfConfig.newMember( i )
   if conf:
      if client:
         conf.clientAllowed = not no
         # update intfConfig in native relay
         for addr, on in c.serverIp.items():
            if on:
               callNativeRelay( 'setIntfDhcpRelay', ( mode, addr, no ) )
      if server:
         conf.serverAllowed = not no
      deleteIntfConfigIfDefault( mode )

#-------------------------------------------------------------------------------
# The "[no|default] ip dhcp relay circuit-id <circuit>" interface command
#-------------------------------------------------------------------------------
def setIntfCircuitId( mode, args ):
   newId = args[ 'ID' ]
   c = dhcpRelayConfig
   i = mode.intf.name
   conf = c.intfConfig.get( i, None )
   if not conf:
      if configConflict( mode ):
         return
      conf = c.intfConfig.newMember( i )
   conf.circuitId = newId
   # set circuit-id in new relay
   callNativeRelay( 'setIntfCircuitId', ( mode, args ) )

def noIntfCircuitId( mode, args):
   c = dhcpRelayConfig
   i = mode.intf.name
   conf = c.intfConfig.get( i, None )
   if conf:
      conf.circuitId = ""
      deleteIntfConfigIfDefault( mode )
      # delete circit-id in new relay
      callNativeRelay( 'noIntfCircuitId', ( mode, args ) )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay log verbose
#--------------------------------------------------------------------------------
def setDhcpRelayLogLevel( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if not no:
      msg = "Log verbose is ignored. This command is left here for backward" \
          + " compatibility reason and has no effect. To enable logging, use" \
          + " 'trace DhcpRelay enable ...' command instead."
      mode.addWarning( msg )
   dhcpRelayConfig.logVerbose = not no

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay server ADDR
#--------------------------------------------------------------------------------
def vrfIpSrcIntf( vrf, ip ):
   srcIntfId = Tac.Value( 'Arnet::IntfId', '' )
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfIpSrcIntfSrcIp", vrf,
                     Arnet.IpGenAddr( ip ), srcIntfId, Arnet.IpGenAddr( '0.0.0.0' ) )

def setDhcpRelayServer( mode, args ):
   server = args.get( 'ADDR' )
   no = CliCommand.isNoOrDefaultCmd( args )
   c = dhcpRelayConfig
   # intf option is not currently supported
   if configConflict( mode ):
      return
   if no:
      del c.serverIp[ server ]
   else:
      c.serverIp[ server ] = True
   # modify native relay's intfConfig helper-addresses here
   # would be better to use callNativeRelay(), but unable to access
   # intfconfig mode, so setting manually
   for nativeIntfName in nativeRelayConfig.intfConfig:
      nativeIntfConfig = nativeRelayConfig.intfConfig.get( nativeIntfName )
      if no:
         del nativeIntfConfig.serverIp[ vrfIpSrcIntf( \
               DEFAULT_VRF, server ) ] 
      else:
         nativeIntfConfig.serverIp[ vrfIpSrcIntf( \
               DEFAULT_VRF, server ) ] = True

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global dhcpRelayConfig
   dhcpRelayConfig = ConfigMount.mount( entityManager, "ip/dhcp/relay/config",
                                      "Ip::Dhcp::Relay::Config", "w" )

   # check for incompatible native DhcpRelay agent
   global nativeRelayConfig
   nativeRelayConfig = LazyMount.mount( entityManager, "ip/helper/dhcprelay/config",
                                        "Ip::Helper::DhcpRelay::Config", "r" )

