# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from typing import Optional

from ArPyUtils.Types import ArException
from CliPlugin import EthIntfCli
from CliPlugin import PhyNegotiationModel
from CliPlugin.PhyNegotiationCli import ( autonegStatesModelsCol,
                                          autonegTrainingModelsCol )
from TypeFuture import TacLazyType

import CliGlobal
import LazyMount
import Tac

ComponentType = TacLazyType( 'Hardware::L1Topology::ComponentType' )
CompNode = TacLazyType(
   'Hardware::L1Topology::ComponentConnectionNode' )
DescriptorHelper = TacLazyType( 'Hardware::L1Topology::DescriptorHelper' )
TraceEndpointDescriptor = TacLazyType(
   'Hardware::L1Topology::TraceEndpointDescriptor' )
AutonegPdBaseType = TacLazyType( 'Interface::Autoneg::AutonegPdBase' )
IdOps = TacLazyType( 'Hardware::L1Topology::IdentifierOperators' )
L1MountPaths = TacLazyType( 'L1::MountConstants' )
L1TopoSysdbPaths = TacLazyType( 'Hardware::L1Topology::SysdbPathConstants' )
PhyAutonegStatusDirV2 = TacLazyType( 'Interface::Autoneg::PhyAutonegStatusDirV2' )
defaultEntityAttrs = set( Tac.Type( "Tac::Entity" ).attributes )

gv = CliGlobal.CliGlobal( autonegRootDir=None,
                          autonegPdRootDir=None,
                          l1TopoDir=None,
                          l1MappingDir=None,
                          topoHelper=None )

def getComponentNodes( intf: EthIntfCli.EthPhyIntf ) -> list[ CompNode ]:
   nodes = []
   intfId = intf.name
   ethIntfMode = intf.ethIntfMode()
   # Verify ethIntfMode
   if ethIntfMode not in gv.topoHelper.getSpeedModesForIntfId( intfId ).intfMode:
      raise ArException( "No ethIntfModes found" )

   travDirection = gv.topoHelper.getDirectionForIntfId( intfId, True )
   compGroup = gv.topoHelper.getComponentGroupForIntfId( intfId, ethIntfMode, True )
   while compGroup:
      nodes.append( compGroup.component[ 0 ] )
      compGroup = gv.topoHelper.nextComponentGroup( compGroup, travDirection, True )
      travDirection = IdOps.invertDirection( travDirection )
   # Reverse the list so that they are from system to line
   nodes.reverse()
   return nodes

def getAutonegPd( descriptor: TraceEndpointDescriptor,
                  intf: EthIntfCli.EthPhyIntf
                  ) -> Optional[ AutonegPdBaseType ]:
   for phyAutonegDir in gv.autonegPdRootDir.values():
      if intf.slotId() not in phyAutonegDir[ "slice" ]:
         # the agent for this path wasn't initiated, skip
         continue
      sliceDir = phyAutonegDir[ "slice" ][ intf.slotId() ]
      # find the autonegPd collection member
      for autonegPdCollName in sliceDir.attributes:
         if autonegPdCollName in defaultEntityAttrs:
            continue
         autonegPdColl = getattr( sliceDir, autonegPdCollName )
         assert Tac.isCollection( autonegPdColl ),\
                "This element should be a collection"
         # autonegPdCollections should be keyed by TraceEndpointDescriptor. We check
         # whether a matching one exists, return None if not.
         if descriptor in autonegPdColl:
            return autonegPdColl[ descriptor ]
   return None

def matchAutonegInfo( component: CompNode,
                      intf: EthIntfCli.EthPhyIntf
                      ) -> tuple:

   autonegSliceDir = gv.autonegRootDir[ intf.slotId() ]
   intfId = intf.name
   for agentDir in autonegSliceDir.values():
      # pylint: disable-next=isinstance-second-argument-not-valid-type
      if not isinstance( agentDir, PhyAutonegStatusDirV2 ):
         continue
      if intfId not in agentDir.compDescriptor:
         continue

      candidateComponent = agentDir.compDescriptor[ intfId ]
      convertedCompDescriptor = None

      candidateComponentType = DescriptorHelper.traceEndpointComponentType(
         candidateComponent )

      if candidateComponentType != component.componentType:
         continue

      if candidateComponentType == ComponentType.componentSerdes:
         convertedCompDescriptor = DescriptorHelper.traceEndpointToSerdes(
            candidateComponent )
         name = convertedCompDescriptor.phyType
      elif candidateComponentType == ComponentType.componentXcvr:
         convertedCompDescriptor = DescriptorHelper.traceEndpointToXcvrLane(
            candidateComponent )
         name = "Xcvr"
      else:
         continue

      if convertedCompDescriptor != component.descriptor:
         continue

      autonegStatus = agentDir.phyAutonegStatus[ intfId ].phyAutonegStatus

      # Xcvrs will also be keyed by SerdesDescriptors in the future, ignoring for now
      if name == "Xcvr":
         autonegPd = None
      else:
         autonegPd = getAutonegPd( candidateComponent, intf )

      return name, autonegStatus, autonegPd
   raise ArException( "Unsupported component" )

def getPhyNegotiationStatusesDetailed(
      intf: EthIntfCli.EthPhyIntf
      ) -> list[ PhyNegotiationModel.PhyNegotiationStatusDetailed ]:
   if not gv.topoHelper:
      gv.topoHelper = Tac.newInstance( "Hardware::L1Topology::TraversalHelper",
                                       gv.l1TopoDir, gv.l1MappingDir )
   phyNegotiationStatuses = []
   if intf.name.startswith( "Management" ):
      name = ""
      phyNegotiationStatuses.append(
         PhyNegotiationModel.PhyNegotiationStatusDetailed().toModel(
            name, intf.status().autonegStatus,
            None, None, None ) )
   else:
      components = getComponentNodes( intf )
      for component in components:
         name, autonegStatus, autonegPd = matchAutonegInfo( component, intf )
         phyNegotiationStatuses.append(
            PhyNegotiationModel.PhyNegotiationStatusDetailed().toModel(
               name, autonegStatus,
               autonegPd, autonegStatesModelsCol.get( name ),
               autonegTrainingModelsCol.get( name ) ) )
   return phyNegotiationStatuses

def showInterfacesNegotiationDetailLegacy( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   exposeInternal = 'all' in args
   intfs = EthIntfCli.getAllIntfs( mode, intf, mod, EthIntfCli.L1Intf,
                                   exposeInternal=exposeInternal )
   if not intfs:
      return

   EthIntfCli.showInterfacesNegotiationDetails( intfs )

def showInterfacesNegotiationDetail( mode, args ):
   exposeInternal = 'all' in args
   intfnegotiationStatuses = {}
   intfs = EthIntfCli.getAllIntfs( mode, args.get( 'INTF' ), args.get( 'MOD' ),
                                   EthIntfCli.L1Intf, exposeInternal=exposeInternal )
   for intf in intfs:
      intfnegotiationStatuses[ intf.name ] = \
         PhyNegotiationModel.IntfNegotiationStatusDetailed(
            phyNegotiationStatuses=getPhyNegotiationStatusesDetailed( intf ) )
   return PhyNegotiationModel.NegotiationStatusesDetailed(
         negotiationStatuses=intfnegotiationStatuses )

def Plugin( entityManager ):
   gv.autonegRootDir = LazyMount.mount(
      entityManager, L1MountPaths.ethPhyAutonegStatusDirPath(),
      'Tac::Dir', 'ri' )
   gv.autonegPdRootDir = LazyMount.mount( entityManager,
                                          "hardware/phy/status/data/autonegPd",
                                          'Tac::Dir', 'ri' )
   gv.l1MappingDir = LazyMount.mount( entityManager, L1TopoSysdbPaths.mappingDirPath,
                                      'Tac::Dir', 'ri' )
   gv.l1TopoDir = LazyMount.mount( entityManager, L1TopoSysdbPaths.topologyDirPath,
                                   'Tac::Dir', 'ri' )
