# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import LazyCallback
import CliGlobal
import LazyMount
import Tac
from TypeFuture import TacLazyType

Polarity = Tac.Type( "Hardware::L1Topology::Polarity" )

qualifiedPolarityOutput = {
   "configDefault": "default",
   Polarity.polarityDefault: "standard",
   Polarity.polaritySwap: "reversed",
}

gv = CliGlobal.CliGlobal(
   lanePolarityDesc=None,
   configPolarityDesc=None,
)

DirectionSetting = LazyCallback( "PhyPolarityModel.DirectionSetting" )
PhyPolaritySetting = LazyCallback( "PhyPolarityModel.PhyPolaritySetting" )
PhyPolaritySettings = LazyCallback( "PhyPolarityModel.PhyPolaritySettings" )
PolaritySetting = LazyCallback( "PhyPolarityModel.PolaritySetting" )

IntfId = TacLazyType( "Arnet::IntfId" )
IntfSlotGroupId = TacLazyType( "L1::IntfSlotTypes::IntfSlotGroupId" )
L1TopoSysdbPaths = TacLazyType( "Hardware::L1Topology::SysdbPathConstants" )

def showPhyPolarityHandler( mode, args ):
   result = PhyPolaritySettings()
   slotDefaultPolarities = gv.lanePolarityDesc.intfSlotDefaultPolarity
   configPolarities = gv.configPolarityDesc.slotPolarityOverride
   if not slotDefaultPolarities:
      mode.addError( "No Slots found/Toggle not enabled" )
      return None

   # Destruct to retrieve the polarity and intfSlotLanes
   for intfSlotLaneDesc, defaultPolarity in slotDefaultPolarities.items():
      slotGroupId = intfSlotLaneDesc.slotGroupId
      isFixed = slotGroupId == IntfSlotGroupId.unknownIntfSlotGroupId
      slotPrefix = intfSlotLaneDesc.slotPrefix
      intfSlotId = str( intfSlotLaneDesc.intfSlotId )

      # BUG1004718 Standardize the creation of slotName from a
      # InterfaceSlotDescriptor
      if isFixed:
         slotName = f"{ slotPrefix }{ intfSlotId }"
      else:
         slotName = f"{ slotPrefix }{ str( slotGroupId ) }/{ intfSlotId }"
      lane = intfSlotLaneDesc.laneId

      if desiredSlot := args.get( "SLOT" ):
         intfLongName = IntfId.fromShortName( desiredSlot.capitalize() )
         if not intfLongName.lower() == slotName.lower():
            continue

         if args.get( "LANES" ):
            desiredLanes = args.get( "LANES" ).values()
            if lane not in desiredLanes:
               continue

      direction = "Tx" if intfSlotLaneDesc.tx else "Rx"

      configPolarity = None
      # Check for override
      if intfSlotLaneDesc in configPolarities:
         slotPolOverride = configPolarities[ intfSlotLaneDesc ]
         if intfSlotLaneDesc.tx and intfSlotLaneDesc.laneId \
                                             in slotPolOverride.txLaneOverride:
            lanePolOverride = \
               slotPolOverride.txLaneOverride[ intfSlotLaneDesc.laneId ]
            configPolarity = lanePolOverride.polarity
         elif not intfSlotLaneDesc.tx and intfSlotLaneDesc.laneId \
                                             in slotPolOverride.rxLaneOverride:
            lanePolOverride = \
               slotPolOverride.rxLaneOverride[ intfSlotLaneDesc.laneId ]
            configPolarity = lanePolOverride.polarity

      if configPolarity is None:
         configured = "configDefault"
      else:
         configured = configPolarity

      if slotName not in result.interfaceSlots:
         result.interfaceSlots[ slotName ] = PhyPolaritySetting()
      if lane not in result.interfaceSlots[ slotName ].lanes:
         result.interfaceSlots[ slotName ].lanes[ lane ] = DirectionSetting()
      if direction not in result.interfaceSlots[ slotName ].lanes[ lane ].direction:
         polaritySetting = PolaritySetting()
         polaritySetting.moduleDefault = qualifiedPolarityOutput[ defaultPolarity ]
         polaritySetting.configured = qualifiedPolarityOutput[ configured ]
         if polaritySetting.configured != "default":
            polaritySetting.operational = polaritySetting.configured
         else:
            polaritySetting.operational = polaritySetting.moduleDefault
         result.interfaceSlots[ slotName ].lanes[ lane ].direction[ direction ] \
                                                                  = polaritySetting

   return result

# -----------------------------------------------------------------------------
# Cli Plugin initialization
# -----------------------------------------------------------------------------
def Plugin( entityManager ):
   gv.lanePolarityDesc = LazyMount.mount( entityManager,
                                          L1TopoSysdbPaths.lookupTableDirPath,
                                          "Hardware::L1Topology::LookupDir", "ri" )
   gv.configPolarityDesc = LazyMount.mount( entityManager,
                               L1TopoSysdbPaths.polarityOverrideDirPath,
                               "Hardware::L1Topology::PolarityOverrideDir", "ri" )
