#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArPyUtils
from CliModel import (
   Dict,
   Enum,
   Model,
)
import TableOutput
import Tac

class PolaritySetting( Model ):
   moduleDefault = Enum( values=[ "standard", "reversed" ],
                         help="The module default polarity" )
   configured = Enum( values=[ "default", "standard", "reversed" ],
                      help="The configured value of polarity" )
   operational = Enum( values=[ "standard", "reversed" ],
                       help="The operational value of polarity" )

class DirectionSetting( Model ):
   direction = Dict( keyType=str, valueType=PolaritySetting,
                     help="A mapping of direction to polarities." )

class PhyPolaritySetting( Model ):
   lanes = Dict( keyType=int, valueType=DirectionSetting,
                 help="A mapping of lanes to direction." )

class PhyPolaritySettings( Model ):
   interfaceSlots = Dict( keyType=str, valueType=PhyPolaritySetting,
                          help="A mapping of interfaceSlots to its lanes." )

   def render( self ):
      headers = ( "Slot", "Lane", "Direction", "Module Default",
                  "Configured", "Operational" )

      table = TableOutput.createTable( headers )
      formatSlotColumn = TableOutput.Format( justify="left" )
      formatLaneColumn = TableOutput.Format( justify="right" )
      formatDirectionColumn = TableOutput.Format( justify="left" )
      formatDefaultColumn = TableOutput.Format( justify="left" )
      formatConfiguredColumn = TableOutput.Format( justify="left" )
      formatOperationalColumn = TableOutput.Format( justify="left" )
      cols = [ formatSlotColumn, formatLaneColumn, formatDirectionColumn,
               formatDefaultColumn, formatConfiguredColumn, formatOperationalColumn ]

      # Remove additional column paddings
      for col in cols:
         col.padLimitIs( True )

      table.formatColumns( *cols )
      for slotName in ArPyUtils.naturalsorted( self.interfaceSlots ):
         laneSetting = self.interfaceSlots[ slotName ].lanes
         for lane in sorted( laneSetting ):
            dirSetting = laneSetting[ lane ].direction
            for direction in ArPyUtils.naturalsorted( dirSetting ):
               polaritySetting = dirSetting[ direction ]

               table.newRow( slotName,
                             lane,
                             direction,
                             polaritySetting.moduleDefault,
                             polaritySetting.configured,
                             polaritySetting.operational )

      print( table.output() )
