# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.Qos import PolicingModeBase
from CliPlugin import EthIntfCli
from CliPlugin import IntfCli
from CliPlugin import SubIntfCli
from CliPlugin import QosCliServicePolicy
from CliPlugin.QosCliModel import ( ModePolicingModel, InterfacePolicerAllModel,
                                    InterfacePolicingCounterAllModel )
from CliPlugin.QosCliCommon import ( matcherSize, matcherAdjustment, matcherPlus,
                                     matcherMinus, matcherBytes, setIntfConfig )
from QosCliLib import ( PolicingModelContainer, IntfPolicerModelContainer,
                        PolicingCountersModelContainer )
from QosLib import isLagPort
from QosTypes import ( tacPolicerType, tacPolicerPktSizeAdjProfileName, tacBurstUnit,
                       tacRateUnit )
import ConfigMount
import LazyMount
import Tac
import CliParser
import BasicCli
import Tracing
import Cell
import CliGlobal
import CliCommand
import CliMatcher

__defaultTraceHandle__ = Tracing.Handle( 'CliDynamicPolicingCli' )
t0 = Tracing.trace0

# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
gv = CliGlobal.CliGlobal(
   dict(
      qosInputConfig=None,
      qosConfig=None,
      qosStatus=None,
      qosAclHwStatus=None,
      intfPolicingHwStatus=None,
      cliCounterConfig=None,
      qosHwStatus=None,
      qosSliceHwStatus=None,
      lagInputConfig=None,
      bridgingHwCapabilities=None,
   )
)

# -----------------------------------------------------------------------------------
# Name Rules for Interface Policing
# -----------------------------------------------------------------------------------
def getProfileNameRule( mode ):
   if gv.qosInputConfig.policingConfig is None:
      return []
   suggestedCompletions = list( gv.qosInputConfig.policingConfig.profile )
   suggestedCompletions = suggestedCompletions + \
      list( gv.qosInputConfig.policingConfig.policerPktSizeAdjProfile )
   return suggestedCompletions

def getPolicerInstanceNameRule( mode ):
   if gv.qosInputConfig.policingConfig is None:
      return []
   suggestedCompletions = gv.qosInputConfig.policingConfig.policerInstance
   return suggestedCompletions

# -----------------------------------------------------------------------------------
# Guards for Interface Policing
# -----------------------------------------------------------------------------------
def guardPolicingMode( mode, token ):
   if gv.qosAclHwStatus.interfacePolicingSupported or \
      gv.qosHwStatus.perPortPolicerPacketSizeAdjSupported or \
      gv.qosAclHwStatus.interfaceControlPlanePolicingDedicatedPolicerSupported or \
      gv.qosAclHwStatus.interfaceEgressPolicingDedicatedPolicerSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPolicingGroupPolicer( mode, token ):
   if gv.qosAclHwStatus.interfacePolicingSupported and \
         ( gv.qosAclHwStatus.interfacePolicingGroupPolicerSupported or
            gv.bridgingHwCapabilities.vxlanVniGroupPolicerSupported ):
      return None
   return CliParser.guardNotThisPlatform

def guardPolicingTrTcmConfig( mode, token ):
   if gv.qosAclHwStatus.interfacePolicingTrTcmSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIntfDedicatedPolicer( mode, token ):
   if mode.intf.isSubIntf() and \
      gv.qosAclHwStatus.interfacePolicingDedicatedPolicerSupported and \
      ( gv.qosAclHwStatus.interfacePolicingSupported or
        gv.qosAclHwStatus.interfaceControlPlanePolicingDedicatedPolicerSupported or
        gv.qosAclHwStatus.interfaceEgressPolicingDedicatedPolicerSupported ):
      return None
   elif ( not mode.intf.isSubIntf() and
          gv.qosAclHwStatus.interfacePolicingDedicatedPolicerSupported and
          gv.qosAclHwStatus.interfacePolicingSupported and
          gv.qosAclHwStatus.interfacePolicingOnFppSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def guardIntfGroupPolicer( mode, token ):
   if mode.intf.isSubIntf() and \
      gv.qosAclHwStatus.interfacePolicingGroupPolicerSupported and \
      gv.qosAclHwStatus.interfacePolicingSupported:
      return None
   elif ( not mode.intf.isSubIntf() and
         gv.qosAclHwStatus.interfacePolicingGroupPolicerSupported and
         gv.qosAclHwStatus.interfacePolicingSupported and
         gv.qosAclHwStatus.interfacePolicingGroupPolicerOnFppSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def guardIntfPolicing( mode, token ):
   if guardIntfDedicatedPolicer( mode, token ) and \
         guardIntfGroupPolicer( mode, token ):
      return CliParser.guardNotThisPlatform
   return None

def guardIntfCoppPolicing( mode, token ):
   if not gv.qosAclHwStatus.interfaceControlPlanePolicingDedicatedPolicerSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardIntfPolicerOutputDir( mode, token ):
   if not gv.qosAclHwStatus.interfaceEgressPolicingDedicatedPolicerSupported:
      return CliParser.guardNotThisPlatform
   return None

# Show counters guard
def guardPolicerPacketSizeAdj( mode, token ):
   if gv.qosHwStatus.perPortPolicerPacketSizeAdjSupported:
      return None
   return CliParser.guardNotThisPlatform

# -------------------------------------------------------------------------------
# Utility Functions
# -------------------------------------------------------------------------------
def policerPktSizeAdjRangeFn( mode, context ):
   return ( 1, gv.qosHwStatus.maxPolicerPktSizeAdjBytes )


# -------------------------------------------------------------------------------
# Tokens for Policing Mode, Interface-Policers in SubIntf Mode and Show Policing
# -------------------------------------------------------------------------------
nodeProfile = CliCommand.guardedKeyword( 'profile',
      helpdesc="Policer configuration", guard=guardPolicingMode )
nodeProfileName = CliMatcher.DynamicNameMatcher( getProfileNameRule,
      helpdesc="Policer profile name",
   pattern=r'(?!counters$|interface$|summary$|group$|input$|'
           r'default-pkt-size-adj-profile$)[A-Za-z0-9_:{}\[\]-]+' )
nodePoliceHRate = CliCommand.guardedKeyword( 'rate',
      helpdesc="Set higher rate", guard=guardPolicingTrTcmConfig )
nodePoliceHBs = CliMatcher.KeywordMatcher( 'burst-size',
      helpdesc="Set burst-size for higher rate" )
nodePolicer = CliCommand.guardedKeyword( 'policer',
      helpdesc="Associate policer name with a policer profile",
      guard=guardPolicingGroupPolicer )
nodePolicerName = CliMatcher.DynamicNameMatcher( getPolicerInstanceNameRule,
      helpdesc="Policer name",
   pattern=r'(?!counters$|interface$|summary$|group$|input$)[A-Za-z0-9_:{}\[\]-]+' )
nodePolicerProfile = CliMatcher.KeywordMatcher( 'profile',
      helpdesc="Associate policer name with a policer profile" )
nodeIntfPolicer = CliCommand.guardedKeyword( 'policer',
      helpdesc="Policer configuration on the interface", guard=guardIntfPolicing )
nodeIntfPolicerTypeCopp = CliCommand.guardedKeyword( 'cpu',
      helpdesc="Apply policing to CPU-bound packets",
      guard=guardIntfCoppPolicing,
      alias="TYPE" )
nodeIntfProfile = CliCommand.guardedKeyword( 'profile',
      helpdesc="Associate interface with a policer profile",
      guard=guardIntfDedicatedPolicer )
nodeIntfGroupPolicer = CliCommand.guardedKeyword( 'group',
      helpdesc="Associate interface with a group policer",
      guard=guardIntfGroupPolicer )
nodeInput = CliCommand.Node(
      CliMatcher.KeywordMatcher( 'input',
      helpdesc="Apply policing to inbound packets" ),
      alias="TYPE" )
nodeOutput = CliCommand.guardedKeyword( 'output',
      helpdesc="Apply policing to output packets",
      guard=guardIntfPolicerOutputDir,
      alias="TYPE" )
nodePacket = CliCommand.guardedKeyword( 'packet',
      helpdesc='Configure packet size parameters', guard=guardPolicerPacketSizeAdj )
matcherBytesAdjustmentMatcher = CliMatcher.DynamicIntegerMatcher(
      rangeFn=policerPktSizeAdjRangeFn, helpdesc='Adjustment value( in bytes )' )
matcherPoliceCirValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc="Specify lower rate" )

# -----------------------------------------------------------------------------------
# Cli commands for Interface Policing
# -----------------------------------------------------------------------------------

# -----------------------------------------------------------------------------------
# [ no | default ] profile PROFILE rate <value> {bps, kbps, mbps, gbps} burst-size
# <value> {bytes, kbytes, mbytes} rate <value> {bps, kbps, mbps, gbps} burst-size
# <value> {bytes, kbytes, mbytes}
# -----------------------------------------------------------------------------------
class ProfileConfigHigherRateCmd( CliCommand.CliCommandClass ):
   syntax = ( 'profile PROFILE rate CIR [ CIRUNIT ] burst-size BURST [ BURSTUNIT ] '
              '[highrate PIR [ PIRUNIT ] highbsize HBURST [ HBUNIT ]]' )
   noOrDefaultSyntax = 'profile PROFILE ...'
   data = {
      'profile': nodeProfile,
      'PROFILE': nodeProfileName,
      'rate': QosCliServicePolicy.nodePoliceLRate,
      'CIR': matcherPoliceCirValue,
      'CIRUNIT': QosCliServicePolicy.CirPirUnit(),
      'burst-size': QosCliServicePolicy.nodePoliceLBs,
      'BURST': QosCliServicePolicy.matcherPoliceBurstValue,
      'BURSTUNIT': QosCliServicePolicy.BurstHigherUnit(),
      'highrate': nodePoliceHRate,
      'PIR': QosCliServicePolicy.matcherPolicePirValue,
      'PIRUNIT': QosCliServicePolicy.CirPirUnit(),
      'highbsize': nodePoliceHBs,
      'HBURST': QosCliServicePolicy.matcherPoliceExcessBurstValue,
      'HBUNIT': QosCliServicePolicy.BurstHigherUnit(),
   }

   handler = "PolicingCli.doProfileConfigHigherRateCmd"

   noOrDefaultHandler = "PolicingCli.noProfileConfigHigherRateCmd"

# -----------------------------------------------------------------------------------
# [ no | default ] policer POLICER profile PROFILE
# -----------------------------------------------------------------------------------
class PolicerConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'policer POLICER profile PROFILE'
   noOrDefaultSyntax = 'policer POLICER ...'
   data = {
      'policer': nodePolicer,
      'POLICER': nodePolicerName,
      'profile': nodePolicerProfile,
      'PROFILE': nodeProfileName,
   }

   handler = "PolicingCli.doPolicerConfigCmd"

   noOrDefaultHandler = "PolicingCli.noPolicerConfigCmd"

# -----------------------------------------------------------------------------------
# [ no | default ] policer profile PROFILE group POLICER input | output | cpu
# -----------------------------------------------------------------------------------
class InterfacePolicerConfig( CliCommand.CliCommandClass ):
   syntax = ( 'policer (([profile PROFILE] group POLICER) | (profile PROFILE)) '
            '(input | output | cpu)' )
   noOrDefaultSyntax = syntax
   data = {
      'policer': nodeIntfPolicer,
      'profile': nodeIntfProfile,
      'PROFILE': nodeProfileName,
      'group': nodeIntfGroupPolicer,
      'POLICER': nodePolicerName,
      'input': nodeInput,
      'output': nodeOutput,
      'cpu': nodeIntfPolicerTypeCopp,
   }

   handler = "PolicingCli.doInterfacePolicerConfig"

   noOrDefaultHandler = "PolicingCli.noInterfacePolicerConfig"

# -----------------------------------------------------------------------------------
# [ no | default ] profile PROFILE packet size adjustment ( plus | minus )
# <value> bytes
# -----------------------------------------------------------------------------------
class PolicerPacketSizeAdj( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE packet size adjustment ( plus | minus ) BYTES bytes'
   noOrDefaultSyntax = 'profile PROFILE packet size adjustment ...'
   data = {
      'profile': nodeProfile,
      'PROFILE': nodeProfileName,
      'packet': nodePacket,
      'size': matcherSize,
      'adjustment': matcherAdjustment,
      'plus': matcherPlus,
      'minus': matcherMinus,
      'BYTES': matcherBytesAdjustmentMatcher,
      'bytes': matcherBytes,
   }

   handler = "PolicingCli.doPolicerPacketSizeAdj"

   noOrDefaultHandler = "PolicingCli.noPolicerPacketSizeAdj"

# -----------------------------------------------------------------------------------
# The Policing Mode
# -----------------------------------------------------------------------------------
class PolicingMode( PolicingModeBase, BasicCli.ConfigModeBase ):
   name = "Policing Mode Configuration"

   def __init__( self, parent, session ):
      PolicingModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def identicalProfile( self, profile, cir, cirUnit, bc, bcUnit, pir, pirUnit, be,
                         beUnit ):
      t0( 'Checking if profile:', profile.profileName, 'is identical' )
      pol = profile.profileConfig
      if ( ( pol.cir != cir ) or ( pol.cirUnit != cirUnit ) or ( pol.bc != bc ) or
           ( pol.bcUnit != bcUnit ) or ( pol.pir != pir ) or
           ( pol.pirUnit != pirUnit ) or ( pol.be != be ) or
           ( pol.beUnit != beUnit ) ):
         return False
      return True

   def adjustRateAndBurstUnits( self, cirUnit, bcUnit, pirUnit, beUnit ):
      if cirUnit is None:
         if bcUnit == tacBurstUnit.burstUnitPackets:
            cirUnit = tacRateUnit.rateUnitPps
         else:
            cirUnit = tacRateUnit.rateUnitbps
      if bcUnit is None:
         if cirUnit == tacRateUnit.rateUnitPps:
            bcUnit = tacBurstUnit.burstUnitPackets
         else:
            bcUnit = tacBurstUnit.burstUnitBytes
      if pirUnit is None:
         if cirUnit != tacRateUnit.rateUnitPps:
            pirUnit = tacRateUnit.rateUnitbps
         else:
            pirUnit = tacRateUnit.rateUnitPps
      if beUnit is None:
         if pirUnit != tacRateUnit.rateUnitPps:
            beUnit = tacBurstUnit.burstUnitBytes
         else:
            beUnit = tacBurstUnit.burstUnitPackets
      return cirUnit, bcUnit, pirUnit, beUnit

   def checkExistingProfile( self, profileName ):
      t0( 'Checking if profile:', profileName, 'exists' )
      return profileName in gv.qosInputConfig.policingConfig.profile

   def checkExistingPolicer( self, policerName ):
      t0( 'Checking if policerInstance:', policerName, 'exists' )
      return policerName in gv.qosInputConfig.policingConfig.policerInstance

   def clear( self, args ):  # pylint: disable=arguments-renamed
      if config := gv.qosInputConfig.policingConfig:
         config.policerInstance.clear()
         config.policerPktSizeAdjProfile.clear()
         config.profile.clear()

   def configureProfileParameters( self, profile, cir, cirUnit, bc, bcUnit, pir,
                                   pirUnit, be, beUnit ):
      profile.profileConfig.cir = cir
      profile.profileConfig.cirUnit = cirUnit
      profile.profileConfig.bc = bc
      profile.profileConfig.bcUnit = bcUnit
      profile.profileConfig.pir = pir
      profile.profileConfig.pirUnit = pirUnit
      profile.profileConfig.be = be
      profile.profileConfig.beUnit = beUnit

   def changeProfileConfig( self, profileName, cir, cirUnit, bc, bcUnit, pir,
                            pirUnit, be, beUnit ):
      t0( 'Change profile config for profile: ', profileName )
      assert profileName in gv.qosInputConfig.policingConfig.profile
      profile = gv.qosInputConfig.policingConfig.profile[ profileName ]
      if not self.identicalProfile( profile, cir, cirUnit, bc, bcUnit, pir,
                                    pirUnit, be, beUnit ):
         self.configureProfileParameters( profile, cir, cirUnit, bc, bcUnit, pir,
                                          pirUnit, be, beUnit )
         profile.uniqueId = Tac.Value( 'Qos::UniqueId' )

   def changePolicerConfig( self, policerName, profileName ):
      t0( 'Change profile association config for policerInstance: ', policerName,
          'with new profile: ', profileName )
      assert policerName in gv.qosInputConfig.policingConfig.policerInstance
      policer = gv.qosInputConfig.policingConfig.policerInstance[ policerName ]
      if policer.profileName != profileName:
         policer.profileName = profileName
         policer.uniqueId = Tac.Value( 'Qos::UniqueId' )

   def configureProfile( self, profileName, policerMode='committed',
                         cir=None, cirUnit=None, bc=None, bcUnit=None, pir=None,
                         pirUnit=None, be=None, beUnit=None ):
      t0( 'Handler to configure profile: ', profileName, ' with associated rates' )
      cirUnit, bcUnit, pirUnit, beUnit = self.adjustRateAndBurstUnits( cirUnit,
                                             bcUnit, pirUnit, beUnit )
      if policerMode == 'committed':
         pir = be = 0
      if ( not self.session_.startupConfig() and
           QosCliServicePolicy.policerOutOfRange( self, cir, cirUnit, bc, bcUnit,
               pir, pirUnit, be, beUnit, policerMode ) ):
         # In case of startup-config, ignore the rate check since platform
         # agents are not up to set the rate limits.
         return
      exists = self.checkExistingProfile( profileName )
      if not exists:
         # Add new profile to gv.qosInputConfig.policingConfig.profile
         newProfile = gv.qosInputConfig.policingConfig.profile.\
                                                            newMember( profileName )
         newProfile.profileConfig = ( "", cir, bc, pir, be )
         self.configureProfileParameters( newProfile, cir, cirUnit, bc, bcUnit,
                                          pir, pirUnit, be, beUnit )
         newProfile.uniqueId = Tac.Value( 'Qos::UniqueId' )
      else:
         self.changeProfileConfig( profileName, cir, cirUnit, bc, bcUnit, pir,
                                   pirUnit, be, beUnit )

   def configurePolicer( self, policerName, profileName ):
      t0( 'Handler to configure policerInstance: ', policerName, ' with profile: ',
          profileName )
      exists = self.checkExistingPolicer( policerName )
      if not exists:
         # Add new policerInstance to
         # gv.qosInputConfig.policingConfig.policerInstance
         newPolicer = gv.qosInputConfig.policingConfig.policerInstance.newMember(
            policerName )
         newPolicer.profileName = profileName
         newPolicer.uniqueId = Tac.Value( 'Qos::UniqueId' )
      else:
         self.changePolicerConfig( policerName, profileName )

   def setPolicerPktSizeAdjProfile( self, profileName, compensation ):
      t0( 'Handle to configure policer packet size adjustment profile: ',
           profileName, ' packetSizeAdj: ', compensation )
      gv.qosInputConfig.policingConfig.policerPktSizeAdjProfile.newMember(
         profileName, compensation )

   def noPolicerProfile( self, profileName ):
      t0( 'Removing profile: ', profileName )
      del gv.qosInputConfig.policingConfig.profile[ profileName ]

   def noPolicerPktSizeAdjProfile( self, profileName ):
      t0( 'Removing packet size adjustment profile: ', profileName )
      del gv.qosInputConfig.policingConfig.policerPktSizeAdjProfile[ profileName ]


PolicingMode.addCommandClass( ProfileConfigHigherRateCmd )
PolicingMode.addCommandClass( PolicerConfigCmd )
PolicingMode.addCommandClass( PolicerPacketSizeAdj )


# -------------------------------------------------------------------------------
# Policing Mode
# -------------------------------------------------------------------------------
def gotoPolicingMode( mode, args ):
   if gv.qosInputConfig.policingConfig is None:
      t0( 'gotoPolicingMode: Creating policingConfig in qos/input/config/cli' )
      policingConfigName = "QosPolicing"
      gv.qosInputConfig.policingConfig = ( policingConfigName, )
   childMode = mode.childMode( PolicingMode )
   mode.session_.gotoChildMode( childMode )

def checkExistingInterfacePolicer( intfPairKey, policerType='input' ):
   t0( 'Checking if', intfPairKey, 'exists' )
   intfPolicer = None
   if policerType == 'input':
      intfPolicer = gv.qosInputConfig.policingConfig.intfPolicer
   elif policerType == 'output':
      intfPolicer = gv.qosInputConfig.policingConfig.intfPolicerOutput
   elif policerType == 'cpu':
      intfPolicer = gv.qosInputConfig.policingConfig.intfPolicerCpu
   return intfPairKey in intfPolicer

def isInterfacePolicerSame( intfPairKey, profile, policerInstance, policerType ):
   t0( 'IsInterfacePolicerSame for profile: ', profile, ' and policer: ',
       policerInstance )
   intfPolicer = None
   if policerType == 'input':
      intfPolicer = gv.qosInputConfig.policingConfig.intfPolicer[ intfPairKey ]
   elif policerType == 'output':
      intfPolicer = gv.qosInputConfig.policingConfig.intfPolicerOutput[ intfPairKey ]
   elif policerType == 'cpu':
      intfPolicer = \
            gv.qosInputConfig.policingConfig.intfPolicerCpu[ intfPairKey ]
   profileCheck = False
   policerCheck = False
   profileToCheck = intfPolicer.policerName.get( tacPolicerType.dedicatedPolicer )
   profileCheck = profile in ( None, profileToCheck )
   policerToCheck = intfPolicer.policerName.get( tacPolicerType.policerInstance )
   policerCheck = policerInstance in ( None, policerToCheck )
   return profileCheck and policerCheck

def addIntfPolicerToConfig( intfPairKey, profile, policerInstance, policerType ):
   t0( 'Adding ', intfPairKey, 'with profile: ', profile, ' and policerInstance: ',
       policerInstance, ' to the intf-Policer collection' )
   intfPolicer = None
   if policerType == 'input':
      intfPolicer = \
         gv.qosInputConfig.policingConfig.intfPolicer.newMember( intfPairKey )
   elif policerType == 'output':
      intfPolicer = \
         gv.qosInputConfig.policingConfig.intfPolicerOutput.newMember( intfPairKey )
   elif policerType == 'cpu':
      intfPolicer = \
         gv.qosInputConfig.policingConfig.intfPolicerCpu.newMember(
                  intfPairKey )

   if profile:
      intfPolicer.policerName[ tacPolicerType.dedicatedPolicer ] = profile

   if policerInstance:
      intfPolicer.policerName[ tacPolicerType.policerInstance ] = policerInstance

   intfPolicer.uniqueId = Tac.Value( 'Qos::UniqueId' )

def deleteIntfPolicerFromConfig( intfPairKey, profile, policerInstance,
                                 policerType ):
   t0( 'Deleting ', intfPairKey, ' from intf-Policer collection' )
   if policerType in ( 'input', 'all' ) and \
         intfPairKey in gv.qosInputConfig.policingConfig.intfPolicer:
      curProfile = gv.qosInputConfig.policingConfig.intfPolicer[ intfPairKey ].\
            policerName.get( tacPolicerType.dedicatedPolicer, '' )
      curPolicer = gv.qosInputConfig.policingConfig.intfPolicer[ intfPairKey ].\
            policerName.get( tacPolicerType.policerInstance, '' )
      if ( not profile and not policerInstance ) or (
            profile == curProfile and curPolicer == policerInstance ) or (
            profile == curProfile and curPolicer == '' ) or (
            curProfile == '' and curPolicer == policerInstance ):
         del gv.qosInputConfig.policingConfig.intfPolicer[ intfPairKey ]
   if policerType in ( 'output', 'all' ) and \
         intfPairKey in gv.qosInputConfig.policingConfig.intfPolicerOutput:
      curProfile = gv.qosInputConfig.policingConfig.\
                  intfPolicerOutput[ intfPairKey ].\
                  policerName.get( tacPolicerType.dedicatedPolicer, '' )
      if not profile or curProfile == profile:
         del gv.qosInputConfig.policingConfig.intfPolicerOutput[ intfPairKey ]
   if policerType in ( 'cpu', 'all' ) and \
         intfPairKey in gv.qosInputConfig.policingConfig.intfPolicerCpu:
      curProfile = gv.qosInputConfig.policingConfig.intfPolicerCpu[ intfPairKey ].\
            policerName.get( tacPolicerType.dedicatedPolicer, '' )
      if not profile or curProfile == profile:
         del gv.qosInputConfig.policingConfig.intfPolicerCpu[ intfPairKey ]

def configureInterfacePolicer( mode, profile=None, policerInstance=None,
                               policerType='input', no=False ):
   t0( 'Handler to configure Interface-Policers on ', mode.intf.name )
   if not gv.qosInputConfig.policingConfig:
      t0( 'Creating policingConfig in qos/input/config/cli' )
      gv.qosInputConfig.policingConfig = ( 'QosPolicing', )

   intfName = mode.intf.name
   if isLagPort( intfName ):
      intfPair = Tac.Value( "Qos::IntfPairKey", "", intfName )
   else:
      intfPair = Tac.Value( "Qos::IntfPairKey", intfName, "" )
   if not no:
      exists = checkExistingInterfacePolicer( intfPair, policerType )
      if not exists:
         addIntfPolicerToConfig( intfPair, profile, policerInstance, policerType )
      else:
         if not isInterfacePolicerSame( intfPair, profile, policerInstance,
                                        policerType ):
            deleteIntfPolicerFromConfig( intfPair, profile, policerInstance,
                                         policerType )
            addIntfPolicerToConfig( intfPair, profile, policerInstance, policerType )
   else:
      deleteIntfPolicerFromConfig( intfPair, profile, policerInstance, policerType )

def setIntfPolicerPacketSizeAdj( mode, args ):
   profileName = args[ 'PROFILE' ]
   intf = mode.intf.name
   setIntfConfig( intf, cfgPolicerPktSizeAdjProfileName=profileName )

def noIntfPolicerPacketSizeAdj( mode, args ):
   profileName = tacPolicerPktSizeAdjProfileName.defaultProfileName
   intf = mode.intf.name
   setIntfConfig( intf, cfgPolicerPktSizeAdjProfileName=profileName )

# -----------------------------------------------------------------------------------
# Show CLI handlers
# -----------------------------------------------------------------------------------
def showPolicingHandler( mode, args ):
   modePolicingModel = ModePolicingModel()
   policingModelContainer = PolicingModelContainer( gv.qosInputConfig, gv.qosConfig,
                                             gv.qosStatus, gv.intfPolicingHwStatus,
                                             gv.qosSliceHwStatus, gv.lagInputConfig,
                                             modePolicingModel )
   profileName = args.get( 'PROFILE' )
   policerName = args.get( 'POLICER' )
   pktSizeAdjProfileOutput = 'packet' in args
   profileOutput = 'profile' in args and not pktSizeAdjProfileOutput
   policerOutput = 'group' in args
   if profileName:
      if pktSizeAdjProfileOutput:
         policingModelContainer.populatePolicerPktSizeAdjProfile( profileName )
      else:
         policingModelContainer.populateProfile( profileName )
   elif policerName:
      policingModelContainer.populatePolicerInstance( policerName )
   else:
      if not profileOutput and not policerOutput and not pktSizeAdjProfileOutput:
         profileOutput = True
         policerOutput = True
         pktSizeAdjProfileOutput = True
      policingModelContainer.populateAllPolicerProfiles( profileOutput,
                                                         policerOutput,
                                                         pktSizeAdjProfileOutput )
   return modePolicingModel

def getEthAndSubIntfs( mode, intf ):
   intfType = ( EthIntfCli.EthIntf, SubIntfCli.SubIntf )
   return IntfCli.Intf.getAll( mode, intf, None, intfType=intfType )

def showPolicingIntfHandler( mode, args ):
   intfPolicerAllModel = InterfacePolicerAllModel()
   intfPolicerModelContainer = IntfPolicerModelContainer( gv.qosAclHwStatus,
                                                          gv.qosConfig, gv.qosStatus,
                                                          gv.intfPolicingHwStatus,
                                                          intfPolicerAllModel )
   intf = args.get( 'INTF' )
   intfs = getEthAndSubIntfs( mode, intf )
   detail_ = 'detail' in args
   policerType = args.get( 'TYPE', 'all' )
   if intf is None or not intfs:
      intfPolicerModelContainer.populateAllIntfPolicers( detail_, policerType )
   else:
      intfPolicerModelContainer.populateIntfRangePolicers(
         intfs, detail_, policerType )
   return intfPolicerAllModel

def showPolicingIntfCountersHandler( mode, args ):
   intfPolicingCountersAllModel = InterfacePolicingCounterAllModel()
   modelContainer = PolicingCountersModelContainer( gv.qosConfig, gv.qosStatus,
                                                    gv.qosAclHwStatus,
                                                    gv.intfPolicingHwStatus,
                                                    intfPolicingCountersAllModel )
   intfs = getEthAndSubIntfs( mode, args.get( 'INTF' ) )
   policerType = args.get( 'TYPE', 'all' )
   modelContainer.populateIntfPolicingCountersForIntfs( intfs, policerType )
   return intfPolicingCountersAllModel

def clearPolicingIntfCountersHandler( mode, args ):
   # We only issue clear counters request for policing directions where counter
   # reporting for that specific policer direction is supported
   def getSupportedPolicerTypes():
      policerTypes = []
      if gv.qosAclHwStatus.interfacePolicingCountersSupported:
         policerTypes.append( 'input' )
      if gv.qosAclHwStatus.interfaceEgressPolicingCountersSupported:
         policerTypes.append( 'output' )
      if gv.qosAclHwStatus.interfaceControlPlanePolicingCountersSupported:
         policerTypes.append( 'cpu' )
      return policerTypes

   intf = args.get( 'INTF' )
   intfs = getEthAndSubIntfs( mode, intf )
   policerType = args.get( 'TYPE', getSupportedPolicerTypes() )
   if not isinstance( policerType, list ):
      policerType = [ policerType ]

   if intf is None:
      # Clear for all interfaces
      if 'input' in policerType:
         gv.cliCounterConfig.clearIntfPolicingCountersRequestTime = Tac.now()

      if 'output' in policerType:
         gv.cliCounterConfig.clearIntfOutputPolicingCountersRequestTime = Tac.now()

      if 'cpu' in policerType:
         gv.cliCounterConfig.clearIntfControlPlanePolicingCountersRequestTime = \
            Tac.now()
   else:
      # Update for each interface
      for intf in intfs:
         intfName = intf.name
         if isLagPort( intfName ):
            intfPair = Tac.Value( "Qos::IntfPairKey", "", intfName )
         else:
            intfPair = Tac.Value( "Qos::IntfPairKey", intfName, "" )

         if 'input' in policerType:
            if checkExistingInterfacePolicer( intfPair, 'input' ):
               intfPolicer = gv.qosInputConfig.policingConfig.intfPolicer[ intfPair ]
               intfPolicer.clearCountersTime = Tac.now()

         if 'output' in policerType:
            if checkExistingInterfacePolicer( intfPair, 'output' ):
               intfPolicer = \
                  gv.qosInputConfig.policingConfig.intfPolicerOutput[ intfPair ]
               intfPolicer.clearCountersTime = Tac.now()

         if 'cpu' in policerType:
            if checkExistingInterfacePolicer( intfPair, 'cpu' ):
               intfPolicer = gv.qosInputConfig.policingConfig.\
                                                         intfPolicerCpu[ intfPair ]
               intfPolicer.clearCountersTime = Tac.now()

def doProfileConfigHigherRateCmd( mode, args ):
   if args.get( 'PIR' ):
      mode.configureProfile( profileName=args[ 'PROFILE' ],
            policerMode='trTCM', cir=int( args[ 'CIR' ] ),
            cirUnit=args[ 'CIR_SPEED_UNIT' ], bc=int( args[ 'BURST' ] ),
            bcUnit=args[ 'BURST_RATE_UNIT' ], pir=int( args[ 'PIR' ] ),
            pirUnit=args[ 'PIR_SPEED_UNIT' ], be=int( args[ 'HBURST' ] ),
            beUnit=args[ 'HBURST_RATE_UNIT' ] )
   else:
      mode.configureProfile( profileName=args[ 'PROFILE' ],
            cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
            bc=int( args[ 'BURST' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ] )

def noProfileConfigHigherRateCmd( mode, args ):
   profileName = args[ 'PROFILE' ]
   mode.noPolicerProfile( profileName )

def doPolicerConfigCmd( mode, args ):
   mode.configurePolicer( policerName=args[ 'POLICER' ],
                          profileName=args[ 'PROFILE' ] )

def noPolicerConfigCmd( mode, args ):
   policerName = args[ 'POLICER' ]
   del gv.qosInputConfig.policingConfig.policerInstance[ policerName ]

def doInterfacePolicerConfig( mode, args ):
   policerType = args.get( 'TYPE', 'input' )
   configureInterfacePolicer( mode, profile=args.get( 'PROFILE' ),
                              policerInstance=args.get( 'POLICER' ),
                              policerType=policerType )

def noInterfacePolicerConfig( mode, args ):
   policerType = args.get( 'TYPE', 'all' )
   profile = args.get( 'PROFILE', None )
   policerInstance = args.get( 'POLICER', None )
   configureInterfacePolicer( mode, profile=profile,
                              policerInstance=policerInstance,
                              policerType=policerType, no=True )

def doPolicerPacketSizeAdj( mode, args ):
   profileName = args[ 'PROFILE' ]
   minus = 'minus' in args
   compensation = args[ 'BYTES' ]
   if minus:
      compensation = -compensation
   mode.setPolicerPktSizeAdjProfile( profileName, compensation )

def noPolicerPacketSizeAdj( mode, args ):
   profileName = args[ 'PROFILE' ]
   mode.noPolicerPktSizeAdjProfile( profileName )

def Plugin( entityManager ):
   gv.qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   gv.qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   gv.qosAclHwStatus = LazyMount.mount( entityManager,
                        "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   gv.qosHwStatus = LazyMount.mount( entityManager,
                        "qos/hardware/status/global", "Qos::HwStatus", "r" )
   qosSliceHwStatusDirPath = ( "cell/" + str( Cell.cellId() ) +
                                "/qos/hardware/status/slice" )
   gv.qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   intfPolicingHwStatusDir = "qos/hardware/policing/status"
   gv.intfPolicingHwStatus = LazyMount.mount( entityManager, intfPolicingHwStatusDir,
                                           "Tac::Dir", "ri" )
   gv.cliCounterConfig = LazyMount.mount( entityManager, "qos/cliCounterConfig",
                                       "Qos::CounterConfig", "w" )
   gv.lagInputConfig = LazyMount.mount( entityManager, "lag/input/config/cli",
                                     "Lag::Input::Config", "r" )
   gv.bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   gv.qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
