# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import math
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.PowerManagerCli import _systemInitialized, managementMode
import LazyMount

PowerManagerModel = CliDynamicPlugin( "PowerManagerModel" )

entityMib = None
powerManager = None
powerManagerHwConfig = None
powerManagerCliConfig = None

#--------------------------------------------------------------------------------
# show system environment power domain
#--------------------------------------------------------------------------------
def showSystemEnvPowerDomainHandler( mode, args ):
   if not _systemInitialized():
      mode.addError( 'System is not yet initialized' )
      return None

   model = PowerManagerModel.PowerDomainCollection()

   for domainName, domain in powerManagerHwConfig.powerDomain.items():
      # Strip 'PowerDomain' prefix from 'PowerDomain1'
      model.powerDomains[ domainName[ 11 : ] ] = \
            PowerManagerModel.PowerDomain( powerSupply=domain.powerSupply[ 11 : ],
                                           interfaces=list( domain.poePort ) )
   return model

#-----------------------------------------------------------------------------
# [ no | default ] environment insufficient-power action [ shutdown | ignore ]
#-----------------------------------------------------------------------------
def envInsufficientPowerActionHandler( mode, args ):
   if "ignore" in args:
      powerManagerCliConfig.modularMode = managementMode.warn
   elif "shutdown" in args:
      powerManagerCliConfig.modularMode = managementMode.strict

def envInsufficientPowerActionNoOrDefaultHandler( mode, args ):
   powerManagerCliConfig.modularMode = managementMode.strict

#-----------------------------------------------------------------------------
# show system environment power histogram detail
#-----------------------------------------------------------------------------
def showSystemEnvPowerHistogramDetailHandler( mode, args ):
   percentPerBucket = 5
   numBuckets = int( math.ceil(
                     len( powerManager.histogram ) / percentPerBucket ) )
   powerHistogramDict = {}
   for idx in range( numBuckets ):
      powerHistogramDict[ idx * percentPerBucket ] = 0
   for percent, uptime in powerManager.histogram.items():
      idx = percent // percentPerBucket
      bucketPercent = idx * percentPerBucket
      powerHistogramDict[ bucketPercent ] += int( uptime )
   return PowerManagerModel.PowerHistogramDetail(
      buckets=powerHistogramDict,
      totalPower=powerManager.totalPower )
#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global entityMib
   global powerManager
   global powerManagerHwConfig
   global powerManagerCliConfig

   entityMib = LazyMount.mount( entityManager, 'hardware/entmib',
                                'EntityMib::Status', 'r' )
   powerManager = LazyMount.mount( entityManager,
                                   'environment/archer/power/status/powerManager',
                                   'PowerManager::PowerManagerStatus', 'r' )
   powerManagerHwConfig = LazyMount.mount( entityManager,
                          'environment/archer/power/config/powerManager',
                          'PowerManager::PowerManagerConfig', 'r' )
   powerManagerCliConfig = LazyMount.mount( entityManager,
                           'environment/archer/power/config/powerManagerCliConfig',
                           'PowerManager::PowerManagerCliConfig', 'w' )
