# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArPyUtils
from CliModel import Dict, List, Model, Str, Float
from IntfModels import Interface
from TableOutput import Format, createTable

class PowerDomain( Model ):
   powerSupply = Str( help="Power supply name" )
   interfaces = List( valueType=Interface,
                      help="List of interfaces in power domain" )

def makeTable( header ):
   t = createTable( header )
   fr = Format( justify="right" )
   fr.noPadLeftIs( True )
   fr.padLimitIs( True )
   fl = Format( justify="left", maxWidth=60, wrap=True )
   fl.noPadLeftIs( True )
   fl.padLimitIs( True )
   t.formatColumns( fr, fr, fl )
   return t

class PowerDomainCollection( Model ):
   powerDomains = Dict( optional=True, keyType=str, valueType=PowerDomain,
                        help="Power domains, keyed by name" )

   def render( self ):
      sortedDomainNames = ArPyUtils.naturalsorted( self.powerDomains )
      tableColumns = [ [ "Power", [ "Domain" ], ],
                       [ "Power", [ "Supply" ], ],
                       [ "Interfaces" ] ]
      table = makeTable( tableColumns )
      for domain in sortedDomainNames:
         sortedIntfs = \
               ArPyUtils.naturalsorted( self.powerDomains[ domain ].interfaces )
         table.newRow( domain,
                       self.powerDomains[ domain ].powerSupply,
                       ", ".join( sortedIntfs ) )
      print( '\n'.join( map( str.rstrip, table.output().split( '\n' )[ : -1 ] ) ) )

class PowerHistogramDetail( Model ):
   buckets = Dict( keyType=int, valueType=int,
                            help=( "Mapping of power ranges (in percent) to "
                                   "uptime (seconds) in the range" ) )
   totalPower = Float( help="Total system power in watts" )

   def render( self ):
      sortedBuckets = sorted( self.buckets.items(), key=lambda kv: kv[ 0 ] )
      powerPerBucket = self.totalPower / len( sortedBuckets )
      totalUptime = sum( self.buckets.values() )
      hours = totalUptime // 3600
      minutes = ( totalUptime % 3600 ) // 60
      print( ( f"\nTotal system power: {self.totalPower}W\n"
               f"System uptime: {hours} hrs {minutes} minutes\n" ) )
      hdr = ( ( "% Power", "r" ),
              ( "Power Range", "r" ),
              ( "Time", "r" ),
              ( "% Uptime", "r" ) )
      table = createTable( hdr )
      for idx, bucket in enumerate( sortedBuckets ):
         bucketLow = int( idx * powerPerBucket + 1 )
         bucketHigh = int( ( idx + 1 ) * powerPerBucket )
         uptimePercent = round( bucket[ 1 ] / totalUptime * 100 )
         table.newRow( f"{bucket[ 0 ]}%",
                       f"{bucketLow}-{bucketHigh}W",
                       f"{bucket[ 1 ]}s",
                       f"{uptimePercent}%" )
      fRight = Format( justify="right" )
      fRight.noPadLeftIs( True )
      fRight.padLimitIs( True )
      table.formatColumns( fRight, fRight, fRight, fRight )
      print( table.output(), end="" )
