# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys

import CliDynamicSymbol
from CliPlugin import NetworkToolsCli
from CliPlugin.ProbeCli import config
import CmdExtension
from IpLibConsts import DEFAULT_VRF
import LazyMount
import Tac

ProbeCode = Tac.Type( 'IcmpResponder::ProbeClient::ProbeCode' )

def doProbe( mode, args ):
   cliCmdExt = CmdExtension.getCmdExtender()
   destination = str( args[ 'DESTINATION' ] )

   srcProtoVrf = NetworkToolsCli.doResolveSrcProtoVrf( mode, args, destination,
                                                       networkTool='probe' )
   if srcProtoVrf is None:
      # there was an error, just return
      return
   src, proto, vrf = srcProtoVrf

   pArgs = [ 'arsudo', 'ProbeClient' ]
   if 'IFNAME' in args:
      pArgs += [ '--ifname', args[ 'IFNAME' ] ]
   elif 'IFINDEX' in args:
      pArgs += [ '--ifindex', str( args[ 'IFINDEX' ] ) ]
   elif 'ADDR' in args:
      pArgs += [ '--addr', str( args[ 'ADDR' ] ) ]
   elif 'NEIGHBOR' in args:
      pArgs += [ '--remote', '--addr', str( args[ 'NEIGHBOR' ] ) ]
   else:
      mode.addError( 'one of ifindex, ifname, ifaddr, remote is required' )
      return

   if 'COUNT' in args:
      pArgs += [ '--count', str( args[ 'COUNT' ][ 0 ] ) ]
   if 'INTERVAL' in args:
      pArgs += [ '--wait', str( args[ 'INTERVAL' ][ 0 ] ) ]
   if src:
      pArgs += [ '--source', src ]

   if proto == 'ipv4':
      pArgs += [ '-4' ]
   elif proto == 'ipv6':
      pArgs += [ '-6' ]

   pArgs += [ '--', destination ]

   try:
      p = cliCmdExt.subprocessPopen( pArgs, mode.session, vrfName=vrf,
                                     stdout=sys.stdout, stderr=sys.stderr )
      p.communicate()
   except OSError as e:
      mode.addError( e.strerror )

status = None

ProbeModel = CliDynamicSymbol.CliDynamicPlugin( "ProbeModel" )

def showProbe( mode, args ):
   m = ProbeModel.ProbeResponderModel()
   # Reflecting configuration
   m.enabled = config.enabled
   m.allowIfname = config.allowIfname
   m.allowIfindex = config.allowIfindex
   m.allowAddr = config.allowAddr
   m.allowRemote = config.allowRemote
   # Reflecting status
   m.requestsReceived = status.requestsReceived
   m.queueTooLong = status.queueTooLong
   m.responsesSent = status.responsesSent
   m.pktTooShort = status.pktTooShort
   m.incorrectIcmpChecksum = status.incorrectIcmpChecksum
   m.badExtensionHeader = status.badExtensionHeader
   m.incorrectExtensionChecksum = status.incorrectExtensionChecksum
   m.badExtensionObject = status.badExtensionObject
   m.requestTypeNotAllowed = status.requestTypeNotAllowed
   m.interfaceLookupFailed = status.interfaceLookupFailed
   m.wrongVrfRequested = status.wrongVrfRequested
   m.remoteRequestMalformed = status.remoteRequestMalformed
   m.txErrors = status.txErrors
   m.lastTxError = ( os.strerror( status.lastTxErrno ) if status.lastTxErrno
                                                       else "none" ).lower()
   #
   for code, count in status.codeCount.items():
      histo = ProbeModel.ProbeResponderCodeModel()
      histo.description = ProbeCode( code ).humanReadable()
      histo.count = count
      m.codes[ code ] = histo

   vrfModel = ProbeModel.ProbeResponderVrfModel()
   vrfModel.vrfs[ DEFAULT_VRF ] = m
   return vrfModel

def Plugin( entityManager ):
   global status
   status = LazyMount.mount( entityManager, "icmp/probe/status",
                             "IcmpResponder::ProbeResponder::Status", "r" )
