# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
import BasicCli
from BasicCliUtil import anyCaseRegex
from BgpLib import getVpwsEviName
import CliCommand
from CliDynamicSymbol import CliDynamicPlugin
import CliMatcher
from CliMatcher import DynamicNameMatcher
from CliMode.Pseudowire import (
   FxcMode,
   LdpPseudowireMode,
   StaticPseudowireMode,
   LdpPseudowireProfileMode,
   MplsLdpPseudowiresMode,
   MplsPseudowiresMode,
   MplsStaticPseudowiresMode,
   PatchMode,
   PatchPanelMode,
   PatchQosBaseMode,
)
from CliPlugin import (
   EthIntfCli,
   LagIntfCli,
   MplsCli,
   SubIntfCli,
   SwitchIntfCli,
   TunnelIntfCli,
   IraNexthopGroupCli,
)
from CliPlugin import IpAddrMatcher
from CliPlugin.IpRibLibCliModels import ResolutionRibProfileConfig
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliPlugin.IpRibLib import (
   ResolutionRibsExprFactory,
)
from CliPlugin.Pseudowire import (
   bgpPseudowireNameMatcher,
   labelMatcher,
   mtuMatcher,
   mtuNumMatcher,
   MplsStaticPseudowiresCmd,
)
from CliPlugin.PwaCli import (
   getPwProfiles,
   ldpPwNameMatcher,
   ldpPwProfileMatcher,
   mplsStaticPwNameMatcher,
   pwIdMatcher,
   staticPwNameMatcher,
)
from CliPlugin.TunnelRibCli import (
   systemColoredTunnelRibName,
   systemTunnelRibName,
)
from PseudowireLib import (
   COLOR_MIN,
   COLOR_MAX,
   ConnectorType,
   CosToTcType,
   DscpToTcType,
   FlowLabelMode,
   PwPingCcType,
   PwPingCvType,
   QosMap,
   QosMapType,
   connIndex0,
   connIndex1,
   connIndex2,
   getBgpConnector,
   getBgpConnectorKey,
   getInterfaceConnector,
   getInterfaceConnectorKey,
   getLdpAutoDiscoveryProfileConnectorKey,
   getLdpConnectorKey,
   getMplsStaticConnector,
   getMplsStaticConnectorKey,
   getMplsLdpConnector,
   getAlternateConnectorName,
   pwPingCcEnumToVal,
   pwPingCvEnumToVal,
   validPwNameRegex as validNameRegex,
   vlanIdAny as portConnVlanId,
)
import CliParser
import ConfigMount
import LazyMount
import Tac
from Toggles.PseudowireToggleLib import (
      toggleVplsBgpSignalingEnabled,
      toggleVpwsStandbyErrDisableEnabled,
      toggleStaticMplsPwNhgTransportEnabled,
)
import Tracing
from ArPyUtils.Decorators import (
   traced,
   tracedFuncName,
)
from TypeFuture import TacLazyType

th = Tracing.Handle( 'PseudowireCli' )
t0 = th.t0
t1 = th.t1

CliConnector = TacLazyType( 'Pseudowire::CliConnector' )
CliFxcHelper = TacLazyType( 'PseudowireAgent::CliFxcHelper' )
PatchType = TacLazyType( 'Pseudowire::Patch' )
PseudowireBgpResponseKey = TacLazyType( 'Pseudowire::PseudowireBgpResponseKey' )
PseudowireType = TacLazyType( 'Pseudowire::PseudowireType' )
ArnetIntfId = TacLazyType( 'Arnet::IntfId' )
ArnetMplsLabel = TacLazyType( 'Arnet::MplsLabel' )
NexthopGroupName = TacLazyType( 'Tac::String' )

PseudowireProfileCli = None
if toggleVplsBgpSignalingEnabled():
   PseudowireProfileCli = CliDynamicPlugin( 'PseudowireProfileCli' )

# Depending on ArBgp-cli is needed for BGP VPWS (forced dependency, see AID10)
# pkgdeps: import CliPlugin.BgpMacVrfConfigCli

pwConfig = None
pwRunnability = None
pwHwCapability = None
pwaMockLdpStatus = None
pwaMockBgpStatus = None
bgpMacVrf = None

# Common tokens
globalConnectorMatcher = CliMatcher.KeywordMatcher( 'connector',
      helpdesc='Connector' )
connectorMatcher = CliMatcher.KeywordMatcher( 'connector',
      helpdesc='Configure a patch connector' )
bgpMatcher = CliMatcher.KeywordMatcher( 'bgp', helpdesc="BGP pseudowire" )
vpwsMatcher = CliMatcher.KeywordMatcher( 'vpws', helpdesc="EVPN VPWS pseudowire" )

def mplsStaticLabelRangeFn( mode, context ):
   labelRange = MplsCli.labelRangeValue( MplsCli.LabelRangeInfo.rangeTypeStatic )
   labelMin = labelRange.base
   labelMax = labelRange.base + labelRange.size - 1
   return labelMin, labelMax

def mplsLabelRangeFn( mode, context ):
   return ArnetMplsLabel.min, ArnetMplsLabel.max

mplsStaticLabelMatcher = CliMatcher.DynamicIntegerMatcher(
      rangeFn=mplsStaticLabelRangeFn,
      helpdesc='Value of the local label' )

mplsLabelMatcher = CliMatcher.DynamicIntegerMatcher(
      rangeFn=mplsLabelRangeFn,
      helpdesc='Value of the neighbor label' )

# Allow any connector name but pseudowire and interface
disallowRegex = anyCaseRegex( "(pseudowire|interface)" )
connNameRegex = f"^(?!{disallowRegex}$){validNameRegex}$"
connNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [ conn.name for conn in
                     pwConfig.patch[ mode.patchName ].connector.values() ]
      if mode.patchName in pwConfig.patch else [], 'Connector name',
      pattern=connNameRegex )

# Qos Tokens
matcherDscpMapName = CliMatcher.DynamicNameMatcher(
      lambda mode: [ connector.qosMap.mapName for connector in
                     pwConfig.connector.values()
                     if connector.qosMap.mapType == QosMapType.pwDecapDscpToTc ],
      helpdesc='DSCP to Traffic-Class Map Name',
      pattern=r'(?!default-map$)[A-Za-z0-9_:{}\[\]-]+' )

matcherCosMapName = CliMatcher.DynamicNameMatcher(
      lambda mode: [ connector.qosMap.mapName for connector in
                     pwConfig.connector.values()
                     if connector.qosMap.mapType == QosMapType.pwDecapCosToTc ],
      helpdesc='COS to Traffic-Class Map Name',
      pattern=r'(?!default-map$)[A-Za-z0-9_:{}\[\]-]+' )

def flowLabelSupportedGuard( mode, token ):
   if pwHwCapability.flowLabelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def pwDecapDscpQosMapSupportedGuard( mode, token ):
   if pwHwCapability.pwDecapDscpQosMapSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def pwDecapCosQosMapSupportedGuard( mode, token ):
   if pwHwCapability.pwDecapCosQosMapSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsStaticPwSupportedGuard( mode, token ):
   if pwHwCapability.mplsStaticPseudowireSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

# -------------------------------------------------------------------------------
# Patch panel config mode and related helpers
# -------------------------------------------------------------------------------

class PatchPanelConfigMode( PatchPanelMode, BasicCli.ConfigModeBase ):
   name = "Patch panel configuration"

   _nameErrorFmt = 'Name "{}" is already in use as a {}'

   def __init__( self, parent, session ):
      PatchPanelMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   @tracedFuncName( trace=t1 )
   def gotoPatchMode( self, args, funcName=None ):
      patchName = args[ 'PATCH_NAME' ]
      existingPatch = pwConfig.patch.get( patchName )
      if existingPatch and existingPatch.fxc:
         error = self._nameErrorFmt.format( patchName, "flexible cross connect" )
         t0( funcName, 'ERROR:', error )
         self.addError( error )
         return
      childMode = self.childMode( PatchConfigMode, patchName=patchName )
      self.session_.gotoChildMode( childMode )

   def noPatch( self, args ):
      patchName = args[ 'PATCH_NAME' ]
      if patchName in pwConfig.patch:
         noPwPatchQosInfo( patchName )

      del pwConfig.patch[ patchName ]
      pwRunnability.config = bool( pwConfig.patch )

   @tracedFuncName( trace=t1 )
   def gotoFxcMode( self, args, funcName=None ):
      patchName = args[ 'NAME' ]
      existingPatch = pwConfig.patch.get( patchName )
      if existingPatch and not existingPatch.fxc:
         error = self._nameErrorFmt.format( patchName, "patch" )
         t0( funcName, 'ERROR:', error )
         self.addError( error )
         return
      childMode = self.childMode( FlexibleCrossConnectConfigMode,
                                  patchName=patchName )
      self.session_.gotoChildMode( childMode )

   def noFxc( self, args ):
      t0( 'noFxc', repr( args ) )
      patchName = args[ 'NAME' ]
      del pwConfig.patch[ patchName ]
      pwRunnability.config = bool( pwConfig.patch )

def gotoPatchPanelMode( mode, args ):
   childMode = mode.childMode( PatchPanelConfigMode )
   mode.session_.gotoChildMode( childMode )

def noPatchPanel( mode, args ):
   for patchName in pwConfig.patch:
      noPwPatchQosInfo( patchName )

   pwConfig.patch.clear()
   pwConfig.intfReviewDelayRange = pwConfig.intfReviewDelayRangeDefault
   pwConfig.patchBgpVpwsErrdisable = pwConfig.patchBgpVpwsErrdisableDefault
   pwConfig.bgpVpwsStandbyErrdisable = pwConfig.bgpVpwsStandbyErrdisableDefault
   pwRunnability.config = False

def updateMtu( value ):
   if not value:
      value = pwConfig.getMtuDefault()
   pwConfig.mtu = value

class PatchConfigModeHelper:
   def __init__( self, mode, patchName, patchClassDescription ):
      """
      Create a helper object for the patch / FXC config mode for `patchName`

      patchClassDescription is used in CLI error messages and describes the context
      in which the helper is used, e.g. "Patch":
       --> "Patch" patchName "is not present"
      """
      self.mode = mode
      self.patchName = patchName
      self.patchClassDescription = patchClassDescription

   def patchPresentOrError( self, patch ):
      # Ground has shifted underneath our feet. All subsequent commands
      # should fail.
      if not patch:
         self.mode.addError( "{} {} not present".format( self.patchClassDescription,
                                                         self.patchName ) )
         return False
      return True

def addPseudowireAttribute( connKey, nbrAddr=None, keepNbrAddr=None,
                            pwId=1, keepPwId=None,
                            mtu=0, keepMtu=None,
                            color=None, colorPresent=None,
                            controlWord=False, keepControlWord=None,
                            flowLabelMode=FlowLabelMode.disabled,
                            pwType=None, keepFlowLabelMode=None,
                            transport=None,
                            qosMap=QosMap(),
                            keepQosMap=None,
                            patchQosMode=False,
                            keepPatchQosMode=None,
                            staticPwConfigMode=False,
                            keepStaticPwConfigMode=None,
                            ldpPwConfigMode=False,
                            keepLdpPwConfigMode=None,
                            resolutionRibProfileConfig=None,
                            localLabel=None,
                            neighborLabel=None,
                            nexthopGroupTransport=None, ):
   oldConnector = pwConfig.connector[ connKey ]

   def newAttr( oldVal, defaultVal, newVal, keep ):
      attr = oldVal
      if keep is True:
         attr = newVal
      elif keep is False:
         attr = defaultVal
      return attr

   newNbrAddr = newAttr( oldConnector.neighborAddr,
                         Tac.Value( "Arnet::IpGenAddr" ),
                         nbrAddr, keepNbrAddr )
   newNbrAddrPresent = newAttr( oldConnector.neighborAddrPresent,
                                False,
                                keepNbrAddr, keepNbrAddr )

   newPwId = newAttr( oldConnector.pwId, 0, pwId, keepPwId )
   newPwIdPresent = newAttr( oldConnector.pwIdPresent,
                             False,
                             keepPwId, keepPwId )

   newMtu = newAttr( oldConnector.mtu, 0, mtu, keepMtu )
   newMtuPresent = newAttr( oldConnector.mtuPresent,
                            False,
                            keepMtu, keepMtu )

   newColor = newAttr( oldConnector.color, 0, color, colorPresent )
   newColorPresent = newAttr( oldConnector.colorPresent, False,
                              colorPresent, colorPresent )

   newControlWord = newAttr( oldConnector.controlWord,
                             False,
                             controlWord, keepControlWord )

   newFlowLabelMode = newAttr( oldConnector.flowLabelMode,
                               FlowLabelMode.disabled,
                               flowLabelMode, keepFlowLabelMode )

   if pwType is None:
      newPwType = oldConnector.pwType
   else:
      newPwType = pwType

   if transport is None:
      transport = oldConnector.transportIntf

   if localLabel is None:
      localLabel = oldConnector.localLabel

   if neighborLabel is None:
      neighborLabel = oldConnector.neighborLabel

   if nexthopGroupTransport is None:
      nexthopGroupTransport = oldConnector.nexthopGroupName

   newQosMap = newAttr( oldConnector.qosMap, QosMap(),
                        qosMap, keepQosMap )

   newPatchQosMode = newAttr( oldConnector.patchQosMode, False,
                              patchQosMode, keepPatchQosMode )

   newLdpPwConfigMode = newAttr( oldConnector.ldpPwConfigMode, False,
                                 ldpPwConfigMode, keepLdpPwConfigMode )

   newStaticPwConfigMode = newAttr( oldConnector.staticPwConfigMode, False,
                                 staticPwConfigMode, keepStaticPwConfigMode )

   if resolutionRibProfileConfig is None:
      newResolutionRibProfileConfig = oldConnector.resolutionRibProfileConfig
   else:
      newResolutionRibProfileConfig = resolutionRibProfileConfig

   pwConfig.connector[ connKey ] = Tac.Value( "Pseudowire::Connector",
                 neighborAddr=newNbrAddr,
                 neighborAddrPresent=newNbrAddrPresent,
                 pwId=newPwId,
                 pwIdPresent=newPwIdPresent,
                 mtu=newMtu,
                 mtuPresent=newMtuPresent,
                 color=newColor,
                 colorPresent=newColorPresent,
                 controlWord=newControlWord,
                 flowLabelMode=newFlowLabelMode,
                 pwType=newPwType,
                 transportIntf=transport,
                 qosMap=newQosMap,
                 patchQosMode=newPatchQosMode,
                 ldpPwConfigMode=newLdpPwConfigMode,
                 staticPwConfigMode=newStaticPwConfigMode,
                 resolutionRibProfileConfig=newResolutionRibProfileConfig,
                 localLabel=localLabel,
                 neighborLabel=neighborLabel,
                 nexthopGroupName=nexthopGroupTransport, )

def noPwPatchQosInfo( patchName ):
   for connName in pwConfig.patch[ patchName ].connector:
      connKey = pwConfig.patch[ patchName ].connector[ connName ].connectorKey
      if connKey.connectorType == 'local' or connKey not in pwConfig.connector:
         continue
      if connKey.connectorType == 'bgp':
         del pwConfig.connector[ connKey ]
      elif connKey.connectorType == 'mplsStatic':
         if pwConfig.connector[ connKey ].staticPwConfigMode is False:
            del pwConfig.connector[ connKey ]
      else:
         addPseudowireAttribute( connKey, keepQosMap=False,
                                 keepPatchQosMode=False )
         if pwConfig.connector[ connKey ].ldpPwConfigMode is False:
            del pwConfig.connector[ connKey ]

class PatchQosMode( PatchQosBaseMode, BasicCli.ConfigModeBase ):
   name = "Qos Configuration on Pseudowire"

   def __init__( self, parent, session, patchName, connName, pwName, vpwsName=None,
                 altPwName=None ):
      PatchQosBaseMode.__init__( self, patchName, connName, pwName=pwName,
                                 vpwsName=vpwsName, altPwName=altPwName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setPwDecapDscpQosMap( self, args ):
      mapName = args.get( 'MAP_NAME' )
      qosMap = QosMap( mapName, DscpToTcType )
      # this could even be extended furthur if we want to support multiple alternates
      # within a single ldp connector
      connKeyList = []
      if self.connType == 'bgp':
         connKeyList.append( getBgpConnectorKey( self.vpwsName, self.pwName ) )
      else:
         assert self.connType == 'ldp'
         connKeyList.append( getLdpConnectorKey( self.pwName ) )
         if self.altPwName:
            connKeyList.append( getLdpConnectorKey( self.altPwName ) )

      for connKey in connKeyList:
         if connKey not in pwConfig.connector:
            pwConfig.connector[ connKey ] = Tac.Value( "Pseudowire::Connector",
                        qosMap=qosMap,
                        patchQosMode=( self.connType == 'ldp' ) )
         elif ( pwConfig.connector[ connKey ].qosMap.mapType !=
                QosMapType.pwDecapCosToTc ):
            if self.connType == 'bgp':
               pwConfig.connector[ connKey ] = Tac.Value( "Pseudowire::Connector",
                                                          qosMap=qosMap )
            else:
               addPseudowireAttribute( connKey, qosMap=qosMap, keepQosMap=True,
                                       patchQosMode=True, keepPatchQosMode=True )

   def noPwDecapDscpQosMap( self, args ):
      connKeyList = []
      if self.connType == 'bgp':
         connKeyList.append( getBgpConnectorKey( self.vpwsName, self.pwName ) )
      else:
         assert self.connType == 'ldp'
         connKeyList.append( getLdpConnectorKey( self.pwName ) )
         if self.altPwName:
            connKeyList.append( getLdpConnectorKey( self.altPwName ) )

      for connKey in connKeyList:
         if ( connKey in pwConfig.connector and
            pwConfig.connector[ connKey ].qosMap.mapType ==
                  QosMapType.pwDecapDscpToTc ):
            if self.connType == 'bgp':
               del pwConfig.connector[ connKey ]
            else:
               addPseudowireAttribute( connKey, keepQosMap=False,
                                       keepPatchQosMode=False )
               if pwConfig.connector[ connKey ].ldpPwConfigMode is False:
                  del pwConfig.connector[ connKey ]

   def setPwDecapCosQosMap( self, args ):
      mapName = args.get( 'MAP_NAME' )
      qosMap = QosMap( mapName, CosToTcType )
      connKeyList = []
      if self.connType == 'bgp':
         connKeyList.append( getBgpConnectorKey( self.vpwsName, self.pwName ) )
      else:
         assert self.connType == 'ldp'
         connKeyList.append( getLdpConnectorKey( self.pwName ) )
         if self.altPwName:
            connKeyList.append( getLdpConnectorKey( self.altPwName ) )

      for connKey in connKeyList:
         if connKey not in pwConfig.connector:
            pwConfig.connector[ connKey ] = Tac.Value( "Pseudowire::Connector",
                        qosMap=qosMap,
                        patchQosMode=( self.connType == 'ldp' ) )
         elif pwConfig.connector[ connKey ].qosMap.mapType != \
               QosMapType.pwDecapDscpToTc:
            if self.connType == 'bgp':
               pwConfig.connector[ connKey ] = Tac.Value( "Pseudowire::Connector",
                                                          qosMap=qosMap )
            else:
               addPseudowireAttribute( connKey, qosMap=qosMap, keepQosMap=True,
                                       patchQosMode=True, keepPatchQosMode=True )

   def noPwDecapCosQosMap( self, args ):
      connKeyList = []
      if self.connType == 'bgp':
         connKeyList.append( getBgpConnectorKey( self.vpwsName, self.pwName ) )
      else:
         assert self.connType == 'ldp'
         connKeyList.append( getLdpConnectorKey( self.pwName ) )
         if self.altPwName:
            connKeyList.append( getLdpConnectorKey( self.pwName ) )

      for connKey in connKeyList:
         if connKey in pwConfig.connector and \
               pwConfig.connector[ connKey ].qosMap.mapType == \
               QosMapType.pwDecapCosToTc:
            if self.connType == 'bgp':
               del pwConfig.connector[ connKey ]
            else:
               addPseudowireAttribute( connKey, keepQosMap=False,
                                       keepPatchQosMode=False )
               if pwConfig.connector[ connKey ].ldpPwConfigMode is False:
                  del pwConfig.connector[ connKey ]

class PatchConfigMode( PatchMode, BasicCli.ConfigModeBase ):
   name = "Patch configuration"

   def __init__( self, parent, session, patchName ):
      PatchMode.__init__( self, patchName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      self.modeHelper = PatchConfigModeHelper( self, patchName, "Patch" )

      if patchName not in pwConfig.patch:
         patch = pwConfig.newPatch( patchName )
         patch.fxc = False
         pwRunnability.config = True
      else:
         if pwConfig.patch[ patchName ].fxc:
            assert not pwConfig.patch[ patchName ].fxc, "name {} was reused".format(
               patchName )

   def nextConnectorIndex( self, patch ):
      '''Assumes number of connectors in the patch are at most 2'''
      if not connIndex0 in patch.connector:
         return connIndex0
      if not connIndex1 in patch.connector:
         return connIndex1

      assert False, "Both indexes can't be occupied"
      return None

   def findConnector( self, cliConnector, patch, ignoreConnName=False ):
      for connIndex in patch.connector:
         if ( cliConnector == patch.connector[ connIndex ] or
              ( ignoreConnName and cliConnector.connectorKey ==
                patch.connector[ connIndex ].connectorKey ) ):
            return connIndex

      return None

   def patchPresentOrError( self, patch ):
      return self.modeHelper.patchPresentOrError( patch )

   def gotoPatchQosMode( self, connName, pwName, vpwsName=None, altPwName=None ):
      childMode = self.childMode( PatchQosMode, patchName=self.patchName,
                                  connName=connName, pwName=pwName,
                                  vpwsName=vpwsName, altPwName=altPwName )
      self.session_.gotoChildMode( childMode )

   # -------------------------------------------------------------------
   # Common connector related methods
   # -------------------------------------------------------------------
   def getConnectors( self, getIntfConnectors, connName=None, dot1qTag=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return None

      connectorList = []
      if getIntfConnectors:
         # If getting interface connectors, filter out any LDP pseudowire connectors
         connectorList = {
             connIndex: connector
            for connIndex, connector in patch.connector.items()
            if connector.interface }
         # If dot1qTag/connName is specified,
         # then filter out remaining entries based on connName/dot1qTag
         if connName:
            connectorList = {
                connIndex: connector
               for connIndex, connector in connectorList.items()
               if str( connector.interface ) == str( connName ) }
         if dot1qTag:
            connectorList = {
                connIndex: connector
               for connIndex, connector in connectorList.items()
               if connector.dot1qTag == dot1qTag }
      else:
         # If getting LDP pseudowire connectors, filter out any interface connectors
         connectorList = {
             connIndex: connector
            for connIndex, connector in patch.connector.items()
            if not connector.interface }
         # If connName is specified,
         # then filter out remaining entries based on connName
         if connName:
            connectorList = {
                connIndex: connector
               for connIndex, connector in connectorList.items()
               if connector.connectorKey.name == connName }

      return connectorList

   def noConnectorName( self, connName ):
      '''Delete all connectors in the patch with the name connName.'''
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      for key in patch.connector:
         if patch.connector[ key ].name == connName:
            del patch.connector[ key ]

   def pickDefaultConnectorName( self, patch, alt=False ):
      '''
      Return default connector names of "1" and "2" if none are provided.
      If "1" is already in use, return "2". Returns "" if 2 connectors
      are already defined.

      For patch with alternate Connector, try to find the primary ldp connector name 
      and return <name>@Alternate, if no primary ldp connector, return ""
      '''

      # with ldp redundancy feature there could be three connectors in patch
      if len( patch.connector ) == 3:
         return ""
      elif len( patch.connector ) == 2:
         # case is more complicate with remote primary and alternate
         # Note: remote primary will always be added before remote alternate
         remoteCount = 0
         remotePrimary = 0
         remoteAlternate = 0
         for c in patch.connector.values():
            if c.connectorKey.connectorType != ConnectorType.local:
               remoteCount += 1
               if c.alternate:
                  remoteAlternate += 1
               else:
                  remotePrimary += 1
         if not alt and remoteCount == 2:
            # adding localConnector
            pass
         elif alt and remoteAlternate == 0:
            # adding remote alternate
            pass
         else:
            return ""

      if alt:
         primaryName = None
         for c in patch.connector.values():
            if c.connectorKey.connectorType == ConnectorType.ldp:
               primaryName = c.name
         return getAlternateConnectorName( primaryName )
      else:
         found1 = any( c.name == "1" for c in patch.connector.values() )
         return "2" if found1 else "1"

   # BUG662924, Combine hasConnectorInPatch and hasConnectorInPatchUpdated
   # into a single function to use CliConnector.isEqual
   def hasConnectorInPatch( self, patch, connName, connectorKey,
                            noErrDisable=False ):
      '''Check if there is already a connector in the patch with the given
         connName (if passed) and connectorKey. Passing a valid connName is optional,
         if it passed as None or empty string, then the connectorKey and
         noErrDisable config is checked.
      '''
      for connector in patch.connector.values():
         if connName:
            if connector.name == connName:
               return ( connector.connectorKey == connectorKey and
                        connector.noErrDisable == noErrDisable )
         else:
            if ( connector.connectorKey == connectorKey and
                 connector.noErrDisable == noErrDisable ):
               return True
      return False

   def hasConnectorInPatchUpdated( self, patch, connName, cliConnector ):
      '''Check if there is already a connector in the patch with the given
         connName (if passed) and CliConnector. Passing a valid connName is optional,
         if it passed as None or empty string, then the connectorKey and other
         config attributes are checked.
      '''
      for connector in patch.connector.values():
         if connector.isEqual( cliConnector, not connName ):
            return True
      return False

   # -------------------------------------------------------------------
   # Connector interface/local connector related methods
   # -------------------------------------------------------------------

   def canAddInterfaceConnector( self, patch, newConnName, newConnKey ):
      numConnectorsInPatch = 0

      # Check if an incompatible connector is configured in this patch
      for cliConnector in patch.connector.values():
         if cliConnector.name == newConnName:
            # this connector is going to be replaced, therefore ignore its presence
            continue
         if cliConnector.connectorKey == newConnKey:
            # this connector is going to be renamed, therefore ignore its presence
            continue
         # alternate connector will not be counted as it is duo with primary one
         if not cliConnector.alternate:
            numConnectorsInPatch += 1
         if cliConnector.connectorKey.isLocal():
            if newConnKey.isSubIntf() != cliConnector.connectorKey.isSubIntf():
               self.addError( "Local to local patch with only one subinterface "
                              "connector not supported" )
               return False
         elif ( newConnKey.isLegacyTagged() and
                cliConnector.connectorKey.connectorType == ConnectorType.bgp ):
            self.addError( "Patch with BGP VPWS and tagged connectors "
                           "not supported" )
            return False

      if numConnectorsInPatch == 2:
         self.addError( "Patch has two connectors already" )
         return False

      return True

   def setInterfaceConnector( self, intf, dot1qTag=None, connName=None,
                              noErrDisable=False ):
      '''
      The caller is trying to add a new CliConnector (with or without
      the connector name specified) or modify in-place an existing 
      CliConnector (connector name required).

      Note that giving the name of an existing connName necessarily
      overwrites it and doesn't add another connector with the same name.
      '''
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      # If not dot1qTag is provided, assume it's a port connector
      if dot1qTag is None:
         dot1qTag = portConnVlanId

      connKey = getInterfaceConnectorKey( intf, dot1qTag=dot1qTag )

      tempConnector = getInterfaceConnector( intf, dot1qTag, connName or '',
                                             noErrDisable=noErrDisable )

      # Backwards compatibility,
      # all interfaces have noerrDisable set to default false
      noErrDisableExisting = False
      existingConnector = None
      existingIndex = None

      for index, connector in patch.connector.items():
         if connector.isEqual( tempConnector, not connName ):
            # If the new connector is identical to the original one, then don't
            # do anything to avoid churn
            return

         if connector.connectorKey == tempConnector.connectorKey:
            existingIndex = index
            existingConnector = connector
            noErrDisableExisting = connector.noErrDisable
            break

      if not self.canAddInterfaceConnector( patch, connName, connKey ):
         return

      # If a connector is configured with the same connName and connKey value
      # just the noErrDisable flag is different then update the value instead of
      # add/delete to prevent churn
      if ( existingConnector is not None and existingConnector.name == connName and
           noErrDisableExisting != noErrDisable ):
         cliConnector = getInterfaceConnector( intf, dot1qTag, connName,
                                               noErrDisable=noErrDisable )
         patch.connector[ existingIndex ] = cliConnector
         return

      # To update the existing connector both noInterfaceConnector and
      # noConnectorName need to be called
      # Example:
      # patch foo
      #    connector 1 interface Ethernet2.5
      #    connector 2 pseudowire ldp pw6
      # If a new connector is configured in the patch
      #    connector 2 interface Ethernet2.5
      # noInterfaceConnector will delete config for existing Ethernet2.5
      # noConnectorName will delete the config for the ldp pw
      # The final config will be
      # patch foo
      #    connector 2 interface Ethernet2.5
      # Just calling only one will not update the config correctly

      # Delete existing connector with same key, as it is going to be renamed
      self.noInterfaceConnector( intf, dot1qTag, noErrDisable=noErrDisableExisting )

      if connName:
         # Delete existing connector with same name, as it is going to be replaced
         self.noConnectorName( connName )

         # Delete existing alternate connector
         # This is needed when overwrite an ldp connector with alternate to local one
         altConnName = getAlternateConnectorName( connName )
         self.noConnectorName( altConnName )
      else:
         connName = self.pickDefaultConnectorName( patch )
         assert connName

      # connector can be added successfully
      cliConnector = getInterfaceConnector( intf, dot1qTag, connName,
                                            noErrDisable=noErrDisable )
      connIndex = self.nextConnectorIndex( patch )
      patch.connector[ connIndex ] = cliConnector

   def noInterfaceConnector( self, intf, dot1qTag=None, connName=None,
                             noErrDisable=False ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      # If not dot1qTag is provided, assume it's a port connector
      if dot1qTag is None:
         dot1qTag = portConnVlanId
      delCliConn = getInterfaceConnector( intf, dot1qTag, connName or "",
                                          noErrDisable=noErrDisable )

      for connIndex, cliConn in patch.connector.items():
         # If connName is provided, it must also match
         if cliConn.isEqual( delCliConn, not connName ):
            del patch.connector[ connIndex ]
            break

   # -------------------------------------------------------------------
   # Remote pseudowire connector related methods
   # -------------------------------------------------------------------
   def canAddRemoteConnector( self, patch, newConnName, newConnKey ):
      numConnectorsInPatch = 0

      # Check if a BGP or LDP connector is configured in this patch
      for cliConnector in patch.connector.values():
         if cliConnector.name == newConnName:
            # this connector is going to be replaced, therefore ignore its presence
            continue
         if cliConnector.connectorKey == newConnKey:
            # this connector is going to be renamed, therefore ignore its presence
            continue
         # alternate connector is duo with the primary connector and should not be
         # count
         if not cliConnector.alternate:
            numConnectorsInPatch += 1
         if ( cliConnector.connectorKey.connectorType in ( ConnectorType.ldp,
              ConnectorType.bgp, ConnectorType.mplsStatic )
              and not cliConnector.alternate ):
            # notes: with ldp pseudowire redundancy feature, there could be two
            # remote connectors in patch and the alternate connector will be replaced
            # in setConnector...

            self.addError( "Patch with two remote connectors not supported" )
            return False

      # ldp patch allows more than 2 remote connectors
      if numConnectorsInPatch == 2:
         self.addError( "Patch has two connectors already" )
         return False

      return True

   # -------------------------------------------------------------------
   # MPLS LDP pseudowire connector related methods
   # -------------------------------------------------------------------
   def tryAddLdpPwConnector( self, patch, ldpPwName, connName=None, alt=False ):
      connName = connName or self.pickDefaultConnectorName( patch, alt=alt )
      t0( "Picked name: ", connName )
      if not connName:
         # We have two connectors already
         # Notes with ldpPwRedundancy feature, this also means alternate exists
         self.addError( "Patch has two connectors already" )
         return False

      cliConnector = getMplsLdpConnector( ldpPwName, connName, alt )

      # Delete duplicate connectors, if any, so the new connName gets priority.
      self.noConnectorPwLdp( True, ldpPwName )

      # only check avaliability for primary connector as we will always add and
      # delete primary and alternate connector together
      if not alt and not self.canAddRemoteConnector( patch, connName,
                                         cliConnector.connectorKey ):
         return False

      connIndex = connIndex2 if alt else self.nextConnectorIndex( patch )
      patch.connector[ connIndex ] = cliConnector
      return True

   def setConnectorPwLdp( self, args ):
      '''
      The caller is trying to add a new CliConnector (with or without
      the new connector name specified) or modify in-place an existing
      CliConnector (connector name required).

      Note that giving the name of an existing connName necessarily
      overwrites it and doesn't add another connector with the same name.
      '''
      ldpPwName = args[ 'LDP_PW_NAME' ]
      connName = args.get( 'CONN_NAME' )
      alternate = args.get( 'alternate' )
      altConnName = getAlternateConnectorName( connName )
      ldpAltPwName = args.get( 'LDP_ALTERNATE_PW_NAME' )
      patch = pwConfig.patch.get( self.patchName )

      if alternate and ldpPwName == ldpAltPwName:
         self.addError( "Invalid Cli input with same primary and alternate"
                        " pseudowire names" )
         return None

      if not self.patchPresentOrError( patch ):
         return None

      if alternate and not ldpAltPwName:
         return None

      # No connName case with highest priority
      if not connName:
         primaryRet = self.tryAddLdpPwConnector( patch, ldpPwName )
         if alternate:
            alternateRet = self.tryAddLdpPwConnector( patch, ldpAltPwName, alt=True )
         else:
            alternateRet = True
         if primaryRet and alternateRet:
            self.gotoPatchQosMode( connName, ldpPwName, altPwName=ldpAltPwName )
         return primaryRet and alternateRet

      newConnectorKey = getLdpConnectorKey( ldpPwName )
      newAltConnectorKey = getLdpConnectorKey( ldpAltPwName ) if alternate else None

      # Primary connector configured case:
      if self.hasConnectorInPatch( patch, connName, newConnectorKey ):
         # If both primary and alternate connector is identical to the original
         # ones, then don't add the new ones, as it causes churn
         # If only primary is identical, replace the alternate
         if alternate:
            if self.hasConnectorInPatch( patch, altConnName, newAltConnectorKey ):
               self.gotoPatchQosMode( connName, ldpPwName, altPwName=ldpAltPwName )
               return None
            else:
               altConnName = getAlternateConnectorName( connName )
               self.noConnectorName( altConnName )
               ret = self.tryAddLdpPwConnector( patch, ldpAltPwName, alt=True )
               if ret:
                  self.gotoPatchQosMode( connName, ldpPwName,
                                         altPwName=ldpAltPwName )
                  return None
               else:
                  return ret
         else:
            altConnName = getAlternateConnectorName( connName )
            self.noConnectorName( altConnName )
            self.gotoPatchQosMode( connName, ldpPwName, altPwName=ldpAltPwName )
            return None

      primaryRet = True
      alternateRet = True
      # Otherwise connector name is specified and no primary cocnnector
      altConnName = getAlternateConnectorName( connName )
      self.noConnectorName( connName )
      self.noConnectorName( altConnName )
      primaryRet = self.tryAddLdpPwConnector( patch, ldpPwName, connName )
      if alternate:
         alternateRet = self.tryAddLdpPwConnector( patch, ldpAltPwName, altConnName,
                                                   alt=True )
      # gotoPatchQosMode with both primary and alternate pwName, alternate could be
      # None
      if primaryRet and alternateRet:
         self.gotoPatchQosMode( connName, ldpPwName, altPwName=ldpAltPwName )
      return primaryRet and alternateRet

   def noConnectorPwLdp( self, no, ldpPwName, connName=None, alternate=False ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      # alternate ldpPwName may not be privided, in such case we need to search it
      # currently alternate connector is hardcoded as patch.connector["2"]=
      if alternate and ldpPwName == "" and "2" in patch.connector.keys():
         ldpPwName = patch.connector[ "2" ].connectorKey.name
         delCliConn = getMplsLdpConnector( ldpPwName, connName or "", alternate )
      else:
         delCliConn = getMplsLdpConnector( ldpPwName, connName or "", alternate )

      for connIndex, cliConn in patch.connector.items():
         # If connName is provided, it must also match
         if cliConn.isEqual( delCliConn, not connName ):
            del patch.connector[ connIndex ]
            delConnKey = getLdpConnectorKey( ldpPwName )
            if delConnKey in pwConfig.connector:
               addPseudowireAttribute( delConnKey, keepQosMap=False,
                                       keepPatchQosMode=False )
               if pwConfig.connector[ delConnKey ].ldpPwConfigMode is False:
                  del pwConfig.connector[ delConnKey ]

   # -------------------------------------------------------------------
   # BGP EVPN pseudowire connector related methods
   # -------------------------------------------------------------------
   def canAddBgpRemoteConnector( self, patch, connName, connKey ):
      if not self.canAddRemoteConnector( patch, connName, connKey ):
         return False

      # Check if a legacy tagged connector is configured in this patch
      for cliConnector in patch.connector.values():
         if cliConnector.name == connName:
            # this connector is going to be replaced, therefore ignore its presence
            continue
         if cliConnector.connectorKey.isLegacyTagged():
            self.addError( "Patch with BGP VPWS and tagged connectors "
                           "not supported" )
            return False

      return True

   def setConnectorPwBgp( self, vpwsName, pwName, connName=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      connKey = getBgpConnectorKey( vpwsName, pwName )

      tempConnector = getBgpConnector( vpwsName, pwName, connName or '' )
      if self.hasConnectorInPatchUpdated( patch, connName, tempConnector ):
         # The new connector is identical to the original one.
         # Don't do anything, as it would cause churn.
         self.gotoPatchQosMode( connName, pwName, vpwsName )
         return

      if not self.canAddBgpRemoteConnector( patch, connName, connKey ):
         return

      if connName:
         # Delete existing connector with same name, as it is going to be replaced
         self.noConnectorName( connName )
         # Delete existing connector with same key, as it is going to be renamed
         self.noConnectorPwBgp( vpwsName, pwName )
         # Delete existing alternate connector
         # This is needed when overwrite an ldp connector with alternate to local one
         altConnName = getAlternateConnectorName( connName )
         self.noConnectorName( altConnName )
      else:
         connName = self.pickDefaultConnectorName( patch )
         assert connName

      cliConnector = getBgpConnector( vpwsName, pwName, connName )
      connIndex = self.nextConnectorIndex( patch )
      patch.connector[ connIndex ] = cliConnector

      self.gotoPatchQosMode( connName, pwName, vpwsName )

   def noConnectorPwBgp( self, vpwsName, pwName, connName=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      delCliConn = getBgpConnector( vpwsName, pwName, connName or "" )

      for connIndex, cliConn in patch.connector.items():
         # If connName is provided, it must also match
         if cliConn.isEqual( delCliConn, not connName ):
            del patch.connector[ connIndex ]
            connKey = getBgpConnectorKey( vpwsName, pwName )
            del pwConfig.connector[ connKey ]

   def canAddMplsStaticRemoteConnector( self, patch, connName, connKey ):
      if not self.canAddRemoteConnector( patch, connName, connKey ):
         return False

      return True

   def setConnectorPwMplsStatic( self, pwName, connName=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      connKey = getMplsStaticConnectorKey( pwName )

      tempConnector = getMplsStaticConnector( pwName, connName or '' )
      if self.hasConnectorInPatchUpdated( patch, connName, tempConnector ):
         # The new connector is identical to the original one.
         # Don't do anything, as it would cause churn.
         return

      if not self.canAddMplsStaticRemoteConnector( patch, connName, connKey ):
         return

      if connName:
         # Delete existing connector with same name, as it is going to be replaced
         self.noConnectorName( connName )
         # Delete existing connector with same key, as it is going to be renamed
         self.noConnectorPwMplsStatic( pwName )
      else:
         connName = self.pickDefaultConnectorName( patch )
         assert connName

      cliConnector = getMplsStaticConnector( pwName, connName )
      connIndex = self.nextConnectorIndex( patch )
      patch.connector[ connIndex ] = cliConnector

   def noConnectorPwMplsStatic( self, pwName, connName=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      delCliConn = getMplsStaticConnector( pwName, connName or "" )

      for connIndex, cliConn in patch.connector.items():
         # If connName is provided, it must also match
         if cliConn.isEqual( delCliConn, not connName ):
            del patch.connector[ connIndex ]
            connKey = getMplsStaticConnectorKey( pwName )
            if ( connKey in pwConfig.connector and
                 pwConfig.connector[ connKey ].staticPwConfigMode is False ):
               del pwConfig.connector[ connKey ]

   def shutdown( self ):
      patch = pwConfig.patch.get( self.patchName )
      if not patch:
         return

      patch.enabled = False

   def noShutdown( self ):
      patch = pwConfig.patch.get( self.patchName )
      if not patch:
         return

      patch.enabled = True

class FlexibleCrossConnectConfigMode( FxcMode, BasicCli.ConfigModeBase ):
   _fxc = "Flexible cross connect"
   name = f"{_fxc} configuration"

   def __init__( self, parent, session, patchName ):
      t0( "FlexibleCrossConnectConfigMode init", patchName )
      FxcMode.__init__( self, patchName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.modeHelper = PatchConfigModeHelper( self, patchName, self._fxc )
      t0( "FlexibleCrossConnectConfigMode init", repr( patchName ) )
      if patchName not in pwConfig.patch:
         patch = pwConfig.patch.newMember( patchName )
         patch.fxc = True
         pwRunnability.config = True
      else:
         assert pwConfig.patch[ patchName ].fxc, "name {} was reused".format(
            patchName )

   @staticmethod
   def nextIndex( fxc ):
      return CliFxcHelper.nextIndex( fxc )

   @staticmethod
   def connectorCount( fxc ):
      return len( fxc.connector )

   def patchPresentOrError( self, patch ):
      return self.modeHelper.patchPresentOrError( patch )

   def _noRemote( self, fxc ):
      del fxc.connector[ fxc.fxcRemoteConnectorName ]
      fxc.fxcRemoteConnectorName = ""

   @traced( trace=t1 )
   def _noConnectorName( self, fxc, connName ):
      del fxc.connector[ connName ]
      if fxc.fxcRemoteConnectorName == connName:
         self._noRemote( fxc )

   @tracedFuncName( trace=t1 )
   def noConnectorName( self, connName, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not fxc:
         t0( funcName, "patch not found", self.patchName )
      else:
         self._noConnectorName( fxc, connName )

   def addErrorNoRoom( self, fxcName ):
      self.addError(
         "{fxc} {fxcName} cannot have more than {limit} connectors".format(
            fxc=self._fxc, fxcName=fxcName, limit=PatchType.fxcMaxConnector ) )

   @tracedFuncName( trace=t1 )
   def _setConnectorCommon( self, connKey, connName=None, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( fxc ):
         t0( funcName, "patch not found", self.patchName )
         return

      if connKey.isLocal():
         existingName = CliFxcHelper.findConnector( fxc, connKey, "" )
         exists = bool( existingName )
      else:
         existingName = fxc.fxcRemoteConnectorName
         exists = fxc.fxcRemoteConnector.connectorKey == connKey

      # Clean up an existing connector, if it exists with the same name, or else
      # check if this is a no-op and return early.
      if exists:
         if connName == existingName:
            # Nothing to do - same name and same connector
            t1( funcName, "nothing changed", connKey.stringValue, repr( connName ) )
            return

         if connName:
            # Re-adding the same connector, different name -> rename.
            # This seems like user error, but this is what we're doing for patches,
            # so maintain this behavior.

            # NOTE: This will invalidate fxc.fxcRemoteConnector.isValid for the
            # remote case so that we don't hit the "only 1 remote connector" error.
            self._noConnectorName( fxc, existingName )
         else:
            t1( funcName, "Re-use the existing name", repr( existingName ),
                "(no-op)" )
            return

      # Special case for remote: only 1 remote connector allowed
      isRemote = not connKey.isLocal()
      if ( isRemote and fxc.fxcRemoteConnector.isValid()
           and connName != fxc.fxcRemoteConnectorName ):
         self.addError(
            "{fxc} {fxcName} can only have a single remote connector".format(
               fxc=self._fxc, fxcName=self.patchName ) )
         return

      if self.connectorCount( fxc ) >= PatchType.fxcMaxConnector:
         self.addErrorNoRoom( self.patchName )
         return

      if not connName:
         nextIndex = self.nextIndex( fxc )
         # Handle race condition between multiple threads (otherwise it's handled by
         # the fxcMaxConnector check above).
         if nextIndex == PatchType.fxcInvalidIndex: # pragma: no cover
            self.addErrorNoRoom( self.patchName )
            return
         t1( funcName, "set connName from nextIndex", nextIndex )
         connName = str( nextIndex )

      if connName in fxc.connector:
         t1( funcName, "remove connector", connName )
         self._noConnectorName( fxc, connName )

      cliConnector = CliConnector( connKey )
      cliConnector.name = connName
      t1( funcName, "connector", connName, "->", connKey.stringValue )
      if not connKey.isLocal():
         fxc.fxcRemoteConnectorName = connName
      fxc.connector[ connName ] = cliConnector

   @tracedFuncName( trace=t1 )
   def setInterfaceConnector( self, intf, connName=None, funcName=None ):
      connKey = getInterfaceConnectorKey( intf )
      return self._setConnectorCommon( connKey, connName=connName )

   @tracedFuncName( trace=t1 )
   def setConnectorPwBgp( self, vpwsName, pwName, connName=None, funcName=None ):
      connKey = getBgpConnectorKey( vpwsName, pwName )
      return self._setConnectorCommon( connKey, connName=connName )

   @tracedFuncName( trace=t1 )
   def noInterfaceConnector( self, intf, connName=None, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not fxc:
         t0( funcName, "patch not found", self.patchName )
         return

      connKey = getInterfaceConnectorKey( intf )
      removeName = CliFxcHelper.findConnector( fxc, connKey, connName or "" )
      if removeName:
         t1( funcName, "remove", connKey.stringValue, repr( connName ),
             repr( removeName ) )
         del fxc.connector[ removeName ]
         return

      t1( funcName, "no match for", connKey.stringValue, repr( connName ) )

   @tracedFuncName( trace=t1 )
   def noConnectorPwBgp( self, vpwsName, pwName, connName=None, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not fxc:
         t0( funcName, "patch not found", self.patchName )
         return

      connKey = getBgpConnectorKey( vpwsName, pwName )
      if fxc.fxcRemoteConnector.connectorKey == connKey:
         if connName == fxc.fxcRemoteConnectorName or connName is None:
            t1( funcName, "remove", connKey.stringValue, repr( connName ),
                fxc.fxcRemoteConnectorName )
            self._noRemote( fxc )
            return
      t1( funcName, "no match for", connKey.stringValue, repr( connName ) )

   @tracedFuncName( trace=t1 )
   def _setEnabledCommon( self, enabled, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not fxc:
         t0( funcName, "patch not found", self.patchName )
         return

      fxc.enabled = enabled

   @traced( trace=t1 )
   def shutdown( self ):
      self._setEnabledCommon( False )

   @traced( trace=t1 )
   def noShutdown( self ):
      self._setEnabledCommon( True )

   @tracedFuncName( trace=t1 )
   def setVidNormalization( self, args, funcName=None ):
      vidNormalization = args.get( "VALUE" )
      fxc = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( fxc ):
         t0( funcName, "patch not found", self.patchName )
         return

      value = {
         None: "noNormalization",
         "single": "singleVid",
         "double": "doubleVid",
      }[ vidNormalization ]

      t0( funcName, "set vidNormalization", value )
      fxc.vidNormalization = value

def _patches( _mode ):
   return pwConfig.patch.members()

# --------------------------------------------------------------------------------
# [ no | default ] patch PATCH_NAME
# under "patch panel" config mode
# --------------------------------------------------------------------------------
class PatchPatchnameCmd( CliCommand.CliCommandClass ):
   syntax = 'patch PATCH_NAME'
   noOrDefaultSyntax = syntax
   # yapf: disable
   data = {
      'patch': 'Configure a patch',
      'PATCH_NAME': DynamicNameMatcher( _patches, helpdesc='Patch name' ),
   }
   # yapf: enable

   handler = PatchPanelConfigMode.gotoPatchMode
   noOrDefaultHandler = PatchPanelConfigMode.noPatch

PatchPanelConfigMode.addCommandClass( PatchPatchnameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] flexible-cross-connect NAME
# under "patch panel" config mode
# --------------------------------------------------------------------------------
class PatchPanelFxcCmd( CliCommand.CliCommandClass ):
   syntax = 'flexible-cross-connect NAME'
   noOrDefaultSyntax = syntax
   data = {
      'flexible-cross-connect': 'Configure a flexible cross connect',
      'NAME': DynamicNameMatcher( _patches, helpdesc='Flexible cross connect name' ),
   }

   handler = PatchPanelConfigMode.gotoFxcMode
   noOrDefaultHandler = PatchPanelConfigMode.noFxc

PatchPanelConfigMode.addCommandClass( PatchPanelFxcCmd )

# --------------------------------------------------------------------------------
# [ no | default ] connector [ CONN_NAME ] interface INTF [ dot1q vlan DOT1Q_TAG ]
# commands under the "patch [name]" config mode
# Note: For the no versions of the commands, only exact matching connectors will
# be removed
# --------------------------------------------------------------------------------
interfaceMatcher = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Local connector configuration' )
nodeDot1q = CliMatcher.KeywordMatcher( 'dot1q', helpdesc='802.1Q VLAN tag' )
nodeVlan = CliMatcher.KeywordMatcher( 'vlan', helpdesc='802.1Q VLAN tag' )
dot1qTagMatcher = CliMatcher.IntegerMatcher( 1, 4094, helpdesc='802.1Q VLAN tag' )

# We only allow ethernet and port-channel interfaces in config
intfMatcher = IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= SwitchIntfCli.SwitchIntf.matcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher

# and ethernet sub-interfaces
intfOrSubIntfNameMatcher = IntfMatcher()
intfOrSubIntfNameMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfOrSubIntfNameMatcher |= SwitchIntfCli.SwitchIntf.matcher
intfOrSubIntfNameMatcher |= LagIntfCli.EthLagIntf.matcher
intfOrSubIntfNameMatcher |= SubIntfCli.subMatcher
intfOrSubIntfNameMatcher |= LagIntfCli.subMatcher

class InterfaceConnectorCmdNoErrDisable( CliCommand.CliCommandClass ):
   syntax = 'connector [ CONN_NAME ] interface INTF \
               ( ( dot1q vlan DOT1Q_TAG ) | ( no-errdisable ) )'
   noOrDefaultSyntax = syntax
   data = {
      'connector': connectorMatcher,
      'CONN_NAME': connNameMatcher,
      'interface': interfaceMatcher,
      'INTF': intfMatcher,
      'dot1q': nodeDot1q,
      'vlan': nodeVlan,
      'DOT1Q_TAG': dot1qTagMatcher,
      'no-errdisable': 'Prevent error disabling the local interface',
   }

   @staticmethod
   def handler( mode, args ):
      return mode.setInterfaceConnector( args[ 'INTF' ],
         dot1qTag=args.get( 'DOT1Q_TAG' ),
         connName=args.get( 'CONN_NAME' ),
         noErrDisable='no-errdisable' in args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      connName = args.get( 'CONN_NAME' )
      dot1qTag = args.get( 'DOT1Q_TAG' )
      noErrDisable = 'no-errdisable' in args
      return PatchConfigMode.noInterfaceConnector( mode, intf, dot1qTag, connName,
                                                   noErrDisable=noErrDisable )

PatchConfigMode.addCommandClass( InterfaceConnectorCmdNoErrDisable )

class SubInterfaceConnectorCmd( CliCommand.CliCommandClass ):
   syntax = 'connector [ CONN_NAME ] interface INTF'
   noOrDefaultSyntax = syntax
   data = {
      'connector': connectorMatcher,
      'CONN_NAME': connNameMatcher,
      'interface': interfaceMatcher,
      'INTF': intfOrSubIntfNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      return mode.setInterfaceConnector( args[ 'INTF' ],
                                         connName=args.get( 'CONN_NAME' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      connName = args.get( 'CONN_NAME' )
      return mode.noInterfaceConnector( intf, connName=connName )

PatchConfigMode.addCommandClass( SubInterfaceConnectorCmd )
FlexibleCrossConnectConfigMode.addCommandClass( SubInterfaceConnectorCmd )

# --------------------------------------------------------------------------------
# [ no | default ] connector [ CONN_NAME ] pseudowire ldp LDP_PW_NAME [ alternate ]
# [ LDP_ALTERNATE_PW_NAME ]
# command under "patch [name]" config mode
# --------------------------------------------------------------------------------
class ConnectorPwLdp( CliCommand.CliCommandClass ):
   syntax = 'connector [ CONN_NAME ] pseudowire ldp LDP_PW_NAME [ alternate \
      LDP_ALTERNATE_PW_NAME ]'
   noOrDefaultSyntax = syntax
   data = {
      'connector': connectorMatcher,
      'CONN_NAME': connNameMatcher,
      'pseudowire': 'LDP, BGP or MPLS static pseudowire',
      'ldp': 'LDP pseudowire',
      'LDP_PW_NAME': ldpPwNameMatcher,
      'alternate': 'Alternate LDP pseudowire connector',
      'LDP_ALTERNATE_PW_NAME': ldpPwNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return mode.setConnectorPwLdp( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ldpPwName = args[ 'LDP_PW_NAME' ]
      connName = args.get( 'CONN_NAME' )
      ldpAltPwName = args.get( 'LDP_ALTERNATE_PW_NAME' ) or ""
      altConnName = getAlternateConnectorName( connName )
      mode.noConnectorPwLdp( True, ldpAltPwName or "", altConnName,
                             True )
      mode.noConnectorPwLdp( True, ldpPwName, connName, False )

PatchConfigMode.addCommandClass( ConnectorPwLdp )

# --------------------------------------------------------------------------------
# [ no | default ] connector [ CONN_NAME ] pseudowire mpls static MPLS_STATIC_PW_NAME
# command under "patch [name]" config mode
# --------------------------------------------------------------------------------
nodeMplsStaticPseudowireConnector = CliCommand.guardedKeyword( 'mpls',
      helpdesc='MPLS pseudowire',
      guard=mplsStaticPwSupportedGuard )

class ConnectorPwMplsStaticCmd( CliCommand.CliCommandClass ):
   syntax = 'connector [ CONN_NAME ] pseudowire mpls static MPLS_STATIC_PW_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'connector': connectorMatcher,
      'CONN_NAME': connNameMatcher,
      'pseudowire': 'LDP, BGP or MPLS static pseudowire',
      'mpls': nodeMplsStaticPseudowireConnector,
      'static': 'MPLS static pseudowire',
      'MPLS_STATIC_PW_NAME': mplsStaticPwNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return mode.setConnectorPwMplsStatic( args[ "MPLS_STATIC_PW_NAME" ],
                                            args.get( "CONN_NAME" ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsStaticPwName = args[ 'MPLS_STATIC_PW_NAME' ]
      connName = args.get( 'CONN_NAME' )
      return mode.noConnectorPwMplsStatic( mplsStaticPwName, connName )

PatchConfigMode.addCommandClass( ConnectorPwMplsStaticCmd )

# -------------------------------------------------------------------------------
# The "[no|default] connector [connName] pseudowire bgp vpws [name]
#      pseudowire [pw-name]" command under "patch [name]" config mode
# -------------------------------------------------------------------------------
bgpVpwsNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ( sorted( getVpwsEviName( macVrf.name )
                               for macVrf in bgpMacVrf.config.values()
                               if macVrf.isVpwsMacVrf() ) ),
      "EVPN VPWS instance name" )

class PseudowireBgpConnectorCmd( CliCommand.CliCommandClass ):
   syntax = "connector [CONN_NAME] pseudowire bgp vpws VPWS_NAME pwName PW_NAME"
   noOrDefaultSyntax = syntax
   data = {
      "connector": connectorMatcher,
      "CONN_NAME": connNameMatcher,
      "pseudowire": 'LDP, BGP or MPLS static pseudowire',
      "bgp": bgpMatcher,
      "vpws": vpwsMatcher,
      "VPWS_NAME": CliCommand.Node( bgpVpwsNameMatcher, storeSharedResult=True ),
      "pwName": CliMatcher.KeywordMatcher( "pseudowire",
                                           "EVPN VPWS pseudowire name" ),
      "PW_NAME": bgpPseudowireNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setConnectorPwBgp( args[ "VPWS_NAME" ], args[ "PW_NAME" ],
                              args.get( "CONN_NAME" ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noConnectorPwBgp( args[ "VPWS_NAME" ], args[ "PW_NAME" ],
                             args.get( "CONN_NAME" ) )

PatchConfigMode.addCommandClass( PseudowireBgpConnectorCmd )
FlexibleCrossConnectConfigMode.addCommandClass( PseudowireBgpConnectorCmd )

# -------------------------------------------------------------------------------
# The "[no|default] qos map dscp to traffic-class MAP_NAME" command under
#      patch config mode
# -------------------------------------------------------------------------------
nodeDecapDscpQosMap = CliCommand.guardedKeyword( 'qos',
      helpdesc="Configure a QoS map",
      guard=pwDecapDscpQosMapSupportedGuard )

class PwDecapDscpQosMapCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map dscp to traffic-class MAP_NAME'
   noOrDefaultSyntax = 'qos map dscp to traffic-class ...'
   data = {
      'qos': nodeDecapDscpQosMap,
      'map': 'Mapping configuration of different QoS parameters',
      'dscp': 'Specify a DSCP match',
      'to': 'Map to traffic-class',
      'traffic-class': 'Specify a traffic-class match',
      'MAP_NAME': matcherDscpMapName,
   }

   handler = PatchQosMode.setPwDecapDscpQosMap
   noOrDefaultHandler = PatchQosMode.noPwDecapDscpQosMap

PatchQosMode.addCommandClass( PwDecapDscpQosMapCmd )

# -------------------------------------------------------------------------------
# The "[no|default] qos map cos to traffic-class MAP_NAME" command under
#      patch config mode
# -------------------------------------------------------------------------------
nodeDecapCosQosMap = CliCommand.guardedKeyword( 'qos',
      helpdesc="Configure a QoS map",
      guard=pwDecapCosQosMapSupportedGuard )

class PwDecapCosQosMapCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map cos to traffic-class MAP_NAME'
   noOrDefaultSyntax = 'qos map cos to traffic-class ...'
   data = {
      'qos': nodeDecapCosQosMap,
      'map': 'Mapping configuration of different QoS parameters',
      'cos': 'Specify a COS match',
      'to': 'Map to traffic-class',
      'traffic-class': 'Specify a traffic-class match',
      'MAP_NAME': matcherCosMapName,
   }

   handler = PatchQosMode.setPwDecapCosQosMap
   noOrDefaultHandler = PatchQosMode.noPwDecapCosQosMap

PatchQosMode.addCommandClass( PwDecapCosQosMapCmd )

# --------------------------------------------------------------------------------
# ( no | default ) connector CONN_NAME
# command under "patch [name]" config mode
# --------------------------------------------------------------------------------
class ConnectorConnnameCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'connector CONN_NAME'
   data = {
      'connector': connectorMatcher,
      'CONN_NAME': connNameMatcher
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noConnectorName( args[ 'CONN_NAME' ] )

PatchConfigMode.addCommandClass( ConnectorConnnameCmd )
FlexibleCrossConnectConfigMode.addCommandClass( ConnectorConnnameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] shutdown
# command under "patch [name]" config mode
# --------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable patch',
   }

   @staticmethod
   def handler( mode, args ):
      return mode.shutdown()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noShutdown()

PatchConfigMode.addCommandClass( ShutdownCmd )
FlexibleCrossConnectConfigMode.addCommandClass( ShutdownCmd )

# --------------------------------------------------------------------------------
# [no|default] vlan tag normalization (single|double)
# command under "patch [name]" config mode
# --------------------------------------------------------------------------------
class VlanTagNormalizationCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan tag normalization VALUE'
   noOrDefaultSyntax = 'vlan tag normalization ...'
   data = {
      'vlan': 'FXC VLAN configuration',
      'tag': 'FXC VLAN tag configuration',
      'normalization': 'FXC VLAN tag normalization configuration',
      'VALUE': CliMatcher.EnumMatcher( {
         'single': 'Use a single VLAN tag for normalization',
         'double': 'Use double VLAN tags for normalization',
      } ),
   }

   handler = FlexibleCrossConnectConfigMode.setVidNormalization
   noOrDefaultHandler = handler

FlexibleCrossConnectConfigMode.addCommandClass( VlanTagNormalizationCommand )

# -------------------------------------------------------------------------------
# MPLS pseudowires config mode related helpers
# -------------------------------------------------------------------------------
class MplsPseudowiresConfigMode( MplsPseudowiresMode, BasicCli.ConfigModeBase ):
   name = 'MPLS pseudowires configuration'

   def __init__( self, parent, session ):
      MplsPseudowiresMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# Add the MPLS static pseudowires command.
MplsPseudowiresConfigMode.addCommandClass( MplsStaticPseudowiresCmd )

if toggleVplsBgpSignalingEnabled():
   # Add the "profiles" command.
   MplsPseudowiresConfigMode.addCommandClass(
      PseudowireProfileCli.MplsPseudowireProfilesCmd )

def gotoMplsPseudowiresMode( mode, args ):
   childMode = mode.childMode( MplsPseudowiresConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMplsPseudowiresMode( mode=None, args=None ):
   # Delete all profiles.
   if toggleVplsBgpSignalingEnabled():
      PseudowireProfileCli.clearAllProfiles()

   # Delete all MPLS static PWs.
   noMplsStaticPseudowires()


# -------------------------------------------------------------------------------
# MPLS Static pseudowire config mode related helpers
# -------------------------------------------------------------------------------
class MplsStaticPseudowiresConfigMode( MplsStaticPseudowiresMode,
                                       BasicCli.ConfigModeBase ):
   name = "MPLS static pseudowire configuration"

   def __init__( self, parent, session ):
      MplsStaticPseudowiresMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def gotoStaticPseudowireMode( self, args ):
      staticPwName = args[ 'STATIC_PW_NAME' ]
      childMode = self.childMode( StaticPseudowireConfigMode,
            staticPwName=staticPwName )
      self.session_.gotoChildMode( childMode )

   def noStaticPseudowireMode( self, args ):
      staticPwName = args[ 'STATIC_PW_NAME' ]
      connKey = getMplsStaticConnectorKey( staticPwName )
      if connKey in pwConfig.connector:
         del pwConfig.connector[ connKey ]

# -------------------------------------------------------------------------------
# MPLS LDP pseudowire config mode related helpers
# -------------------------------------------------------------------------------
class MplsLdpPseudowiresConfigMode( MplsLdpPseudowiresMode,
                                    BasicCli.ConfigModeBase ):
   name = "MPLS LDP pseudowire configuration"

   def __init__( self, parent, session ):
      MplsLdpPseudowiresMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setOrNoMtu( self, mtu ):
      '''If mtu is None, it implies we are disabling mtu'''
      oldValue = pwConfig.mtu
      if oldValue != mtu:
         updateMtu( mtu )

   def setMtu( self, args ):
      self.setOrNoMtu( args[ 'MTU' ] )

   def noMtu( self, args ):
      self.setOrNoMtu( None )

   def setMtuIgnore( self, args ):
      pwConfig.mtuIgnore = True

   def gotoLdpPseudowireMode( self, args ):
      ldpPwName = args[ 'LDP_PW_NAME' ]
      childMode = self.childMode( LdpPseudowireConfigMode, ldpPwName=ldpPwName )
      self.session_.gotoChildMode( childMode )

   def noLdpPseudowireMode( self, args ):
      if ldpPwName := args.get( 'LDP_PW_NAME' ):
         # Called via a command.
         ldpPwNames = [ ldpPwName ]
      else:
         # Called via `mode.clear`. Get all pseudowires.
         ldpPwNames = [ pw.name for pw in pwConfig.connector ]

      for ldpPwName in ldpPwNames:
         connKey = getLdpConnectorKey( ldpPwName )
         if connKey in pwConfig.connector:
            if pwConfig.connector[ connKey ].patchQosMode is False:
               del pwConfig.connector[ connKey ]
            else:
               addPseudowireAttribute( connKey, keepNbrAddr=False, keepPwId=False,
                                       keepMtu=False, keepControlWord=False,
                                       keepFlowLabelMode=False,
                                       keepLdpPwConfigMode=False )

   def gotoLdpPseudowireProfileMode( self, args ):
      ldpPwProfileName = args[ 'LDP_PW_PROFILE_NAME' ]
      childMode = self.childMode( LdpPseudowireProfileConfigMode,
                                  ldpPwProfileName=ldpPwProfileName )
      self.session_.gotoChildMode( childMode )

   def noLdpPseudowireProfileMode( self, args ):
      if ldpPwProfileName := args.get( 'LDP_PW_PROFILE_NAME' ):
         # Called via a command.
         ldpPwProfileNames = [ ldpPwProfileName ]
      else:
         # Called via `mode.clear`. Get all profiles.
         ldpPwProfileNames = getPwProfiles( self )

      for ldpPwProfileName in ldpPwProfileNames:
         connKey = getLdpAutoDiscoveryProfileConnectorKey( ldpPwProfileName )
         del pwConfig.connector[ connKey ]

def gotoMplsLdpPseudowiresMode( mode, args ):
   childMode = mode.childMode( MplsLdpPseudowiresConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMplsLdpPseudowires( mode=None, args=None ):
   for connKey in pwConfig.connector:
      if connKey.connectorType in ( 'ldp', 'ldpAutoDiscoveryProfile' ):
         addPseudowireAttribute( connKey, keepLdpPwConfigMode=False )
         if pwConfig.connector[ connKey ].patchQosMode is False:
            del pwConfig.connector[ connKey ]
   updateMtu( None )
   clearResolutionRibProfileConfig()

def gotoMplsStaticPseudowiresMode( mode, args ):
   childMode = mode.childMode( MplsStaticPseudowiresConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMplsStaticPseudowires( mode=None, args=None ):
   for connKey in pwConfig.connector:
      if connKey.connectorType == 'mplsStatic':
         del pwConfig.connector[ connKey ]

class StaticPseudowireConfigModeBase( BasicCli.ConfigModeBase ):
   def __init__( self, parent, session ):
      super().__init__( parent, session )
      self.maybeAddPseudowire()

   def getConnectorKey( self ):
      raise NotImplementedError

   def maybeAddPseudowire( self ):
      connKey = self.getConnectorKey()
      if connKey not in pwConfig.connector:
         connector = Tac.Value( "Pseudowire::Connector", staticPwConfigMode=True )
         pwConfig.connector[ connKey ] = connector
      else:
         addPseudowireAttribute( connKey, staticPwConfigMode=True,
                                 keepStaticPwConfigMode=True )

   def setTunnelIntfTransport( self, args ):
      intfId = ArnetIntfId( args[ 'TRANSPORT' ].name )
      addPseudowireAttribute( self.getConnectorKey(), transport=intfId )

   def noTunnelIntfTransport( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), transport=ArnetIntfId() )

   def setLocalLabel( self, args ):
      localLabel = ArnetMplsLabel( args[ 'LOCAL_LABEL' ] )
      addPseudowireAttribute( self.getConnectorKey(), localLabel=localLabel )

   def noLocalLabel( self, args ):
      addPseudowireAttribute( self.getConnectorKey(),
                              localLabel=ArnetMplsLabel.null )

   def setNeighborLabel( self, args ):
      neighborLabel = ArnetMplsLabel( args[ 'NEIGHBOR_LABEL' ] )
      addPseudowireAttribute( self.getConnectorKey(), neighborLabel=neighborLabel )

   def noNeighborLabel( self, args ):
      addPseudowireAttribute( self.getConnectorKey(),
                              neighborLabel=ArnetMplsLabel.null )

   def setControlWord( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), controlWord=True,
                              keepControlWord=True )

   def noControlWord( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), keepControlWord=False )

   def setNextHopTransport( self, args ):
      addPseudowireAttribute( self.getConnectorKey(),
                             nexthopGroupTransport=args[ 'NEXTHOPGROUPNAME' ] )

   def noNexthopGroupTransport( self, args ):
      addPseudowireAttribute( self.getConnectorKey(),
                              nexthopGroupTransport='' )

class LdpPseudowireConfigModeBase( BasicCli.ConfigModeBase ):
   """Shared between normal pseudowire and pseudowire profile config mode."""
   def __init__( self, parent, session ):
      super().__init__( parent, session )
      self.maybeAddPseudowire()

   def getConnectorKey( self ):
      raise NotImplementedError

   def maybeAddPseudowire( self ):
      connKey = self.getConnectorKey()
      if connKey not in pwConfig.connector:
         connector = Tac.Value( "Pseudowire::Connector", ldpPwConfigMode=True )
         pwConfig.connector[ connKey ] = connector
      else:
         addPseudowireAttribute( connKey, ldpPwConfigMode=True,
                                 keepLdpPwConfigMode=True )

   def setNeighbor( self, args ):
      ipGenAddr = Tac.Value( "Arnet::IpGenAddr", args[ 'IPADDR' ] )
      addPseudowireAttribute( self.getConnectorKey(), nbrAddr=ipGenAddr,
                              keepNbrAddr=True )

   def noNeighbor( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), keepNbrAddr=False )

   def setPwId( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), pwId=args[ 'PWID' ],
                              keepPwId=True )

   def noPwId( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), keepPwId=False )

   def setMtu( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), mtu=args[ 'MTU' ],
                              keepMtu=True )

   def noMtu( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), keepMtu=False )

   def setColor( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), color=args[ 'COLOR' ],
                              colorPresent=True )

   def noColor( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), colorPresent=False )

   def setResolutionRibProfile( self, args ):
      addPseudowireAttribute(
         self.getConnectorKey(),
         resolutionRibProfileConfig=args.get( 'TUNNEL_ONLY_RIBS' ) )

   def noResolutionRibProfile( self, args ):
      addPseudowireAttribute(
         self.getConnectorKey(),
         resolutionRibProfileConfig=Tac.Value(
            'Routing::Rib::ResolutionRibProfileConfig' ) )

   def setControlWord( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), controlWord=True,
                              keepControlWord=True )

   def noControlWord( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), keepControlWord=False )

   def setFlowLabel( self, args ):
      if 'transmit' in args:
         flowLabelMode = FlowLabelMode.transmit
      elif 'receive' in args:
         flowLabelMode = FlowLabelMode.receive
      else:
         flowLabelMode = FlowLabelMode.both
      addPseudowireAttribute( self.getConnectorKey(), flowLabelMode=flowLabelMode,
                              keepFlowLabelMode=True )

   def noFlowLabel( self, args ):
      addPseudowireAttribute( self.getConnectorKey(), keepFlowLabelMode=False )

   def setPwType( self, args ):
      pwType = PseudowireType.pwType5 if 'raw' in args else PseudowireType.pwType4
      addPseudowireAttribute( self.getConnectorKey(), pwType=pwType )

   def noPwType( self, args ):
      addPseudowireAttribute( self.getConnectorKey(),
                              pwType=PseudowireType.pwTypeNone )

class StaticPseudowireConfigMode( StaticPseudowireMode,
                                  StaticPseudowireConfigModeBase ):
   name = "Static pseudowire configuration"

   def __init__( self, parent, session, staticPwName ):
      StaticPseudowireMode.__init__( self, staticPwName )
      StaticPseudowireConfigModeBase.__init__( self, parent, session )

   def getConnectorKey( self ):
      return getMplsStaticConnectorKey( self.staticPwName )

class LdpPseudowireConfigMode( LdpPseudowireMode, LdpPseudowireConfigModeBase ):
   name = "LDP pseudowire configuration"

   def __init__( self, parent, session, ldpPwName ):
      LdpPseudowireMode.__init__( self, ldpPwName )
      LdpPseudowireConfigModeBase.__init__( self, parent, session )

   def getConnectorKey( self ):
      return getLdpConnectorKey( self.ldpPwName )

class LdpPseudowireProfileConfigMode( LdpPseudowireProfileMode,
                                      LdpPseudowireConfigModeBase ):
   name = "LDP pseudowire profile configuration"

   def __init__( self, parent, session, ldpPwProfileName ):
      LdpPseudowireProfileMode.__init__( self, ldpPwProfileName )
      LdpPseudowireConfigModeBase.__init__( self, parent, session )

   def getConnectorKey( self ):
      return getLdpAutoDiscoveryProfileConnectorKey( self.ldpPwProfileName )

# --------------------------------------------------------------------------------
# [ no | default ] pseudowire STATIC_PW_NAME
# config mode under "static pseudowires" config mode
# --------------------------------------------------------------------------------
class PseudowireStaticpwnameCmd( CliCommand.CliCommandClass ):
   syntax = 'pseudowire STATIC_PW_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'pseudowire': 'Configure a named pseudowire',
      'STATIC_PW_NAME': staticPwNameMatcher,
   }

   handler = MplsStaticPseudowiresConfigMode.gotoStaticPseudowireMode
   noOrDefaultHandler = MplsStaticPseudowiresConfigMode.noStaticPseudowireMode

MplsStaticPseudowiresConfigMode.addCommandClass( PseudowireStaticpwnameCmd )

# --------------------------------------------------------------------------------
# transport [ TUNNELINTF | nexthop-group NEXTHOPGROUPNAME ]
# command under "pseudowire [name]" config mode
# --------------------------------------------------------------------------------

class TransportCmd( CliCommand.CliCommandClass ):
   data = {
         'TRANSPORT': TunnelIntfCli.TunnelIntf.matcher,
      }

   if toggleStaticMplsPwNhgTransportEnabled():
      syntax = 'transport (TRANSPORT | (nexthop-group NEXTHOPGROUPNAME))'
      noOrDefaultSyntax = '((transport ... )| (transport nexthop-group))'
      data[ 'transport' ] = 'Transport tunnel interface / nexthop-group'
      data[ 'nexthop-group' ] = IraNexthopGroupCli.nexthopGroupCfgNode
      data[ 'NEXTHOPGROUPNAME' ] = IraNexthopGroupCli.nexthopGroupCfgName

   else:
      syntax = 'transport TRANSPORT'
      noOrDefaultSyntax = 'transport ...'
      data[ 'transport' ] = 'Transport tunnel interface'


   @staticmethod
   def handler( mode, args ):
      if 'nexthop-group' in args:
         mode.setNextHopTransport( args )
         mode.noTunnelIntfTransport( args )
      else:
         mode.setTunnelIntfTransport( args )
         mode.noNexthopGroupTransport( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'nexthop-group' in args:
         mode.noNexthopGroupTransport( args )
      else:
         mode.noTunnelIntfTransport( args )

StaticPseudowireConfigMode.addCommandClass( TransportCmd )

# --------------------------------------------------------------------------------
# local label LOCAL_LABEL
# command under "pseudowire [name]" config mode
# --------------------------------------------------------------------------------

# Adds conflicting static label range change configs to conflictingConfigList if
# the new range doesn't include all local labels used for MPLS static PWs.
# CLI hook checkConflictingLabelRangeHook will go through the conflictingConfigList,
# publish the errors, and block the static label range change from taking effect.
def checkConflictingLabelRangeForMplsStaticPwLocalLabel( rangeType, base, size,
                                                conflictingConfigList ):
   if rangeType != MplsCli.LabelRangeInfo.rangeTypeStatic:
      return
   minLabel = base
   maxLabel = base + size - 1
   for connectorKey, connector in pwConfig.connector.items():
      if connectorKey.connectorType != ConnectorType.mplsStatic:
         return
      if not connector.configuredStaticLabel:
         return
      label = connector.localLabel
      if not minLabel <= label <= maxLabel:
         conflictingConfigList.append(
               "MPLS static PW %s local lookup MPLS label %s" %
                     ( connectorKey.name, label ) )

class LocalLabelCmd( CliCommand.CliCommandClass ):
   syntax = 'local label LOCAL_LABEL'
   noOrDefaultSyntax = 'local label ...'
   data = {
      'local': 'Local MPLS label',
      'label': 'Local MPLS label',
      # For static pseudowires, local label identifies the MPLS label
      # used on decap. If specified in the CLI, this label is assigned
      # from the static MPLS label chunk. Hence, we match on the static
      # range for the label value. By default ( if not CLI configured ),
      # the local label is assigned from the pseudowire MPLS label chunk.
      'LOCAL_LABEL': mplsStaticLabelMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setLocalLabel( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noLocalLabel( args )

StaticPseudowireConfigMode.addCommandClass( LocalLabelCmd )

# --------------------------------------------------------------------------------
# neighbor label NEIGHBOR_LABEL
# command under "pseudowire [name]" config mode
# --------------------------------------------------------------------------------

class NeighborLabelCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor label NEIGHBOR_LABEL'
   noOrDefaultSyntax = 'neighbor label ...'
   data = {
      'neighbor': 'Neighbor MPLS label',
      'label': 'Neighbor MPLS label',
      # For static pseudowires, neighbor label identifies the MPLS label
      # used on encap. If specified in the CLI, this label can be any valid
      # MPLS label which the remote tep uses to identify the packet. Hence,
      # we just match the value with valid MPLS label values. By default ( if
      # not CLI configured, the neighbor label is set same as the local label,
      # whatever that is.
      'NEIGHBOR_LABEL': mplsLabelMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setNeighborLabel( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noNeighborLabel( args )

StaticPseudowireConfigMode.addCommandClass( NeighborLabelCmd )

# --------------------------------------------------------------------------------
# [ no | default ] mtu MTU
# command under "mpls ldp pseudowires" config mode
# --------------------------------------------------------------------------------

class MtuMtuCmd( CliCommand.CliCommandClass ):
   syntax = 'mtu MTU'
   noOrDefaultSyntax = 'mtu ...'
   data = {
      'mtu': mtuMatcher,
      'MTU': mtuNumMatcher
   }

   handler = MplsLdpPseudowiresConfigMode.setMtu
   noOrDefaultHandler = MplsLdpPseudowiresConfigMode.noMtu

MplsLdpPseudowiresConfigMode.addCommandClass( MtuMtuCmd )

# --------------------------------------------------------------------------------
# mtu ignore
# command under "mpls ldp pseudowires" config mode
# --------------------------------------------------------------------------------
class MtuIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'mtu ignore'
   data = {
      'mtu': mtuMatcher,
      'ignore': 'Ignore MTU mismatch',
   }

   handler = MplsLdpPseudowiresConfigMode.setMtuIgnore

MplsLdpPseudowiresConfigMode.addCommandClass( MtuIgnoreCmd )

# --------------------------------------------------------------------------------
# [ no | default ] pseudowire LDP_PW_NAME
# config mode under "mpls ldp pseudowires" config mode
# --------------------------------------------------------------------------------
class PseudowireLdppwnameCmd( CliCommand.CliCommandClass ):
   syntax = 'pseudowire LDP_PW_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'pseudowire': 'Configure a named pseudowire',
      'LDP_PW_NAME': ldpPwNameMatcher,
   }

   handler = MplsLdpPseudowiresConfigMode.gotoLdpPseudowireMode
   noOrDefaultHandler = MplsLdpPseudowiresConfigMode.noLdpPseudowireMode

MplsLdpPseudowiresConfigMode.addCommandClass( PseudowireLdppwnameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] profile PROFILE_NAME
# config mode under "mpls ldp pseudowires" config mode
# --------------------------------------------------------------------------------
class LdpPseudowireProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'profile LDP_PW_PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'profile': 'Configure a pseudowire profile',
      'LDP_PW_PROFILE_NAME': ldpPwProfileMatcher,
   }

   handler = MplsLdpPseudowiresConfigMode.gotoLdpPseudowireProfileMode
   noOrDefaultHandler = MplsLdpPseudowiresConfigMode.noLdpPseudowireProfileMode

MplsLdpPseudowiresConfigMode.addCommandClass( LdpPseudowireProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] neighbor resolution tunnel-rib ( system-tunnel-rib | TUNNEL_NAME )
# under "mpls ldp pseudowires" config mode
# --------------------------------------------------------------------------------
# --------------------------------------------------------------------------------
# [ no | default ] neighbor resolution TUNNEL_ONLY_RIBS
# command under "mpls ldp pseudowires" config mode
# --------------------------------------------------------------------------------

def clearResolutionRibProfileConfig():
   methods = [
      ( 'tunnelRib', systemColoredTunnelRibName ),
      ( 'tunnelRib', systemTunnelRibName ),
   ]
   pwConfig.resolutionRibProfileConfig = \
      ResolutionRibProfileConfig.tacResolutionRibProfileConfig(
         methods, False )

class MplsLdpPseudowiresTunnelRibCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor resolution TUNNEL_ONLY_RIBS'
   noOrDefaultSyntax = 'neighbor resolution ...'
   data = {
      'neighbor': 'LDP neighbor',
      'resolution': 'Resolution of all LDP pseudowires',
      'TUNNEL_ONLY_RIBS': ResolutionRibsExprFactory(),
   }

   def handler( self, args ):
      resRibProfile = args.get( 'TUNNEL_ONLY_RIBS' )
      pwConfig.resolutionRibProfileConfig = resRibProfile

   def noOrDefaultHandler( self, args ):
      clearResolutionRibProfileConfig()

MplsLdpPseudowiresConfigMode.addCommandClass( MplsLdpPseudowiresTunnelRibCmd )

# --------------------------------------------------------------------------------
# [ no | default ] neighbor resolution TUNNEL_ONLY_RIBS
# command under "pseudowire [name]" config mode
# --------------------------------------------------------------------------------
class LdpPseudowireTunnelRibCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor resolution TUNNEL_ONLY_RIBS'
   noOrDefaultSyntax = 'neighbor resolution ...'
   data = {
      'neighbor': 'LDP neighbor',
      'resolution': 'Resolution of this LDP pseudowire',
      'TUNNEL_ONLY_RIBS': ResolutionRibsExprFactory(),
   }

   @staticmethod
   def handler( mode, args ):
      mode.setResolutionRibProfile( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noResolutionRibProfile( args )

LdpPseudowireConfigMode.addCommandClass( LdpPseudowireTunnelRibCmd )

# --------------------------------------------------------------------------------
# [ no | default ] neighbor IPADDR
# command under "pseudowire [name]" config mode
# --------------------------------------------------------------------------------
class NeighborIpaddrCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor IPADDR'
   noOrDefaultSyntax = 'neighbor ...'
   data = {
      'neighbor': 'LDP neighbor',
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setNeighbor( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noNeighbor( args )

LdpPseudowireConfigMode.addCommandClass( NeighborIpaddrCmd )

# --------------------------------------------------------------------------------
# [ no | default ] pseudowire-id PWID
# command under "pseudowire [name]" config mode
# --------------------------------------------------------------------------------
class PseudowireIdPwidCmd( CliCommand.CliCommandClass ):
   syntax = 'pseudowire-id PWID'
   noOrDefaultSyntax = 'pseudowire-id ...'
   data = {
      'pseudowire-id': 'Pseudowire ID',
      'PWID': pwIdMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setPwId( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noPwId( args )

LdpPseudowireConfigMode.addCommandClass( PseudowireIdPwidCmd )

# -------------------------------------------------------------------------------
# The "[no|default] mtu [bytes]" command under "pseudowire [name]" config mode
# -------------------------------------------------------------------------------
class LdpPseudowireMtuMtuCmd( CliCommand.CliCommandClass ):
   syntax = 'mtu MTU'
   noOrDefaultSyntax = 'mtu ...'
   data = {
      'mtu': mtuMatcher,
      'MTU': mtuNumMatcher
   }

   @staticmethod
   def handler( mode, args ):
      mode.setMtu( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noMtu( args )

LdpPseudowireConfigMode.addCommandClass( LdpPseudowireMtuMtuCmd )
LdpPseudowireProfileConfigMode.addCommandClass( LdpPseudowireMtuMtuCmd )
if toggleVplsBgpSignalingEnabled():
   PseudowireProfileCli.addProfileCommandClass( LdpPseudowireMtuMtuCmd )

# -------------------------------------------------------------------------------
# The "[no|default] color [value]" command under "pseudowire [name]" config mode
# -------------------------------------------------------------------------------
class LdpPseudowireColorCmd( CliCommand.CliCommandClass ):
   syntax = 'color COLOR'
   noOrDefaultSyntax = 'color ...'
   data = {
      'color': 'Color',
      'COLOR': CliMatcher.IntegerMatcher( COLOR_MIN, COLOR_MAX,
                                          helpdesc='Color value' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.setColor( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noColor( args )

LdpPseudowireConfigMode.addCommandClass( LdpPseudowireColorCmd )

# --------------------------------------------------------------------------------
# [ no | default ] control-word
# command under "pseudowire [name]" config mode
# --------------------------------------------------------------------------------
class ControlWordCmd( CliCommand.CliCommandClass ):
   syntax = 'control-word'
   noOrDefaultSyntax = syntax
   data = {
      'control-word': 'Enable control word',
   }

   @staticmethod
   def handler( mode, args ):
      mode.setControlWord( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noControlWord( args )

LdpPseudowireConfigMode.addCommandClass( ControlWordCmd )
LdpPseudowireProfileConfigMode.addCommandClass( ControlWordCmd )
if toggleVplsBgpSignalingEnabled():
   PseudowireProfileCli.addProfileCommandClass( ControlWordCmd )

StaticPseudowireConfigMode.addCommandClass( ControlWordCmd )
# -------------------------------------------------------------------------------
# The "[no|default] label flow [ transmit | receive ]" command under
# "pseudowire [name]" config mode
# -------------------------------------------------------------------------------
class FlowLabelCmd( CliCommand.CliCommandClass ):
   syntax = 'label flow [ transmit | receive ]'
   noOrDefaultSyntax = 'label flow ...'
   data = {
      'label': CliCommand.guardedKeyword( 'label', helpdesc='Label operation',
                                          guard=flowLabelSupportedGuard ),
      'flow': 'Enable flow label',
      'transmit': 'Enable flow label for transmit only',
      'receive': 'Enable flow label for receive only'
   }

   @staticmethod
   def handler( mode, args ):
      mode.setFlowLabel( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noFlowLabel( args )

LdpPseudowireConfigMode.addCommandClass( FlowLabelCmd )
LdpPseudowireProfileConfigMode.addCommandClass( FlowLabelCmd )
if toggleVplsBgpSignalingEnabled():
   PseudowireProfileCli.addProfileCommandClass( FlowLabelCmd )

# --------------------------------------------------------------------------------
# [ no | default ] pseudowire-type ( raw | tagged )
# command under "pseudowire/profile [name]" config mode
# --------------------------------------------------------------------------------
class PwTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'pseudowire-type ( raw | tagged )'
   noOrDefaultSyntax = 'pseudowire-type ...'
   data = {
      'pseudowire-type': 'Pseudowire type',
      'raw': 'Pseudowire type 5',
      'tagged': 'Pseudowire type 4',
   }

   @staticmethod
   def handler( mode, args ):
      mode.setPwType( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noPwType( args )

LdpPseudowireConfigMode.addCommandClass( PwTypeCmd )
LdpPseudowireProfileConfigMode.addCommandClass( PwTypeCmd )
# TODO: BUG792276, BUG980715
# Currently Pseudowire profiles that are configured under mpls pseudowires only work
# with VPLS-BGP groups, which do not use the pseudowire type config in a pseudowire
# profile. Therefore the pseudowire type config has been disabled. Once BUG 792276
# is implemented and VPLS-LDP groups can also use 'mpls pseudowire' profiles, then
# we should re-enable the pseudowire type config for 'mpls pseudowire' profiles.
# if toggleVplsBgpSignalingEnabled():
#    PseudowireProfileCli.addProfileCommandClass( PwTypeCmd )

# -------------------------------------------------------------------------------
# The "[no|default] debug pseudowire-id PWID neighbor PWPEER peer-label LABEL
# type PWTYPE [mtu MTU] [control-word] [flow-label FLMODE] [dot1q vlan DOT1QTAG]
# [group-id GROUPID]" hidden command under "mpls ldp pseudowires" config mode
# -------------------------------------------------------------------------------
class DebugPwLdpCmd( CliCommand.CliCommandClass ):
   syntax = '''debug pseudowire-id PWID neighbor PWPEER peer-label LABEL
               type PWTYPE [mtu MTU] [control-word] [flow-label FLMODE]
               [dot1q vlan DOT1QTAG] [group-id GROUPID]
            '''
   noOrDefaultSyntax = '''debug [pseudowire-id PWID neighbor PWPEER] ...'''
   data = {
      'debug': 'LDP pseudowire debug',
      'pseudowire-id': 'Pseudowire ID',
      'PWID': CliMatcher.IntegerMatcher( 1, 0xFFFFFFFF,
                                           helpdesc='Pseudowire ID' ),
      'neighbor': 'Pseudowire neighbor',
      'PWPEER': IpAddrMatcher.IpAddrMatcher( helpdesc='Pseudowire neighbor' ),
      'peer-label': 'Peer label',
      'LABEL': labelMatcher,
      'type': 'Pseudowire type',
      'PWTYPE': CliMatcher.IntegerMatcher( 4, 5,
                                          helpdesc='Pseudowire type (4 or 5)' ),
      'mtu': mtuMatcher,
      'MTU': mtuNumMatcher,
      'control-word': 'Enable control word',
      'flow-label': 'Configure flow label mode',
      'FLMODE': CliMatcher.EnumMatcher( {
         'both': 'Configure flow label for transmit and receive',
         'transmit': 'Configure flow label for transmit',
         'receive': 'Configure flow label for receive',
      } ),
      'dot1q': '802.1Q VLAN tag',
      'vlan': '802.1Q VLAN tag',
      'DOT1QTAG': CliMatcher.IntegerMatcher( 1, 4094,
                                               helpdesc='802.1Q VLAN tag' ),
      'group-id': 'Pseudowire group ID',
      'GROUPID': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                                              helpdesc='Pseudowire group ID' ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      pwId = args[ 'PWID' ]
      pwPeer = args[ 'PWPEER' ]
      peerLabel = args[ 'LABEL' ]
      pwType = args[ 'PWTYPE' ]
      if pwType == 4:
         pwType = 'pwType4'
      elif pwType == 5:
         pwType = 'pwType5'
      mtu = args.get( 'MTU' )
      controlWord = 'control-word' in args
      flowLabelMode = args.get( 'FLMODE', 'disabled' )
      flowLabelMode = getattr( FlowLabelMode, flowLabelMode )
      requestedDot1qTag = args.get( 'DOT1QTAG' )
      groupId = args.get( 'GROUPID' )

      respKey = Tac.Value( 'Pseudowire::PseudowireLdpResponseKey',
                           Arnet.IpGenAddr( pwPeer ), pwId )
      mockResp = pwaMockLdpStatus.newResponse( respKey )

      mockResp.peerLabel = peerLabel
      mockResp.remotePwType = pwType
      mockResp.status = 'up'
      mockResp.controlWord = controlWord
      mockResp.peerFlowLabelMode = flowLabelMode
      mockResp.flowLabelType = Tac.enumValue( FlowLabelMode, flowLabelMode )
      mockResp.intfDescription = ''
      mockResp.requestedDot1qTag = requestedDot1qTag or 0
      mockResp.mtu = mtu or 0
      mockResp.groupId = groupId or 0
      mockResp.pwStatusFlags = 0
      mockResp.peerVccvCcTypes = pwPingCcEnumToVal[ PwPingCcType.ra ]
      mockResp.peerVccvCvTypes = pwPingCvEnumToVal[ PwPingCvType.lspPing ]
      mockResp.changeCount += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pwId = args.get( 'PWID' )
      pwPeer = args.get( 'PWPEER' )

      if pwId and pwPeer:
         respKey = Tac.Value( 'Pseudowire::PseudowireLdpResponseKey',
                              Arnet.IpGenAddr( pwPeer ), pwId )
         del pwaMockLdpStatus.response[ respKey ]
      else:
         pwaMockLdpStatus.response.clear()

MplsLdpPseudowiresConfigMode.addCommandClass( DebugPwLdpCmd )

# -------------------------------------------------------------------------------
# The "[no|default] debug pseudowire PWNAME neighbor PWPEER peer-label LABEL
# [mtu MTU] [control-word]" hidden command under "vpws EVINAME" config mode
# -------------------------------------------------------------------------------
def debugPwBgpCmdHandler( mode, args ):
   vpwsName = getVpwsEviName( mode.name )
   pwName = args[ 'PWNAME' ]
   pwPeer = args[ 'PWPEER' ]
   peerLabel = args[ 'LABEL' ]
   localMtu = args.get( 'LOCALMTU' )
   localControlWord = 'control-word-l' in args
   peerMtu = args.get( 'PEERMTU' )
   peerControlWord = 'control-word-p' in args

   respKey = PseudowireBgpResponseKey( vpwsName, pwName )
   mockResp = pwaMockBgpStatus.newBgpResponse( respKey )

   mockResp.controlWord = localControlWord
   if localMtu is not None:
      mockResp.mtu = localMtu
   mockResp.status = "noTunnel"

   peerAddr = Arnet.IpGenAddr( pwPeer )
   peer = mockResp.peerInfo.newMember( peerAddr )
   peer.controlWord = peerControlWord
   if peerMtu is not None:
      peer.mtu = peerMtu
   peer.label = peerLabel
   peer.status = "noTunnel"

def debugPwBgpCmdNoHandler( mode, args ):
   vpwsName = getVpwsEviName( mode.name )
   pwName = args.get( 'PWNAME' )

   if pwName:
      respKey = PseudowireBgpResponseKey( vpwsName, pwName )
      del pwaMockBgpStatus.bgpResponse[ respKey ]
   else:
      pwaMockBgpStatus.bgpResponse.clear()

# ---------------------------------------------
# Local connector interface review on error-disable removal
# ---------------------------------------------
class ConnectorInterfaceRecoveryCmd( CliCommand.CliCommandClass ):
   syntax = """connector interface recovery review delay MIN MAX"""
   noOrDefaultSyntax = """connector interface recovery review delay ..."""
   data = {
         "connector": globalConnectorMatcher,
         "interface": interfaceMatcher,
         "recovery": "Interface recovery from error disabled state",
         "review": "Interface recovery review",
         "delay": "Range of maximum seconds to wait for interface to recover "
                  "before declaring interface state",
         "MIN": CliMatcher.IntegerMatcher( 10, 600, helpdesc="Minimum delay" ),
         "MAX": CliMatcher.IntegerMatcher( 15, 900, helpdesc="Maximum delay" ),
      }

   @staticmethod
   def handler( mode, args ):
      try:
         pwConfig.intfReviewDelayRange = Tac.Value( "Pseudowire::TimeRange",
               args[ 'MIN' ], args[ 'MAX' ] )
      except IndexError: # Tac::RangeException
         mode.addError( "Interface recovery review delay range invalid" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pwConfig.intfReviewDelayRange = pwConfig.intfReviewDelayRangeDefault

PatchPanelConfigMode.addCommandClass( ConnectorInterfaceRecoveryCmd )

# -------------------------------------------------------------------------------
# [ no | default ] connector interface patch bgp vpws remote-failure errdisable
# -------------------------------------------------------------------------------
class ConnectorInterfaceBgpVpwsErrdisableCmd( CliCommand.CliCommandClass ):
   syntax = """connector interface patch bgp vpws remote-failure errdisable"""
   noOrDefaultSyntax = syntax
   data = {
         "connector": globalConnectorMatcher,
         "interface": interfaceMatcher,
         "patch": "Configure for all patches",
         "bgp": bgpMatcher,
         "vpws": vpwsMatcher,
         "remote-failure": "Remote status",
         "errdisable": "Error disable local interface"
      }

   @staticmethod
   def handler( mode, args ):
      pwConfig.patchBgpVpwsErrdisable = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pwConfig.patchBgpVpwsErrdisable = pwConfig.patchBgpVpwsErrdisableDefault

PatchPanelConfigMode.addCommandClass( ConnectorInterfaceBgpVpwsErrdisableCmd )

# -------------------------------------------------------------------------------
# [ no | default ] connector interface patch bgp vpws standby errdisable
# -------------------------------------------------------------------------------
class ConnectorInterfaceBgpVpwsStandbyErrdisableCmd( CliCommand.CliCommandClass ):
   syntax = """connector interface patch bgp vpws standby errdisable"""
   noOrDefaultSyntax = syntax
   data = {
         "connector": globalConnectorMatcher,
         "interface": interfaceMatcher,
         "patch": "Configure for all patches",
         "bgp": bgpMatcher,
         "vpws": vpwsMatcher,
         "standby": "Standby",
         "errdisable": "Error disable local interface"
      }

   @staticmethod
   def handler( mode, args ):
      pwConfig.bgpVpwsStandbyErrdisable = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pwConfig.bgpVpwsStandbyErrdisable = pwConfig.bgpVpwsStandbyErrdisableDefault

if toggleVpwsStandbyErrDisableEnabled():
   PatchPanelConfigMode.addCommandClass(
                              ConnectorInterfaceBgpVpwsStandbyErrdisableCmd )

def Plugin( entityManager ):
   global pwConfig, pwHwCapability
   global pwRunnability
   global pwaMockLdpStatus, pwaMockBgpStatus
   global bgpMacVrf

   pwConfig = ConfigMount.mount( entityManager, "pseudowire/config",
                                 "Pseudowire::Config", "w" )
   pwRunnability = ConfigMount.mount( entityManager, "pseudowire/runnability/config",
                                      "Pseudowire::Runnability", "w" )
   pwHwCapability = LazyMount.mount( entityManager,
                                     "routing/hardware/pseudowire/capability",
                                     "Pseudowire::Hardware::Capability", "r" )
   pwaMockLdpStatus = ConfigMount.mount( entityManager,
                                         "pseudowire/ldp/mockStatus",
                                         "Pseudowire::PseudowireLdpStatus",
                                         "w" )
   pwaMockBgpStatus = ConfigMount.mount( entityManager,
                                         "pseudowire/bgp/mockStatus",
                                         "Pseudowire::PseudowireBgpStatus",
                                         "w" )
   bgpMacVrf = LazyMount.mount( entityManager, "routing/bgp/macvrf",
                                "Routing::Bgp::MacVrfConfigDir", "r" )

   MplsCli.checkConflictingLabelRangeHook.addExtension(
         checkConflictingLabelRangeForMplsStaticPwLocalLabel )
