# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Cli config modes and commands used for configuring common pseudowire profiles (see
# AID10962), for example:
#    mpls pseudowires
#       profiles
#          profile PROFILE_1
#             control-word
#             mtu 9000
#             pseudowire-type raw
#             label flow transmit

import BasicCli
import CliCommand
import CliMatcher
from CliMode.PseudowireProfile import (
   MplsPseudowireProfileMode,
   MplsPseudowireProfilesMode,
)
import ConfigMount
from PseudowireLib import (
   FlowLabelMode,
)
from TypeFuture import TacLazyType

PseudowireProfileConfig = TacLazyType( 'Pseudowire::PseudowireProfileConfig' )
PseudowireProfileDir = TacLazyType( 'Pseudowire::PseudowireProfileDir' )
PseudowireType = TacLazyType( 'Pseudowire::PseudowireType' )

# The mounted PseudowireProfileDir collection containing the common profiles.
pwProfileDir = None

# Matcher that matches (and auto-completes) a common profile name.
pwProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ( pwProfileDir.profile.keys() ),
      'MPLS pseudowire profile name' )

# Sets the specified common pseudowire profile in the collection.
def setProfile( profileName, mtu=None, controlWord=None, pwType=None,
                flowLabelMode=None ):
   def newValue( newValue, oldValue ):
      return newValue if newValue is not None else oldValue

   # Get the current profile or create an empty one with default values.
   if profileName in pwProfileDir.profile:
      curProfile = pwProfileDir.profile[ profileName ]
   else:
      curProfile = PseudowireProfileConfig( profileName )

   # Build the new profile values.
   newMtu = newValue( mtu, curProfile.mtu )
   newPwType = newValue( pwType, curProfile.pwType )
   newControlWord = newValue( controlWord, curProfile.controlWord )
   newFlowLabelMode = newValue( flowLabelMode, curProfile.flowLabelMode )

   # Build and set the profile.
   newProfile = PseudowireProfileConfig( profileName, mtu=newMtu,
                                         controlWord=newControlWord,
                                         pwType=newPwType,
                                         flowLabelMode=newFlowLabelMode )
   pwProfileDir.profile.addMember( newProfile )

# Removes the common pseudowire profile with the specified name.
def delProfile( profileName ):
   if profileName in pwProfileDir.profile:
      del pwProfileDir.profile[ profileName ]

# Removes all common pseudowire profiles.
def clearAllProfiles():
   pwProfileDir.profile.clear()

# -------------------------------------------------------------------------------
# "profiles" config mode (mpls-pw-pf) under "mpls pseudowires" (mpls-pw) config
# mode.
# -------------------------------------------------------------------------------
class MplsPseudowireProfilesConfigMode( MplsPseudowireProfilesMode,
                                        BasicCli.ConfigModeBase ):
   name = 'MPLS pseudowire profiles configuration'

   def __init__( self, parent, session ):
      MplsPseudowireProfilesMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# -------------------------------------------------------------------------------
# "profile <name>" config mode (mpls-pw-pf-<name>) under "profiles" (mpls-pw-pf)
# config mode.
# -------------------------------------------------------------------------------
class MplsPseudowireProfileConfigMode( MplsPseudowireProfileMode,
                                       BasicCli.ConfigModeBase ):
   name = 'MPLS pseudowire profile configuration'

   def __init__( self, parent, session, profileName ):
      MplsPseudowireProfileMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   # Called by LdpPseudowireMtuMtuCmd to set the MTU.
   def setMtu( self, args ):
      setProfile( self.profileName, mtu=args[ 'MTU' ] )

   # Called by LdpPseudowireMtuMtuCmd to clear the MTU.
   def noMtu( self, args ):
      setProfile( self.profileName, mtu=PseudowireProfileConfig.mtuDefault )

   # Called by ControlWordCmd to set the control-word.
   def setControlWord( self, args ):
      setProfile( self.profileName, controlWord=True )

   # Called by ControlWordCmd to clear the control-word.
   def noControlWord( self, args ):
      setProfile( self.profileName,
                  controlWord=PseudowireProfileConfig.controlWordDefault )

   # Called by PwTypeCmd to set the pseudowire type.
   def setPwType( self, args ):
      if 'raw' in args:
         pwType = PseudowireType.pwType5
      else:
         pwType = PseudowireType.pwType4
      setProfile( self.profileName, pwType=pwType )

   # Called by PwTypeCmd to clear the pseudowire type.
   def noPwType( self, args ):
      setProfile( self.profileName, pwType=PseudowireProfileConfig.pwTypeDefault )

   # Called by FlowLabelCmd to set the flow label mode.
   def setFlowLabel( self, args ):
      if 'transmit' in args:
         mode = FlowLabelMode.transmit
      elif 'receive' in args:
         mode = FlowLabelMode.receive
      else:
         mode = FlowLabelMode.both
      setProfile( self.profileName, flowLabelMode=mode )

   # Called by FlowLabelCmd to clear the flow label mode.
   def noFlowLabel( self, args ):
      setProfile( self.profileName,
                  flowLabelMode=PseudowireProfileConfig.flowLabelModeDefault )

# Adds the specified command to the MplsPseudowireProfileConfigMode. Used by the
# Pseudowire dynamic plugin to add the various pseudowire profile commands (e.g. mtu
# control-word, etc) to the common profile mode.
def addProfileCommandClass( cliCommand ):
   MplsPseudowireProfileConfigMode.addCommandClass( cliCommand )

# --------------------------------------------------------------------------------
# [ no | default ] profiles
#
# Note that this command will be added to MplsPseudowiresConfigMode in the
# Pseudowire dynamic plugin.
# --------------------------------------------------------------------------------
class MplsPseudowireProfilesCmd( CliCommand.CliCommandClass ):
   syntax = 'profiles'
   noOrDefaultSyntax = syntax
   data = {
      'profiles': 'Configure MPLS pseudowire profiles',
   }

   @staticmethod
   def handler( mode, args ):
      profilesMode = mode.childMode( MplsPseudowireProfilesConfigMode )
      mode.session_.gotoChildMode( profilesMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearAllProfiles()

# --------------------------------------------------------------------------------
# [ no | default ] profile PROFILE_NAME
# --------------------------------------------------------------------------------
class MplsPseudowireProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'profile': 'Configure an MPLS pseudowire profile',
      'PROFILE_NAME': pwProfileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      # Make sure a profile of the specified name exists - if not, one will be added
      # with default settings.
      profileName = args[ 'PROFILE_NAME' ]
      setProfile( profileName )

      profileMode = mode.childMode( MplsPseudowireProfileConfigMode,
                                    profileName=profileName )
      mode.session_.gotoChildMode( profileMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Delete the profile of the specified name.
      profileName = args[ 'PROFILE_NAME' ]
      delProfile( profileName )

MplsPseudowireProfilesConfigMode.addCommandClass( MplsPseudowireProfileCmd )

def Plugin( entityManager ):
   global pwProfileDir
   pwProfileDir = ConfigMount.mount( entityManager,
                                     PseudowireProfileDir.mountPath,
                                     'Pseudowire::PseudowireProfileDir',
                                     'w' )
