# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Toggles.PseudowireAgentToggleLib import \
   togglePseudowireShowPatchPanelOptimizedEnabled

from collections import defaultdict

import Cell
import LazyMount
import SharedMem
import Smash
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.PwaCli import (
   pwConfig,
   switchTimeToUtc,
)
from CliPlugin.TunnelCli import getTunnelTypeFromTunnelId as tunIdToType
from CliPlugin.TunnelModels import tunnelTypeStrDict as tunTypeToStr
from PseudowireLib import (
   ConnectorStatusType as CST,
   connStatusFromIntfEncapHwStatus,
   flEnumValDisabled,
   flEnumValTransmit,
   flValToEnum,
   FlowLabelMode,
   getLdpConnectorKey,
)
import Tac
from TypeFuture import TacLazyType

pwStatus = None
pwLfib = None
pwaPatchPanel = None
pwaLocalConnectorStatusColl = None
pwaPseudowireAttributeStatusColl = None
pwaPatchInfoColl = None
pwHwDir = None
pwQosMapHwStatus = None
pwaLabelBinding = None
pwaLdpRequest = None
pwaLdpResponse = None
pwaEbraResponse = None
pwaBgpRequest = None
pwaBgpResponse = None
pwaLdpPwResetRequest = None
pwVpwsTunnelTables = {}

EncapType = TacLazyType( 'Tunnel::TunnelTable::EncapType' )
TunnelId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )
TunnelType = TacLazyType( "Tunnel::TunnelTable::TunnelType" )
PwLdpReqKey = TacLazyType( "Pseudowire::PseudowireLdpRequestKey" )
PwLdpRespKey = TacLazyType( "Pseudowire::PseudowireLdpResponseKey" )
PwBgpReqKey = TacLazyType( "Pseudowire::PseudowireBgpRequestKey" )
PwBgpRespKey = TacLazyType( "Pseudowire::PseudowireBgpResponseKey" )
PwConnectorType = TacLazyType( "Pseudowire::ConnectorType" )
PwIntfId = TacLazyType( 'Arnet::PseudowireIntfId' )
PwType = TacLazyType( "Pseudowire::PseudowireType" )
SubIntfId = TacLazyType( "Arnet::SubIntfId" )
VlanId = TacLazyType( "Bridging::VlanId" )
CSH = TacLazyType( 'Pseudowire::PseudowireConnStatusHelper' )
IntfEncapHwStatus = TacLazyType( "Ebra::IntfEncapHwStatus" )
PseudowireUserKey = TacLazyType( 'Pseudowire::PseudowireUserKey' )
PseudowireLdpRequestKey = TacLazyType( 'Pseudowire::PseudowireLdpRequestKey' )
PseudowirePatchState = TacLazyType( 'Pseudowire::PseudowirePatchState' )
PseudowirePatchType = TacLazyType( 'Pseudowire::PseudowirePatchType' )
LdpPwResetRequestEntry = TacLazyType( 'Pseudowire::LdpPwResetRequestEntry' )
PwaModel = CliDynamicPlugin( 'PwaModel' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )

def isPatchUp( status ):
   '''Returns if patch->status is up.
   @status is patchPanel->patch->status which is of
         type PseudowireConnectorStatus (and not the human readable
         versions of them). (The PwaModel.Patch.status is of enum with
         "Up", "Down", "Unprogrammed", "Admin down")'''
   return status == CST.up

# ---------------------------------------------
# Local connector helper methods
# Local connector = tagged + port connectors
# ---------------------------------------------

def getLocalConnectorProgrammedStatus( serviceIntfId ):
   """Check if the local connector has been programmed in ale (new API).

   If there is no entry, then consider the connector down (hardware programming is
   not complete yet), so that patch and connector status show 'unprogrammed'.
   """
   reason = pwHwDir.hwStatus.get( serviceIntfId, IntfEncapHwStatus.failed )
   return connStatusFromIntfEncapHwStatus.get( reason, CST.connectorUnprogrammed )

def getLocalConnectorStatus( connKey, patch, patchType ):
   '''Returns the status of a local connector identified
   by connKey. It has the logic to check the status of the
   local connector in
   1. PatchPanel
   2. LocalConnectorStatus
   3. Ale

   For bug 144948,  we'll also check the status of the patch it belongs to,
   if applicable, in order to determine if the connector status is relevant,
   ( if the patch is admin down, the connectors should also be admin down )
   '''
   connStatus = pwaPatchPanel.localConnStatus.get( connKey )
   if not connStatus:
      return CST.down

   # If the patch is available, then check if patch is enabled.
   # If patch is disabled, then always return 'adminDown' as status of the
   # constituent connectors
   if not patch or not patch.enabled:
      return CST.adminDown

   status = connStatus.status
   if CSH.isDown( status ):
      return status

   # status is Up. Check LCS get status from there.
   connStatus = pwaLocalConnectorStatusColl.connectorStatus.get( connKey, None )
   if not connStatus:
      return CST.noLocalEnd

   status = connStatus.status
   if CSH.isDown( status ):
      return status
   elif status == CST.disabledByPw:
      # The interface is error disabled. Let's return this
      # immediately
      return status

   # Note that we explicitly are not checking for whether encap adjacency is
   # written by PWA before we look at what Ale is reporting. If we checked
   # for encap adjacency, we would see output like:
   #   conn1 status = Cli conflict
   #   conn2 status = Encap not present (instead of Up)
   # The only way to evaluate conn2's status would be that it knows the
   # reason for its non-installation is conn1's status
   #
   # Similarly conn1's status would depend on conn2's status and thus
   # evaluating either of them indenpendently would be unnecassry
   # logic.
   #
   # Relying on "show patch panel forwarding" is more appropriate
   # to look at published state as that only walks over all
   # encap and decap entries.

   return getLocalConnectorProgrammedStatus( connKey.serviceIntfId() )

def getErrDisableTrigger( connKey, patch, patchType ):
   retString = PwaModel.ErrDisableState.none

   if not connKey.isPort() or patchType != PwaModel.PatchTypes.loRmt:
      return None

   patchConfig = pwConfig.patch.get( patch.patchName )

   remoteType = None
   alternate = False
   suppressErrorDisable = None
   for cliConnector in patchConfig.connector.values():
      connType = cliConnector.connectorKey.connectorType
      if connType == PwConnectorType.bgp:
         assert remoteType is None
         remoteType = connType
      elif connType == PwConnectorType.ldp:
         # If an alternate connector is configured we will have
         # 2 ldp remote connectors in the same patch
         assert remoteType is None or remoteType == PwConnectorType.ldp
         remoteType = connType
         alternate = alternate or cliConnector.alternate
      elif connType == PwConnectorType.local:
         assert suppressErrorDisable is None
         suppressErrorDisable = cliConnector.noErrDisable
   if suppressErrorDisable:
      retString = PwaModel.ErrDisableState.suppressedByNoErrorDisable
   elif alternate:
      # If the alternate flag is set we will implicitly not error disable
      retString = PwaModel.ErrDisableState.suppressedByAlternateConnector
   elif remoteType == PwConnectorType.bgp and pwConfig.patchBgpVpwsErrdisable:
      retString = PwaModel.ErrDisableState.remoteFailure
   elif remoteType == PwConnectorType.ldp:
      retString = PwaModel.ErrDisableState.remoteNotForwarding

   return retString

def getTaggedConnectorOptimized( connectorKey, connector, patchInfo ):
   '''Populates the tagged connector specific details in
   conn (PwaModel.Connector) and returns the modified
   connector and status enum.'''
   # pylint: disable=redefined-outer-name
   taggedConnector = patchInfo.taggedConnector.get( connectorKey )
   # pylint: enable=redefined-outer-name
   if not taggedConnector:
      return connector
   connector.status = taggedConnector.status
   connector.taggedConnectorInfo = \
      PwaModel.TaggedConnector(
         interface=taggedConnector.interface,
         dot1qTag=taggedConnector.dot1qTag )

   return connector

def taggedConnector( connKey, patch, patchType, conn ):
   '''Populates the tagged connector specific details in
   conn (PwaModel.Connector) and returns the modified
   connector and status enum.'''
   conn.taggedConnectorInfo = PwaModel.TaggedConnector(
            interface=connKey.localIntfId(),
            dot1qTag=connKey.legacyDot1qTag() )
   conn.status = getLocalConnectorStatus( connKey, patch, patchType )
   return conn

def getPortConnectorOptimized( connectorKey, connector, patchInfo ):
   '''Populates the port connector specific details in
   conn (PwaModel.Connector) and returns the modified
   connector and status enum.'''
   # pylint: disable=redefined-outer-name
   portConnector = patchInfo.portConnector.get( connectorKey )
   # pylint: enable=redefined-outer-name
   if not portConnector:
      return connector
   connector.status = portConnector.status
   connector.portConnectorInfo = \
      PwaModel.PortConnector(
         interface=portConnector.interface,
         errorDisableTrigger=portConnector.errorDisableTrigger )

   return connector

def portConnector( connKey, patch, patchType, conn ):
   '''Populates the port connector specific details in
   conn (PwaModel.Connector) and returns the modified
   connector and status enum.'''
   errDisable = getErrDisableTrigger( connKey, patch, patchType )
   conn.portConnectorInfo = PwaModel.PortConnector( interface=connKey.localIntfId(),
                                                    errorDisableTrigger=errDisable )
   conn.status = getLocalConnectorStatus( connKey, patch, patchType )
   return conn

def getConnectorType( pwIntfId ):
   # VlanId::invalid() indicates that it is a port type
   lcInfo = pwStatus.localConnectorInfo.get( pwIntfId )
   if lcInfo and SubIntfId.isSubIntfId( lcInfo.parentIntfId ):
      return "port" # treat subinterface connector as port to avoid dot1q output
   elif PwIntfId.vlanId( pwIntfId ) == VlanId.invalid:
      return "port"
   else:
      return "tagged"

# ---------------------------------------------
# Remote pseudowire connector helper methods
# ---------------------------------------------

def getRemotePseudowireProgrammedStatus( serviceIntfIds ):
   """"Checks if the remote pseudowire has been programmed in Ale (new API).

   The new Ale API does not publish separate programming states for enap and decap,
   as a result there is no dedicated programming status for a the remote end.
   Simply mirror the local connector programmed status.

   For FXC, return 'partiallyUp' if not all local connectors are programmed, but
   at least one if them is.
   """
   numUp = 0
   numUnprogrammed = 0
   for serviceIntfId in serviceIntfIds:
      status = getLocalConnectorProgrammedStatus( serviceIntfId )
      if status == CST.up:
         numUp += 1
      else:
         numUnprogrammed += 1
   if numUp: # at least one local connector is programmed
      return CST.partiallyUp if numUnprogrammed else CST.up
   else: # all local connectors are unprogrammed
      return CST.connectorUnprogrammed

def getRemotePseudowireConnectorStatus( connKey, patch, loIntfIds=None ):
   '''Returns the status of a remote pseudowire connector identified
   by connKey. It has the logic to check the status of the remote 
   connector in
   1. PatchPanel
   2. LabelBinding (if one exists)
   3. PseudowireLdpRequest or PseudowireBgpRequest (if a request is made)
   4. PseudowireLdpResponse or PseudowireBgpResponse (if a response exists)
   5. PseudowireAttributeStatus
   6. Ale: if loIntfIds is provided, check both encap/local connector status
      otherwise only report if the *localLabel* (decap side) is programmed

   For bug 144948,  we'll also check the status of the patch it belongs to,
   if applicable, in order to determine if the connector status is relevant
   '''
   connStatus = pwaPatchPanel.remoteConnStatus.get( connKey )

   if connStatus is None:
      return CST.down

   # connKey exists in patchpanel. Get status from here first
   status = connStatus.status

   if CSH.isDown( status ):
      return status

   # If the patch is available, then check if patch is enabled.
   # If patch is disabled, then always return 'adminDown' as status of the
   # constituent connectors
   if not patch or not patch.enabled:
      return CST.adminDown

   # Status is patchpanel is up, lets check if label binding succeeded
   localLabel = pwaLabelBinding.localLabel.get(
      PseudowireUserKey.fromConnectorKey( connKey ) )

   # For MPLS static pwe connectors, local label may be configured in the CLI
   # and hence may not be present in the labelBindingColl
   if connKey.connectorType != PwConnectorType.mplsStatic and localLabel is None:
      # localBinding failed
      return CST.noLocalMapping

   # localLabel is assigned. Let's check remote connector request and response
   if connKey.connectorType == PwConnectorType.ldp:
      status = getLdpRequestResponseStatus( connStatus )
   elif connKey.connectorType == PwConnectorType.bgp:
      status = getBgpRequestResponseStatus( connKey )
   # There is no request response for MPLS static connectors
   elif connKey.connectorType != PwConnectorType.mplsStatic:
      assert False, f"Unhandled connectorType {connKey.connectorType}"
   if status != CST.up:
      return status

   # Check PseudowireAttributeStatus
   pwAttrStatus = pwaPseudowireAttributeStatusColl.connectorStatus.get(
      PseudowireUserKey.fromConnectorKey( connKey ) )

   if pwAttrStatus is None:
      return CST.noRemoteEnd

   if CSH.isDown( pwAttrStatus.status ) or CSH.isStandby( pwAttrStatus.status ):
      # BgpConnectorResponseSm publishes an unknown status to
      # PseudowireAttributeStatusColl if there is no BGP response
      if pwAttrStatus.status == CST.unknown:
         return CST.noRemoteEnd
      return pwAttrStatus.status

   # Check if the corresponding encap and decap entries exist. If they don't,
   # ale can't even report that they are unprogrammed
   if not loIntfIds:
      return CST.unprogrammed

   # Note that we explicitly are not checking for whether encap adjacency is
   # written by PWA before we look at what Ale is reporting. If we checked
   # for encap adjacency, we would see output like:
   #   conn1 status = Cli conflict
   #   conn2 status = Encap not present (instead of Up)
   # The only way to evaluate conn2's status would be that it knows the
   # reason for its non-installation is conn1's status
   #
   # Similarly conn1's status would depend on conn2's status and thus
   # evaluating either of them indenpendently would be unnecassry
   # logic.
   #
   # Relying on "show patch panel forwarding" is more appropriate
   # to look at published state as that only walks over all
   # encap and decap entries.

   return getRemotePseudowireProgrammedStatus( loIntfIds )

# ---------------------------------------------
# Ldp pseudowire connector helper methods
# ---------------------------------------------

def getLdpRequestResponseStatus( connStatus ):
   """Returns a status corresponding to the presence of LDP request and response"""
   neighbor = connStatus.connector.neighborAddr
   pwId = connStatus.connector.pwId

   # Check LDP request
   reqKey = PwLdpReqKey( neighbor, pwId )

   if reqKey not in pwaLdpRequest.ldpRequest:
      return CST.noLocalEnd

   # Check LDP response
   respKey = PwLdpRespKey( neighbor, pwId )

   if respKey not in pwaLdpResponse.response:
      # For the transient instance when LDP request is created
      # but LDP response doesn't exist, return noRemoteEnd
      return CST.noRemoteEnd

   return CST.up

def getLdpPwFlowLabelType( neighbor, pwId ):
   '''Returns the flowLabelType from pseudowire LDP response'''
   if neighbor and pwId:
      respKey = PwLdpRespKey( neighbor, pwId )
      response = pwaLdpResponse.response.get( respKey )
      if response is not None:
         return flValToEnum[ response.flowLabelType ]
   return flValToEnum[ flEnumValDisabled ]

def getLdpPwForwardingState( connKey, neighbor, pwId ):
   '''Returns the ( forwarding, errorFlags = PwaModel.LdpPseudowireErrorFlags )
   tuple for the provided remote connector identified by (connKey, neighbor, pwId).
   forwarding - indicates if the peer is forwarding pseudowire bidirectional
                traffic
   errorFlags - if the peer is not forwarding traffic, the appropriate error flags
                are returned.
   '''
   respKey = PwLdpRespKey( neighbor, pwId )
   response = pwaLdpResponse.response.get( respKey )
   if response is not None:
      statusVal = response.pwStatusFlags
      flags = Tac.Value( "Pseudowire::PseudowireLdpStatusFlags", statusVal )

      forwarding = flags.forwardingState != 'fault'
      errorFlags = None
      if not forwarding:
         errorFlags = PwaModel.LdpPseudowireErrorFlags(
                  notForwarding=flags.pwNotForwarding or None,
                  etInFault=flags.localACIngrRecvFault or None,
                  etOutFault=flags.localACEgrTransFault or None,
                  tunInFault=flags.localPsnPwIngrRecvFault or None,
                  tunOutFault=flags.localPsnPwEgrTransFault or None
               )

      return forwarding, errorFlags

   return False, None

def getLdpPwNeighborConfig( connKey, neighbor, pwId ):
   '''Returns LdpPseudowirePeerConfig model for an LDP remote
   connector.'''
   reqKey = PwLdpReqKey( neighbor, pwId )
   respKey = PwLdpRespKey( neighbor, pwId )
   req = pwaLdpRequest.ldpRequest.get( reqKey, None )
   resp = pwaLdpResponse.response.get( respKey, None )
   if not req:
      return None, PwType.pwTypeNone

   nbrState = PwaModel.LdpPseudowirePeerConfig()

   if resp is None:
      return nbrState, req.pwType

   nbrState.mtu = resp.mtu
   nbrState.intfDescription = resp.intfDescription

   nbrState.label = resp.peerLabel
   nbrState.groupId = resp.groupId

   dot1qTagRequested = ( resp.requestedDot1qTag != 0 )
   if dot1qTagRequested:
      nbrState.dot1qTagRequested = resp.requestedDot1qTag

   nbrState._isLocal = False # pylint:disable=W0212

   nbrState.flowLabelCapability = resp.peerFlowLabelMode

   nbrState.vccvCcTypes = resp.peerVccvCcTypes
   nbrState.vccvCvTypes = resp.peerVccvCvTypes

   return nbrState, req.pwType

def getLdpPwLocalConfig( connKey, neighbor, pwId ):
   '''Returns LdpPseudowirePeerConfig model for a local connector.'''
   localState = PwaModel.LdpPseudowirePeerConfig()
   localLabel = pwaLabelBinding.localLabel.get(
      PseudowireUserKey.fromConnectorKey( connKey ) )
   if localLabel:
      localState.label = localLabel

   connStatus = pwaPatchPanel.remoteConnStatus.get( connKey )
   if not connStatus:
      return None

   localState.groupId = connStatus.connector.groupId
   localState.mtu = connStatus.mtu

   localState._isLocal = True # pylint:disable=W0212
   reqKey = PwLdpReqKey( neighbor, pwId )
   req = pwaLdpRequest.ldpRequest.get( reqKey )
   if req:
      localState.vccvCcTypes = req.localVccvCcTypes
      localState.vccvCvTypes = req.localVccvCvTypes
      localState.flowLabelCapability = connStatus.connector.flowLabelMode

   return localState

def getQosMapType( mapType ):
   if mapType == 'pwDecapDscpToTc':
      return 'dscpToTc'
   elif mapType == 'pwDecapCosToTc':
      return 'cosToTc'
   else:
      return 'default-map'

def getLdpConnectorOptimized( connectorKey, connector, patchInfo ):
   '''Populates an LDP pseudowire connector specific details in
   conn (PwaModel.Connector) and returns the modified connector
   and status enum.'''
   ldpConnector = patchInfo.ldpConnector.get( connectorKey )
   if not ldpConnector:
      return connector
   connector.status = ldpConnector.status
   localConfig = None
   neighborConfig = None
   neighborFaultFlags = None
   tunnels = []

   if ldpConnector.localConfig:
      localConfig = ldpConnector.localConfig
      localConfig = \
         PwaModel.LdpPseudowirePeerConfig(
            label=localConfig.label,
            groupId=localConfig.groupId,
            mtu=localConfig.mtu,
            _isLocal=True,
            vccvCcTypes=localConfig.vccvCcTypes,
            vccvCvTypes=localConfig.vccvCvTypes,
            flowLabelCapability=localConfig.flowLabelCapability )

   if ldpConnector.neighborConfig:
      neighborConfig = ldpConnector.neighborConfig
      neighborConfig = \
      PwaModel.LdpPseudowirePeerConfig(
         mtu=neighborConfig.mtu,
         intfDescription=neighborConfig.intfDescription,
         label=neighborConfig.label,
         groupId=neighborConfig.groupId,
         dot1qTagRequested=neighborConfig.dot1qTagRequested,
         _isLocal=False,
         flowLabelCapability=neighborConfig.flowLabelCapability,
         vccvCcTypes=neighborConfig.vccvCcTypes,
         vccvCvTypes=neighborConfig.vccvCvTypes )

   if len( ldpConnector.tunnel ) > 0:
      for _, tunnelInfo in sorted( ldpConnector.tunnel.items() ):
         tunnels.append(
            PwaModel.TunnelInfo(
               tunnelType=tunnelInfo.tunnelTypeStr,
               tunnelIndex=tunnelInfo.tunnelIndex,
               color=tunnelInfo.color ) )

   if ldpConnector.neighborFaultFlags:
      neighborFaultFlags = ldpConnector.neighborFaultFlags
      neighborFaultFlags = PwaModel.LdpPseudowireErrorFlags(
         notForwarding=neighborFaultFlags.notForwarding,
         etInFault=neighborFaultFlags.etInFault,
         etOutFault=neighborFaultFlags.etOutFault,
         tunInFault=neighborFaultFlags.tunInFault,
         tunOutFault=neighborFaultFlags.tunOutFault
      )

   connector.ldpPseudowireConnectorInfo = \
      PwaModel.LdpPseudowireConnector(
         name=ldpConnector.connName,
         neighbor=ldpConnector.neighbor,
         pseudowireId=ldpConnector.pseudowireId,
         color=ldpConnector.color,
         pseudowireType=ldpConnector.pseudowireType,
         controlWord=ldpConnector.controlWord,
         flowLabelUsed=ldpConnector.flowLabelUsed,
         localConfig=localConfig,
         neighborConfig=neighborConfig,
         neighborForwarding=ldpConnector.neighborForwarding,
         neighborFaultFlags=neighborFaultFlags,
         tunnels=tunnels,
         qosMapName=ldpConnector.qosMapName,
         qosMapType=( getQosMapType( ldpConnector.qosMapType )
                      if ldpConnector.qosMapType and
                      ldpConnector.qosMapType != "defaultMapType" else
                      None ) )

   return connector

def ldpPseudowireConnector( connKey, connStatus, patch, conn ):
   '''Populates an LDP pseudowire connector specific details in
   conn (PwaModel.Connector) and returns the modified connector
   and status enum.
   @connStatus - connectorStatus in cliPatch/patchPanel
   '''
   connInfo = PwaModel.LdpPseudowireConnector( name=connKey.name )
   connInfo.tunnels = None

   if connStatus.connector.neighborAddrPresent:
      connInfo.neighbor = connStatus.connector.neighborAddr

   if connStatus.connector.pwIdPresent:
      connInfo.pseudowireId = connStatus.connector.pwId

   if connStatus.connector.colorPresent:
      connInfo.color = connStatus.connector.color

   connInfo.pseudowireType = PwType.pwTypeNone
   connInfo.controlWord = connStatus.connector.controlWord

   # Assume it to be not up. We get exact state later
   connInfo.neighborForwarding = False

   connInfo.flowLabelUsed = getLdpPwFlowLabelType( connInfo.neighbor,
                                                   connInfo.pseudowireId )

   # If either of the mandatory pieces of information are missing,
   # there is nothing else we can add to the show commands
   if ( not connStatus.connector.neighborAddrPresent or
        not connStatus.connector.pwIdPresent or
        connStatus.mtu == 0 ):
      conn.ldpPseudowireConnectorInfo = connInfo
      conn.status = getRemotePseudowireConnectorStatus( connKey, patch )
      return conn

   forwarding, faultFlags = getLdpPwForwardingState( connKey, connInfo.neighbor,
                                                     connInfo.pseudowireId )

   connInfo.neighborForwarding = forwarding
   connInfo.neighborFaultFlags = faultFlags

   connInfo.localConfig = getLdpPwLocalConfig( connKey, connInfo.neighbor,
                                               connInfo.pseudowireId )
   connInfo.neighborConfig, connInfo.pseudowireType = \
         getLdpPwNeighborConfig( connKey, connInfo.neighbor, connInfo.pseudowireId )

   conn.ldpPseudowireConnectorInfo = connInfo

   # Tunnel information
   # To get the tunnel information in use, we need to retrieve this from
   # PseudowireAttributeStatus which is keyed by the connector key.
   # Verify patch status is up before trying to access partner key.
   connToPatch = pwaPatchPanel.connToPatch.get( connKey )
   if not connToPatch or connToPatch.activeCount() != 1:
      conn.status = getRemotePseudowireConnectorStatus( connKey, patch )
      return conn

   if not patch or not isPatchUp( patch.status ):
      conn.status = getRemotePseudowireConnectorStatus( connKey, patch )
      return conn

   loIntfIds = [ loConnKey.serviceIntfId()
                 for loConnKey in connStatus.localConnectorKey ]

   status = getRemotePseudowireConnectorStatus( connKey, patch, loIntfIds=loIntfIds )
   conn.status = status

   rcStatus = pwaPseudowireAttributeStatusColl.connectorStatus.get(
      PseudowireUserKey.fromConnectorKey( connKey ) )

   if not rcStatus or not rcStatus.peerInfo or CSH.isDown( rcStatus.status ):
      return conn

   peerInfo = list( rcStatus.peerInfo.values() )[ 0 ] # There will be only one
   color = peerInfo.color
   connInfo.tunnels = [
      PwaModel.TunnelInfo( tunnelType=tunTypeToStr[ tunIdToType( tunnelId ) ],
                           tunnelIndex=TunnelId( tunnelId ).tunnelIndex(),
                           color=color.value if color.isSet else None )
      for tunnelId in sorted( peerInfo.tunnelId )
   ]

   qosMap = pwStatus.qosMap.get( patch.pwIntfId )
   if qosMap and qosMap in pwQosMapHwStatus.hwStatus:
      connInfo.qosMapName = qosMap.mapName
      connInfo.qosMapType = getQosMapType( qosMap.mapType )

   return conn

def ldpPseudowireConnectorInfo( connKey, patchConfig ):
   '''Generate daata entry for Connector.alternatives
   This model record basic information that needed for rendering an ldp connector 
   with redundancy
   '''
   connInfo = PwaModel.LdpPseudowireConnectorInfo()

   connStatus = pwaPatchPanel.remoteConnStatus.get( connKey )
   if connStatus:
      ldpKeyValid = True

      if connStatus.connector.neighborAddrPresent:
         connInfo.neighbor = connStatus.connector.neighborAddr
      else:
         ldpKeyValid = False

      if connStatus.connector.pwIdPresent:
         connInfo.pseudowireId = connStatus.connector.pwId
      else:
         ldpKeyValid = False

      if connStatus.connector.colorPresent:
         connInfo.color = connStatus.connector.color

      userKey = PseudowireUserKey.fromConnectorKey( connKey )

      connAttrStatus = pwaPseudowireAttributeStatusColl.connectorStatus.get(
         userKey )
      if connAttrStatus:
         connInfo.status = connAttrStatus.status
      else:
         connInfo.status = CST.down

      if ldpKeyValid:
         ldpReqKey = PseudowireLdpRequestKey( connInfo.neighbor,
                                              connInfo.pseudowireId )
         connInfo.active = str( ldpReqKey ) in pwaLdpRequest.ldpRequest
      else:
         connInfo.active = False

   for cliConnector in patchConfig.connector.values():
      if cliConnector.connectorKey == connKey:
         connInfo.alternate = cliConnector.alternate
         break
   else:
      if connStatus:
         connInfo.alternate = connStatus.alternate

   return connInfo

def addLdpPseudowireAlternatives( connKey, patch, conn, patchConfig ):
   '''Add alternatives attributes in PwaModel Connector and return modified connector
   '''
   primaryKey = connKey
   alternateKey = None
   activeKey = None
   for key in patch.patchConnectorKey:
      if key.isLocal():
         continue
      userKey = PseudowireUserKey.fromConnectorKey( key )
      if key != connKey:
         alternateKey = key
      if userKey in pwaPseudowireAttributeStatusColl.connectorStatus:
         activeKey = key
   activeKey = activeKey or primaryKey
   # build alternatives
   alternatives = defaultdict( int )
   alternatives[ primaryKey.name ] = ldpPseudowireConnectorInfo( primaryKey,
                                                                 patchConfig )
   alternatives[
      alternateKey.name ] = ldpPseudowireConnectorInfo( alternateKey, patchConfig )

   conn.alternatives = dict( alternatives )

   return conn, activeKey

# ---------------------------------------------
# BGP pseudowire connector helper methods
# ---------------------------------------------

def getBgpRequestResponseStatus( connKey ):
   """Returns a status corresponding to the presence of BGP request and response"""

   # If remote connector status is present then don't check BGP request and
   # response, because they might not be there if for example the local interface
   # is down. This is different from LDP which creates LDP request/response in
   # this these cases.
   if ( PseudowireUserKey.fromConnectorKey( connKey )
        not in pwaPseudowireAttributeStatusColl.connectorStatus ):
      # Check BGP request
      reqKey = PwBgpReqKey( connKey.name, connKey.vpwsName )

      if reqKey not in pwaBgpRequest.bgpRequest:
         return CST.noLocalEnd

      # Check BGP response
      respKey = PwBgpRespKey( connKey.name, connKey.vpwsName )

      if respKey not in pwaBgpResponse.bgpResponse:
         # For the transient instance when BGP request is created
         # but BGP response doesn't exist, return noRemoteEnd
         return CST.noRemoteEnd

   return CST.up

def getBgpPwLocalConfig( connKey ):
   """Returns BgpPseudowireLocalConfig model for a BGP remote connector."""
   localState = PwaModel.BgpPseudowireLocalConfig()

   localLabel = pwaLabelBinding.localLabel.get(
      PseudowireUserKey.fromConnectorKey( connKey ) )
   if localLabel:
      localState.label = localLabel

   respKey = PwBgpRespKey( connKey.name, connKey.vpwsName )
   resp = pwaBgpResponse.bgpResponse.get( respKey )
   flowLabel = None
   if resp:
      localState.controlWord = resp.controlWord
      localState.mtu = resp.mtu
      if resp.flowLabel:
         flowLabel = FlowLabelMode.both
      else:
         flowLabel = FlowLabelMode.disabled

   return localState, flowLabel

def getBgpPwNeighborConfig( connKey ):
   """Returns BgpPseudowireNeighborConfig model for a BGP remote connector."""
   respKey = PwBgpRespKey( connKey.name, connKey.vpwsName )
   resp = pwaBgpResponse.bgpResponse.get( respKey )
   if not resp:
      return None

   nbrState = {}

   for peerInfo in resp.peerInfo.values():
      nbrConfig = PwaModel.BgpPseudowireNeighborConfig()
      nbrConfig.label = peerInfo.label
      tunnelId = TunnelId( peerInfo.tunnelId )
      nbrConfig.tunnelType = tunTypeToStr[ tunIdToType( peerInfo.tunnelId ) ]
      nbrConfig.tunnelIndex = tunnelId.tunnelIndex()
      nbrConfig.controlWord = peerInfo.controlWord
      nbrConfig.mtu = peerInfo.mtu
      nbrConfig.status = peerInfo.status
      nbrConfig.color = getResolvingVpwsPwTunnelColor( peerInfo.tunnelId )
      nbrState[ peerInfo.addr ] = nbrConfig

   return nbrState

def getBgpConnectorOptimized( connectorKey, connector, patchInfo ):
   '''Populates a BGP pseudowire connector specific details in conn
   (PwaModel.Connector) and returns the modified connector and status enum.'''
   bgpConnector = patchInfo.bgpConnector.get( connectorKey )
   if not bgpConnector:
      return connector
   connector.status = bgpConnector.status
   localConfig = None
   neighborConfigs = {}

   if bgpConnector.localConfig:
      localConfig = bgpConnector.localConfig
      localConfig = \
         PwaModel.BgpPseudowireLocalConfig(
            label=localConfig.label,
            controlWord=localConfig.controlWord,
            mtu=localConfig.mtu )

   if len( bgpConnector.neighborConfig ) > 0:
      for addr, info in bgpConnector.neighborConfig.items():
         neighborConfigs[ addr ] = \
            PwaModel.BgpPseudowireNeighborConfig(
               status=info.status,
               label=info.label,
               tunnelType=tunTypeToStr[ info.tunnelType ],
               tunnelIndex=info.tunnelIndex,
               controlWord=info.controlWord,
               mtu=info.mtu,
               color=info.color )

   connector.bgpPseudowireConnectorInfo = \
      PwaModel.BgpPseudowireConnector(
         vpwsName=bgpConnector.vpwsName,
         pwName=bgpConnector.pwName,
         localConfig=localConfig,
         neighborConfigs=neighborConfigs,
         vpwsType=bgpConnector.vpwsType,
         flowLabelUsed=bgpConnector.flowLabelUsed,
         color=bgpConnector.color,
         qosMapName=bgpConnector.qosMapName,
         qosMapType=( getQosMapType( bgpConnector.qosMapType )
                      if bgpConnector.qosMapType and
                      bgpConnector.qosMapType != "defaultMapType" else
                      None ) )

   return connector

def bgpPseudowireConnector( connKey, connStatus, patch, patchType, conn ):
   """Populates a BGP pseudowire connector specific details in conn
   (PwaModel.Connector) and returns the modified connector and status enum.
   """
   connInfo = PwaModel.BgpPseudowireConnector( vpwsName=connKey.name,
                                               pwName=connKey.vpwsName )
   if connStatus.localConnectorKey:
      if patchType == PwaModel.PatchTypes.fxc:
         connInfo.vpwsType = "fxc"
      elif any( loConnKey.isPort() for loConnKey in connStatus.localConnectorKey ):
         connInfo.vpwsType = "port"
      else:
         connInfo.vpwsType = "vlan"
      connInfo.localConfig, connInfo.flowLabelUsed = getBgpPwLocalConfig( connKey )
   connInfo.neighborConfigs = getBgpPwNeighborConfig( connKey )

   qosMap = pwStatus.qosMap.get( patch.pwIntfId )
   if qosMap and qosMap in pwQosMapHwStatus.hwStatus:
      connInfo.qosMapName = qosMap.mapName
      connInfo.qosMapType = getQosMapType( qosMap.mapType )

   respKey = PwBgpRespKey( connInfo.vpwsName, connInfo.pwName )
   resp = pwaBgpResponse.bgpResponse.get( respKey )
   if resp and resp.color is not None:
      connInfo.color = resp.color

   conn.bgpPseudowireConnectorInfo = connInfo
   loIntfIds = [ loConnKey.serviceIntfId()
                 for loConnKey in connStatus.localConnectorKey ]
   conn.status = getRemotePseudowireConnectorStatus( connKey, patch,
                                                     loIntfIds=loIntfIds )
   return conn

def getMplsStaticConnectorOptimized( connectorKey, connector, patchInfo ):
   '''Populates a MPLS static pseudowire connector specific details in conn
   (PwaModel.Connector) and returns the modified connector and status enum.'''
   mplsStaticConnector = patchInfo.mplsStaticConnector.get( connectorKey )
   if not mplsStaticConnector:
      return connector
   connector.status = mplsStaticConnector.status
   connector.mplsStaticPseudowireConnectorInfo = \
      PwaModel.MplsStaticPseudowireConnector(
         name=mplsStaticConnector.connName,
         transportIntf=mplsStaticConnector.transportIntf,
         localLabel=mplsStaticConnector.localLabel,
         neighborLabel=mplsStaticConnector.neighborLabel,
         tunnelIntfMode=mplsStaticConnector.tunnelIntfMode,
         controlWord=mplsStaticConnector.controlWord
      )

   return connector

def mplsStaticPseudowireConnector( connKey, connStatus, patch, conn ):
   """Populates a MPLS static pseudowire connector specific details in conn
   (PwaModel.Connector) and returns the modified connector and status enum.
   """

   tunnelIntfMode = EncapType.invalidEncap
   localLabel = MplsLabel.null
   neighborLabel = MplsLabel.null
   controlWord = False

   # Extract the attributes from the PwAttrStatus published by the
   # PwaStaticMplsConnectorSm
   pwAttrStatus = pwaPseudowireAttributeStatusColl.connectorStatus.get(
      PseudowireUserKey.fromConnectorKey( connKey ) )
   if pwAttrStatus:
      tunnelIntfMode = pwAttrStatus.encapType
      localLabel = pwAttrStatus.localLabel
      # For MPLS static pws either both encap and decap control word are enabled,
      # or none is enabled.
      controlWord = pwAttrStatus.decapControlWord
      # The neighbor/peer label is published in peerInfo which is a part of the
      # PwAttrStatus. As of now, for static MPLS connectors we only have 1 peerInfo
      if len( pwAttrStatus.peerInfo ) > 0:
         neighborLabel = pwAttrStatus.peerInfo.values()[ 0 ].peerLabel

   conn.mplsStaticPseudowireConnectorInfo = PwaModel.MplsStaticPseudowireConnector(
      name=connKey.name, transportIntf=connStatus.connector.transportIntf,
      localLabel=localLabel, neighborLabel=neighborLabel,
      tunnelIntfMode=tunnelIntfMode, controlWord=controlWord )
   loIntfIds = [ loConnKey.serviceIntfId()
                 for loConnKey in connStatus.localConnectorKey ]
   conn.status = getRemotePseudowireConnectorStatus( connKey, patch,
                                                     loIntfIds=loIntfIds )
   return conn

# ---------------------------------------------
# "show patch panel" and "show patch panel detail" helper
# methods
# ---------------------------------------------

def getConnectorName( patchConfig, connKey ):
   if not patchConfig:
      return ""

   # Find cliConnectors that have connKey
   cliConns = [ cc for cc in patchConfig.connector.values()
                if cc.connectorKey == connKey ]
   if cliConns:
      return cliConns[ 0 ].name

   return ""

def getConnectorOptimized( connectorKey, patchInfo ):
   '''Create and populate a PwaModel.Connector() corresponding to a connector
   identified by connectorKey'''
   if not connectorKey:
      return None

   if connectorKey.isLocal():
      connectorType = "tagged" if connectorKey.isLegacyTagged() else "port"
   else:
      connectorType = connectorKey.connectorType
   assert connectorType in PwaModel.supportedConnectorTypes
   connector = PwaModel.Connector( connType=connectorType )

   if connectorType == "tagged":
      return getTaggedConnectorOptimized( connectorKey, connector, patchInfo )
   if connectorType == "port":
      return getPortConnectorOptimized( connectorKey, connector, patchInfo )
   if connectorType == "ldp":
      return getLdpConnectorOptimized( connectorKey, connector, patchInfo )
   if connectorType == "bgp":
      return getBgpConnectorOptimized( connectorKey, connector, patchInfo )
   if connectorType == "mplsStatic":
      return getMplsStaticConnectorOptimized( connectorKey, connector, patchInfo )
   return None

def getConnector( connKey, patch, patchType, patchConfig ):
   '''Creates and populates a PwaModel.Connector() corresponding
   for a connector identified by connKey.'''
   if not connKey:
      return None

   connStatus = ( pwaPatchPanel.localConnStatus.get( connKey ) if connKey.isLocal()
                  else pwaPatchPanel.remoteConnStatus.get( connKey ) )
   if not connStatus:
      return None

   if connKey.isLocal():
      connType = "tagged" if connKey.isLegacyTagged() else "port"
   else:
      connType = connKey.connectorType
      # use patchConfig to fetch the more accurate alternate indicator
      for cliConnector in patchConfig.connector.values():
         if cliConnector.connectorKey == connKey:
            alternate = cliConnector.alternate
            break
      else:
         # somehow we cannot find the connector in patchConfig
         # fall back to previous implementation
         # this implementation is not robust against unexpected cases where an
         # alternate connector in one patch is marked as primary in another patch
         # in this case, because connStatus is patch-agnostic, connStatus.alternate
         # will be overwritten to False for both patches
         # this will cause a ghost connector being printed out in `show patch panel`
         alternate = connStatus.alternate
      # filter alternate pw connKey with will be handled by its dual connKey
      if alternate:
         return None

   assert connType in PwaModel.supportedConnectorTypes

   connector = PwaModel.Connector( connType=connType )

   if connType == "tagged":
      return taggedConnector( connKey, patch, patchType, connector )
   if connType == "port":
      return portConnector( connKey, patch, patchType, connector )
   if connType == "ldp":
      activeKey = None
      if len( patch.patchConnectorKey ) > 2:
         connector, activeKey = addLdpPseudowireAlternatives( connKey, patch,
                                                              connector,
                                                              patchConfig )
      else:
         activeKey = connKey
      activeConnStatus = pwaPatchPanel.remoteConnStatus.get( activeKey )
      return ldpPseudowireConnector( activeKey, activeConnStatus, patch, connector )
   if connType == "bgp":
      return bgpPseudowireConnector( connKey, connStatus, patch, patchType,
                                     connector )
   if connType == "mplsStatic":
      return mplsStaticPseudowireConnector( connKey, connStatus, patch, connector )
   return None

def getPatchStatus( patch, patchType, remoteConnStatus, localConnStatuses ):
   """Determine the patch status from the patch, its type and the statuses of the
   individual connectors."""

   if not patch:
      return CST.down

   if not patch.enabled:
      return CST.adminDown

   if patchType == PwaModel.PatchTypes.fxc:
      # an FXC patch without remote connector is not valid; therefore it must be down
      if not remoteConnStatus:
         return CST.down
   else:
      # any ordinary patch with less than two connectors is not valid;
      # therefore it must be down
      if ( ( remoteConnStatus and not localConnStatuses ) or
           ( not remoteConnStatus and len( localConnStatuses ) < 2 ) ):
         return CST.down

   if patch.status == CST.cliConflict:
      return CST.cliConflict

   # if there is a remote connector, look at it first to determine if the patch
   # is down or unprogrammed
   if remoteConnStatus:
      if CSH.isDown( remoteConnStatus ):
         return CST.down
      if CSH.isStandby( remoteConnStatus ):
         return CST.standby
      if CSH.isUnprogrammed( remoteConnStatus ):
         return CST.unprogrammed

   # the remote connector is either not present or up
   # now look at the local connectors to determine the final patch status

   haveUp = False
   haveDown = False
   haveUnprogrammed = False

   for status in localConnStatuses:
      if CSH.isDown( status ):
         haveDown = True
      elif CSH.isUnprogrammed( status ):
         haveUnprogrammed = True
      else:
         haveUp = True

   if patchType == PwaModel.PatchTypes.loLo:
      if haveDown:
         return CST.down
      elif haveUnprogrammed:
         return CST.unprogrammed
      else:
         return CST.up
   else: # local-remote patch (FXC or regular)
      if haveUp:
         return CST.partiallyUp if haveDown or haveUnprogrammed else CST.up
      elif haveUnprogrammed:
         return CST.unprogrammed
      else:
         return CST.down

def getPatchOptimized( patchName ):
   '''Returns a PwaModel.Patch object corresponding to patchName'''
   patchInfo = pwaPatchInfoColl.patchInfo.get( patchName )
   patchConfig = pwConfig.patch.get( patchName )

   if not patchInfo or not patchConfig:
      return None

   connectorKeys = ( set( patchInfo.taggedConnector.keys() ) |
                     set( patchInfo.portConnector.keys() ) |
                     set( patchInfo.mplsStaticConnector.keys() ) |
                     set( patchInfo.bgpConnector.keys() ) |
                     set( patchInfo.ldpConnector.keys() ) )
   connectors = {}
   patchStatus = patchInfo.patchState
   patchStatus = patchStatus[ 5 ].lower() + patchStatus[ 6 : ]
   patchType = patchInfo.patchType
   if patchInfo.fxc:
      patchType = PwaModel.PatchTypes.fxc
   elif patchType in ( "patchTypeLdp", "patchTypeBgp", "patchTypeMplsStatic" ):
      patchType = PwaModel.PatchTypes.loRmt
   else:
      patchType = PwaModel.PatchTypes.loLo
   patchLastStatusChange = patchInfo.patchStateLastChange
   for connectorKey in connectorKeys:
      connector = getConnectorOptimized( connectorKey, patchInfo )
      if connector:
         connName = getConnectorName( patchConfig, connectorKey )
         connectors[ connName ] = connector

   return PwaModel.Patch( status=patchStatus, connectors=connectors,
                          lastStatusChange=patchLastStatusChange,
                          _patchType=patchType )


def getPatch( patchName ):
   '''Returns a PwaModel.Patch object corresponding to patchName'''
   patch = pwaPatchPanel.patch.get( patchName )
   patchConfig = pwConfig.patch.get( patchName )

   if not patch or not patchConfig:
      return None

   connKeys = patch.patchConnectorKey
   connectors = {}
   localConnStatuses = []
   remoteConnStatus = None

   # Determine the patch type. Current supported ones are
   # lo-lo and lo-ldp. If any of the connectors is of type
   connTypes = { ck.connectorType for ck in connKeys }
   if patchConfig.fxc:
      patchType = PwaModel.PatchTypes.fxc
   else:
      patchType = ( PwaModel.PatchTypes.loRmt
                    if connTypes.intersection( { "bgp", "ldp", "mplsStatic" } )
                    else PwaModel.PatchTypes.loLo )

   for connKey in connKeys:
      connector = getConnector( connKey, patch, patchType, patchConfig )
      if connector:
         if connKey.isLocal():
            localConnStatuses.append( connector.status )
         else:
            remoteConnStatus = connector.status
         connName = getConnectorName( patchConfig, connKey )
         connectors[ connName ] = connector

   patchStatus = getPatchStatus( patch, patchType, remoteConnStatus,
                                 localConnStatuses )

   patchInfo = pwaPatchInfoColl.patchInfo.get( patchName )
   lastStatusChange = ( switchTimeToUtc( patchInfo.patchStateLastChange )
                        if patchInfo else None )

   return PwaModel.Patch( status=patchStatus, connectors=connectors,
                          lastStatusChange=lastStatusChange, _patchType=patchType )

# --------------------------------------------------------------------------------
# show patch panel summary
# --------------------------------------------------------------------------------
def patchPanelSummaryCmdHandler( mode, args ):
   patchSummary = PwaModel.PatchSummary( patchCount=0 )

   bgpVpwsCount = defaultdict( int )
   ldpCount = defaultdict( int )
   localCount = defaultdict( int )
   mplsStaticCount = defaultdict( int )

   # The CLI model uses the connector status types, but the TAC model returns
   # a patch status type. Map between the two types.
   stateMap = {
      PseudowirePatchState.patchAdminDown: CST.adminDown,
      PseudowirePatchState.patchCliConflict: CST.cliConflict,
      PseudowirePatchState.patchDown: CST.down,
      PseudowirePatchState.patchUnprogrammed: CST.unprogrammed,
      PseudowirePatchState.patchUp: CST.up,
      PseudowirePatchState.patchPartiallyUp: CST.partiallyUp,
      PseudowirePatchState.patchStandby: CST.standby,
   }

   for patch in pwaPatchInfoColl.patchInfo.values():
      patchSummary.patchCount += 1
      if patch.patchType == PseudowirePatchType.patchTypeBgp:
         bgpVpwsCount[ stateMap[ patch.patchState ] ] += 1
      elif patch.patchType == PseudowirePatchType.patchTypeLdp:
         ldpCount[ stateMap[ patch.patchState ] ] += 1
      elif patch.patchType == PseudowirePatchType.patchTypeMplsStatic:
         mplsStaticCount[ stateMap[ patch.patchState ] ] += 1
      else:
         # this includes unknown/incomplete patches, but the model doesn't define
         # a category for such patches hence they are counted as local ones
         localCount[ stateMap[ patch.patchState ] ] += 1

   patchSummary.bgpVpwsCount = dict( bgpVpwsCount )
   patchSummary.ldpCount = dict( ldpCount )
   patchSummary.localCount = dict( localCount )
   patchSummary.mplsStaticCount = dict( mplsStaticCount )
   return patchSummary

# --------------------------------------------------------------------------------
# show patch panel [ PATCH_NAME ] [ detail ]
# --------------------------------------------------------------------------------
def patchPanelDetailCmdHandler( mode, args ):
   patchName = args.get( 'PATCH_NAME' )

   detail = 'detail' in args

   def generatePatches( patchName ):
      if patchName:
         # patchName is specified, only return this if it is valid
         patch = ( getPatchOptimized( patchName )
                   if togglePseudowireShowPatchPanelOptimizedEnabled()
                   else getPatch( patchName ) )
         if patch:
            yield patchName, patch
      else:
         # patchName wasn't specified. Return all patches in
         # patchPanel, sort by patch name here since the render
         # function cannot sort the GeneratorDict output.

         for patchKey in sorted( pwaPatchPanel.patch ):
            patch = ( getPatchOptimized( patchKey )
                      if togglePseudowireShowPatchPanelOptimizedEnabled()
                      else getPatch( patchKey ) )
            if patch:
               yield patchKey, patch

   return PwaModel.Patches( patches=generatePatches( patchName ),
                              _summary=not detail )

# -------------------------------------------------------------------------------
# Helpers for the 'show patch panel forwarding' command
# -------------------------------------------------------------------------------

def getLocalConnectorPatchInfo( connKey ):
   connStatus = pwaPatchPanel.localConnStatus.get( connKey )
   if connStatus:
      return ( connStatus.patchName, connStatus.peerConnectorKey,
               connStatus.peerAlternateConnectorKey )
   else:
      return None, None, None

def getRemoteConnectorPatchInfo( connKey ):
   connStatus = pwaPatchPanel.remoteConnStatus.get( connKey )
   if connStatus:
      return connStatus.patchName, connStatus.localConnectorKey
   else:
      return None, []

def getLocalConnectorFwdingInfo( connKey, decapCw=None ):
   localIntfId = connKey.localIntfId()
   if connKey.isLegacyTagged():
      taggedConnFwdingInfo = PwaModel.TaggedConnectorFwdingInfo(
         interface=localIntfId,
         dot1qVlan=connKey.legacyDot1qTag(),
         decapControlWord=decapCw )
      connFwdingInfo = PwaModel.ForwardingConnectorInfo(
                                          _infoType="tagged",
                                          taggedConnFwdingInfo=taggedConnFwdingInfo )
   else:
      portConnFwdingInfo = PwaModel.PortConnectorFwdingInfo(
         interface=localIntfId, decapControlWord=decapCw )
      connFwdingInfo = PwaModel.ForwardingConnectorInfo(
                                          _infoType="port",
                                          portConnFwdingInfo=portConnFwdingInfo )
   return connFwdingInfo

def getLocalEntry( connKey, currPwaLabelBinding ):
   patchName, peerConnKey, peerAltConnKey = getLocalConnectorPatchInfo( connKey )
   if not patchName or not peerConnKey or not peerConnKey.isValid():
      return None

   # 'show patch panel forwarding' must only show patches that are up or at least
   # partially up or unprogrammed. It should not show patches that are down.
   # See BUG633015
   patchInfo = pwaPatchInfoColl.patchInfo.get( patchName )
   if ( not patchInfo or
        patchInfo.patchState not in ( PseudowirePatchState.patchUp,
                                      PseudowirePatchState.patchUnprogrammed,
                                      PseudowirePatchState.patchPartiallyUp ) ):
      return None

   connectorFwdingInfo = getLocalConnectorFwdingInfo( connKey )
   entry = ( PwaModel.TaggedForwardingInfo( inConnInfo=connectorFwdingInfo )
             if connKey.isLegacyTagged()
             else PwaModel.PortForwardingInfo( inConnInfo=connectorFwdingInfo ) )

   entry.patch = patchName
   if peerConnKey.isLocal():
      entry.outConnInfo = getLocalConnectorFwdingInfo( peerConnKey )
   else:
      pwKey = PseudowireUserKey.fromConnectorKey( peerConnKey )
      pwAttr = pwaPseudowireAttributeStatusColl.connectorStatus.get( pwKey )
      if not pwAttr and peerAltConnKey and peerAltConnKey.isValid():
         # see if the alternate connector is active (BUG826357)
         pwKey = PseudowireUserKey.fromConnectorKey( peerAltConnKey )
         pwAttr = pwaPseudowireAttributeStatusColl.connectorStatus.get( pwKey )
      if not pwAttr:
         return None
      currPwaLabelBinding[ pwAttr.localLabel ] = pwKey
      entry.source = labelSources.get( peerConnKey.connectorType )
      entry.pseudowireType = pwAttr.remotePwType
      # If MPLS static connector, change pwType from pwTypeNone to None for CLI
      # to render it as ""
      if peerConnKey.connectorType == PwConnectorType.mplsStatic:
         entry.pseudowireType = None
      if pwAttr.flowLabelType & flEnumValTransmit:
         entry.flowLabel = 'Out'
      if pwAttr.requestedDot1qTag:
         entry.dot1qVlan = pwAttr.requestedDot1qTag

      labelConnFwdingInfo = PwaModel.LabelConnectorFwdingInfo(
         _infoType=peerConnKey.connectorType )
      for tep, peerInfo in pwAttr.peerInfo.items():
         tunnels = []
         for tunnelId in sorted( peerInfo.tunnelId ):
            tunnelType = tunIdToType( tunnelId )
            # For VPWS Pseudowires, we need to use the tunnel tables to retrieve the
            # actual color of the resolving tunnel since the color in peerInfo
            # only contains the color used to resolve the neighbor address
            if peerConnKey.connectorType == PwConnectorType.bgp:
               color = getResolvingVpwsPwTunnelColor( tunnelId )
            else:
               color = peerInfo.color.value if peerInfo.color.isSet else None
            tunnelIndex = TunnelId( tunnelId ).tunnelIndex()
            tunnelType = tunTypeToStr[ tunnelType ]
            tunnels.append(
               PwaModel.TunnelInfo( tunnelType=tunnelType,
                                    tunnelIndex=tunnelIndex,
                                    color=color ) )
         labelConnFwdingPeerInfo = PwaModel.LabelConnectorFwdingPeerInfo(
            peerLabel=peerInfo.peerLabel,
            tunnels=tunnels,
            encapControlWord=peerInfo.encapControlWord )
         labelConnFwdingInfo.peers[ tep ] = labelConnFwdingPeerInfo

      entry.outConnInfo = PwaModel.ForwardingConnectorInfo(
         _infoType=peerConnKey.connectorType,
         labelConnFwdingInfo=labelConnFwdingInfo )

   entry.status = getLocalConnectorProgrammedStatus( connKey.serviceIntfId() )

   return entry

labelSources = { PwConnectorType.bgp: "EVPN",
                 PwConnectorType.ldp: "LDP",
                 PwConnectorType.mplsStatic: "MPLS static" }

def getLabelEntry( routeKey, currPwaLabelBinding ):
   # Gets the label entry related to the connectorkey from the status
   label = routeKey.topLabel
   labelEntry = PwaModel.LabelForwardingInfo( label=label )

   pwRoute = pwLfib.lfibRoute.get( routeKey )
   if not pwRoute:
      return None

   pwViaSet = pwLfib.viaSet.get( pwRoute.viaSetKey )
   if not pwViaSet:
      return None

   pwVia = pwLfib.pwVia.get( pwViaSet.viaKey[ 0 ] )
   if not pwVia:
      return None

   pwKey = currPwaLabelBinding.get( label )
   if not pwKey:
      return None
   if not pwKey.isPatch():
      return None

   connKey = pwKey.connectorKey()
   labelEntry.patch, localConnKeys = getRemoteConnectorPatchInfo( connKey )
   if not labelEntry.patch or not localConnKeys:
      return None

   serviceIntfIds = []
   for localConnKey in sorted( localConnKeys ):
      serviceIntfId = localConnKey.serviceIntfId()
      serviceIntfIds.append( serviceIntfId )
      outConnInfo = getLocalConnectorFwdingInfo( localConnKey,
                                                 decapCw=pwVia.controlWordPresent )
      if not labelEntry.outConnInfo:
         labelEntry.outConnInfo = ( outConnInfo if len( localConnKeys ) == 1
                                    else PwaModel.ForwardingConnectorInfo() )
      labelEntry.outConnectors.append( outConnInfo )

   labelEntry.status = getRemotePseudowireProgrammedStatus( serviceIntfIds )

   labelEntry.source = labelSources.get( connKey.connectorType )
   if pwVia.flowLabelPresent:
      labelEntry.flowLabel = 'In'
   labelEntry.pseudowireType = pwVia.pwType
   # If MPLS static connector, change pwType from pwTypeNone to None for CLI
   # to render it as ""
   if connKey.connectorType == PwConnectorType.mplsStatic:
      del labelEntry.pseudowireType

   return labelEntry

def getTaggedForwardingTable( currPwaLabelBinding ):
   # Creates a table with all the entries for tagged forwarding that are either
   # up, partially up or unprogrammed (getLocalEntry checks patchInfo.patchState)
   taggedEntries = {}

   for connKey, connStatus in pwaLocalConnectorStatusColl.connectorStatus.items():
      if not connKey.isLegacyTagged():
         continue
      if connStatus.status not in ( 'up', 'errdisabledUp' ):
         continue
      taggedEntry = getLocalEntry( connKey, currPwaLabelBinding )
      if taggedEntry:
         taggedEntries[ taggedEntry.inConnInfo.toStr() ] = taggedEntry

   return PwaModel.TaggedForwardingInfoTable( taggedForwardingInfos=taggedEntries )

def getPortForwardingTable( currPwaLabelBinding ):
   # Creates a table with all the entries for port forwarding that are either
   # up, partially up or unprogrammed (getLocalEntry checks patchInfo.patchState)
   portEntries = {}

   for connKey, connStatus in pwaLocalConnectorStatusColl.connectorStatus.items():
      if connKey.isLegacyTagged():
         continue
      if connStatus.status not in ( 'up', 'errdisabledUp' ):
         continue
      portEntry = getLocalEntry( connKey, currPwaLabelBinding )
      if portEntry:
         portEntries[ portEntry.inConnInfo.toStr() ] = portEntry

   return PwaModel.PortForwardingInfoTable( portForwardingInfos=portEntries )

def getLabelForwardingTable( currPwaLabelBinding ):
   # Creates a table with all the entries for label forwarding that are up
   # (this is guaranteed because lfibRoute only contains such entries)
   labelEntries = {}

   for routeKey in pwLfib.lfibRoute:
      labelEntry = getLabelEntry( routeKey, currPwaLabelBinding )
      if labelEntry:
         labelEntries[ labelEntry.label ] = labelEntry

   return PwaModel.LabelForwardingInfoTable( labelForwardingInfos=labelEntries )

def patchPanelForwardingCmdHandler( mode, args ):
   # Dict to store map of local label to pseudowireKey. This is created in
   # getLocalEntry and used in getLabelEntry for 'show patch panel forwarding'
   # command.
   currPwaLabelBinding = {}
   taggedTable = getTaggedForwardingTable( currPwaLabelBinding )
   portTable = getPortForwardingTable( currPwaLabelBinding )
   labelTable = getLabelForwardingTable( currPwaLabelBinding )

   return PwaModel.PatchForwardingTable( taggedForwardingInfoTable=taggedTable,
                                          portForwardingInfoTable=portTable,
                                          labelForwardingInfoTable=labelTable )

# --------------------------------------------------------------------------------
# clear mpls ldp pseudowire pwName helper function
# --------------------------------------------------------------------------------
def clearMplsLdpPseudowireCmdHandler( mode, args ):
   pwName = args.get( 'pwName' )
   # check if cleared ldp pseudwoire exist
   existPw = [ connKey.name for connKey in pwConfig.connector.keys() ]
   if pwName not in existPw:
      mode.addError( "LDP connector does not exist" )
      return
   currId = pwaLdpPwResetRequest.request.requestId
   newRequest = LdpPwResetRequestEntry( getLdpConnectorKey( pwName ), currId + 1 )
   pwaLdpPwResetRequest.request = newRequest

# --------------------------------------------------------------------------------
# VPWS Pseudowire tunnel color helper function
# --------------------------------------------------------------------------------
def getResolvingVpwsPwTunnelColor( tunnelId ):
   tunnelType = tunIdToType( tunnelId )
   tunnelTable = pwVpwsTunnelTables.get( tunnelType )
   tableEntry = tunnelTable.entry.get( tunnelId ) if tunnelTable else None
   try:
      # TristateU32 color
      color = \
         tableEntry.color.value if tableEntry and tableEntry.color.isSet else None
   except AttributeError:
      # U32 or U32 optional color
      color = tableEntry.color
      # check the color status flag for IS-IS Flex Algo tunnels
      if ( tunnelType == TunnelType.isisFlexAlgoTunnel and
           not tableEntry.isColored ):
         color = None
   return color

# -------------------------------------------------------------------------------
# Mount necessary mountpoints for show commands
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global pwStatus
   global pwLfib
   global pwaPatchPanel
   global pwaLocalConnectorStatusColl, pwaPseudowireAttributeStatusColl
   global pwaPatchInfoColl
   global pwHwDir, pwQosMapHwStatus
   global pwaLdpRequest
   global pwaLdpResponse
   global pwaBgpRequest
   global pwaBgpResponse
   global pwaLabelBinding
   global pwaEbraResponse
   global pwaLdpPwResetRequest

   # Pseudowire agent's published status to ale/hw
   pwStatus = LazyMount.mount( entityManager, 'pseudowire/status',
                               'Pseudowire::Status', 'r' )

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   pwLfib = shmemEm.doMount( 'mpls/protoLfibInputDir/pseudowire', 'Mpls::LfibStatus',
                             Smash.mountInfo( 'reader' ) )

   # rsvpLer tunnel table
   pwRsvpLerTunnelTable = shmemEm.doMount( 'tunnel/table/rsvpLer',
                                           'Tunnel::TunnelTable::RsvpLerTunnelTable',
                                           Smash.mountInfo( 'reader' ) )
   pwVpwsTunnelTables[ pwRsvpLerTunnelTable.tableType ] = pwRsvpLerTunnelTable

   # srTePolicy tunnel table
   pwSrTePolicyTunnelTable = shmemEm.doMount(
      'tunnel/table/srTePolicy',
      'Tunnel::TunnelTable::SrTePolicyTunnelTable',
      Smash.mountInfo( 'reader' ) )
   pwVpwsTunnelTables[ pwSrTePolicyTunnelTable.tableType ] = pwSrTePolicyTunnelTable

   # IsisFlexAlgo tunnel table
   pwIsisFlexAlgoTunnelTable = shmemEm.doMount(
      'tunnel/table/isisFlexAlgoTunnel',
      'Tunnel::TunnelTable::IsisFlexAlgoTunnelTable',
      Smash.mountInfo( 'reader' ) )
   pwVpwsTunnelTables[ pwIsisFlexAlgoTunnelTable.tableType ] = \
      pwIsisFlexAlgoTunnelTable

   # Pseudowire's internal entities for relaying state between state machines
   pwaPatchPanel = LazyMount.mount( entityManager, 'pseudowire/agent/patch',
                                     'Pseudowire::PatchPanel', 'r' )

   pwaLocalConnectorStatusColl = LazyMount.mount( entityManager,
                                       'pseudowire/agent/localConnectorStatusColl',
                                       'PseudowireAgent::LocalConnectorStatusColl',
                                       'r' )
   pwaPseudowireAttributeStatusColl = LazyMount.mount(
      entityManager,
      'pseudowire/agent/pseudowireAttributeStatusColl',
      'Pseudowire::PseudowireAttributeStatusColl',
      'r' )

   pwaPatchInfoColl = LazyMount.mount( entityManager,
                                      'pseudowire/agent/pseudowirePatchInfoColl',
                                      'Pseudowire::PseudowirePatchInfoColl', 'r' )

   pwaLabelBinding = LazyMount.mount( entityManager,
                                      'pseudowire/agent/labelBinding',
                                      'Pseudowire::LabelBindingColl',
                                      'r' )

   # Pseudowire agent's requests to and responses from Ldp agent to setup
   # targeted sessions
   pwaLdpRequest = LazyMount.mount( entityManager,
                           "pseudowire/ldp/config",
                           "Pseudowire::PseudowireLdpConfig",
                           "r" )
   pwaLdpResponse = LazyMount.mount( entityManager,
                           "pseudowire/ldp/status",
                           "Pseudowire::PseudowireLdpStatusPtr",
                           "r" )

   # Pseudowire agent's requests to and responses from BGP agent
   pwaBgpRequest = LazyMount.mount( entityManager,
                           "pseudowire/bgp/config",
                           "Pseudowire::PseudowireBgpConfig",
                           "r" )
   pwaBgpResponse = LazyMount.mount( entityManager,
                           Cell.path( "pseudowire/bgp/status" ),
                           "Pseudowire::PseudowireBgpStatus",
                           "r" )

   # Qos Map Status from SandTunnel
   pwQosMapHwStatus = LazyMount.mount( entityManager,
                           "pseudowire/hardware/status/pwqosmaphwstatus",
                           "Pseudowire::Hardware::PwQosMapHwStatus",
                           "r" )

   # Internal vlanId allocation from Ebra for local connectors
   pwaEbraResponse = LazyMount.mount( entityManager,
                            "bridging/external/status",
                            "EbraExt::ResponseReservedVlanId",
                            "r" )

   # Lists unprogrammed encap and decap adjacencies
   pwHwDir = LazyMount.mount( entityManager,
                              "interface/encap/hw/serviceIntf/status",
                              "Ebra::ServiceIntfHwStatusColl", "r" )

   # Ldp pseudowire reset request
   pwaLdpPwResetRequest = LazyMount.mount( entityManager,
                                           "pseudowire/agent/ldpPwResetRequest",
                                           "Pseudowire::LdpPwResetRequest",
                                           "w" )
