# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import BasicCli
import Cell
import CliCommand
import CliDynamicSymbol
import CliGlobal
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import Tac
import Tracing
import QosLib
from CliPlugin import ( AclCli, QosCliCopp, QosCliServicePolicy )
from CliPlugin.QosCliCopp import ( matcherClass, matcherClassName )
from CliPlugin.QosCliModel import ( InterfaceIdModel, PolicyMapAllModel )
from CliPlugin.QosCliCommon import ( guardIpAcls, guardIp6Acls )
from QosCliLib import PMapModelContainer
from QosLib import ( classMapCpStaticType, classMapCpType, mapTypeToEnum,
                    coppServicePolicyIs, coppStaticClassFromHwStatus, isLagPort,
                    CliError, coppMapType )
from QosTypes import ( tacDirection, tacClassMapCpType, tacPMapNm, tacCMapNm,
                      tacActionRateType, tacRateUnit, tacActionType, tacBurstUnit )

DYNAMIC_CMAP_EDIT_WARN = 'Editing dynamically created ' \
                           '(by agents and not CLI) class map'

__defaultTraceHandle__ = Tracing.Handle( 'QosCliCoppHandler' )
t0 = Tracing.trace0
t8 = Tracing.trace8

# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
gv = CliGlobal.CliGlobal(
   dict(
      qosConfig=None,
      qosInputConfig=None,
      qosStatus=None,
      qosAclHwStatus=None,
      cliQosAclConfig=None,
      hwEpochStatus=None,
      qosHwStatus=None,
      qosAclSliceHwStatus=None,
      qosSliceHwStatus=None,
      intfConfigEthPhySliceDir=None,
      cliCounterConfig=None,
      qosInputProfileConfig=None,
   )
)

qosCliServicePolicyDynamicSubmodes = CliDynamicSymbol.CliDynamicPlugin(
        "QosCliServicePolicyMode" )

# -----------------------------------------------------------------------------------
# Guards begin
# -----------------------------------------------------------------------------------
def coppActionPolicerSupportedGuard( mode, token ):
   if gv.qosHwStatus.coppActionPolicerSupported and \
      gv.qosHwStatus.hwInitialized:
      return None
   return CliParser.guardNotThisPlatform

def guardCoppPoliceRateInBytes( mode, token ):
   if gv.qosHwStatus.coppActionPolicerSupported and \
      gv.qosHwStatus.coppKbpsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCoppPoliceRateInPps( mode, token ):
   if gv.qosHwStatus.coppActionPolicerSupported and \
      gv.qosHwStatus.coppPpsSupported:
      return None
   return CliParser.guardNotThisPlatform

def rateUnitAdapter( mode, args, argsList ):
   rateUnitsToTypes = {
      'pps': tacRateUnit.rateUnitPps,
      'kbps': tacRateUnit.rateUnitKbps,
   }
   args[ 'rateUnit' ] = rateUnitsToTypes[ args.get( 'kbps', 'pps' ) ]

def guardPMapCoppRateUnitPps( mode, token ):
   if ( gv.qosHwStatus.coppSupported or gv.qosHwStatus.intfCoppSupported ) \
         and gv.qosHwStatus.coppPpsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPMapCoppRateUnitKbps( mode, token ):
   if ( gv.qosHwStatus.coppSupported or gv.qosHwStatus.intfCoppSupported ) \
         and gv.qosHwStatus.coppKbpsSupported:
      return None
   return CliParser.guardNotThisPlatform

def coppActionRateShapeAndBw( mode, context ):
   return QosLib.coppRateShapeAndBandwidth( gv.qosHwStatus )

def coppActionShaperSupportedGuard( mode, token ):
   if gv.qosHwStatus.coppActionShaperSupported and \
      gv.qosHwStatus.hwInitialized:
      return None
   return CliParser.guardNotThisPlatform
# -----------------------------------------------------------------------------------
# Guards end
# -----------------------------------------------------------------------------------

def getClassNameRuleCopp( mode ):
   return QosCliServicePolicy.getClassNameRule( mode, coppMapType )

# -----------------------------------------------------------------------------------
# Expressions begin
# -----------------------------------------------------------------------------------
class RateExpression( CliCommand.CliExpression ):
   expression = 'bps | kbps | mbps | pps'
   data = {
         'bps': CliCommand.guardedKeyword( 'bps',
            "rate in bps (default unit)", guard=guardCoppPoliceRateInBytes ),
         'kbps': CliCommand.guardedKeyword( 'kbps',
            "The rate expressed in units of kilobits per second",
            guard=guardCoppPoliceRateInBytes ),
         'mbps': CliCommand.guardedKeyword( 'mbps',
            "rate in Mbps", guard=guardCoppPoliceRateInBytes ),
         'pps': CliCommand.guardedKeyword( 'pps',
            "rate in pps", guard=guardCoppPoliceRateInPps ),
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      rateUnitsToTypes = {
         'bps': tacRateUnit.rateUnitbps,
         'kbps': tacRateUnit.rateUnitKbps,
         'mbps': tacRateUnit.rateUnitMbps,
         'pps': tacRateUnit.rateUnitPps,
      }

      for rate in ( 'bps', 'kbps', 'mbps', 'pps' ):
         if rate in args:
            args[ 'RATE_CIR_UNIT' ] = rateUnitsToTypes[ args[ rate ] ]
            break

class BurstExpression( CliCommand.CliExpression ):
   expression = 'bytes | kbytes | mbytes | packets'
   data = {
         'bytes': CliCommand.guardedKeyword( 'bytes',
            "burst size in bytes (default unit)", guard=guardCoppPoliceRateInBytes ),
         'kbytes': CliCommand.guardedKeyword( 'kbytes',
            "burst size in kbytes", guard=guardCoppPoliceRateInBytes ),
         'mbytes': CliCommand.guardedKeyword( 'mbytes',
            "burst size in mbytes", guard=guardCoppPoliceRateInBytes ),
         'packets': CliCommand.guardedKeyword( 'packets',
            "burst size in packets", guard=guardCoppPoliceRateInPps ),
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      burstUnitsToTypes = {
         'bytes': tacBurstUnit.burstUnitBytes,
         'kbytes': tacBurstUnit.burstUnitKBytes,
         'mbytes': tacBurstUnit.burstUnitMBytes,
         'packets': tacBurstUnit.burstUnitPackets
      }

      for rate in ( 'bytes', 'kbytes', 'mbytes', 'packets' ):
         if rate in args:
            args[ 'BURST_BC_UNIT' ] = burstUnitsToTypes[ args[ rate ] ]
            break
# -----------------------------------------------------------------------------------
# Expressions end
# -----------------------------------------------------------------------------------

# --------------------------------------------------------------------------------
# abort commands ( class-map )
# abort commands ( policy-map )
# abort commands ( policy-map-class )
# abort commands ( PolicyMapClassModeQos )
# abort commands ( QosProfileMode )
# abort commands ( ClassMapModeQos )
# abort commands ( PolicyMapModeQos )
# --------------------------------------------------------------------------------

class ClassMapModeCopp( qosCliServicePolicyDynamicSubmodes.ClassMapMode ):
   name = "Class Map Copp Configuration"


ClassMapModeCopp.addShowCommandClass( QosCliCopp.CmapModeShowCmd )

# ------------------------------------------------------------------------
# Match and set value binding rules ( class-map )
# ------------------------------------------------------------------------
# --------------------------------------------------------------------------------
# [ no | default ] match ip access-group GROUP
# --------------------------------------------------------------------------------
class MatchIpAccessGroupCoppCmd( CliCommand.CliCommandClass ):
   syntax = 'match ip access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': 'Match the access rule specified',
      'ip': CliCommand.guardedKeyword( 'ip', "Specify Ip Access-groups",
         guard=guardIpAcls ),
      'access-group': 'Match with given access group',
      'GROUP': AclCli.standardIpAclNameMatcher,
   }

   handler = 'QosCliCoppHandler.doMatchIpAccessGroupCoppCmd'
   noOrDefaultHandler = handler


ClassMapModeCopp.addCommandClass( MatchIpAccessGroupCoppCmd )

# --------------------------------------------------------------------------------
# [ no | default ] match ipv6 access-group GROUP
# --------------------------------------------------------------------------------
class MatchIpv6AccessGroupCoppCmd( CliCommand.CliCommandClass ):
   syntax = 'match ipv6 access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': 'Match the access rule specified',
      'ipv6': CliCommand.guardedKeyword( 'ipv6', "Specify Ipv6 Access-groups",
         guard=guardIp6Acls ),
      'access-group': 'Match with given access group',
      'GROUP': AclCli.standardIp6AclNameMatcher,
   }

   handler = 'QosCliCoppHandler.doMatchIpv6AccessGroupCoppCmd'
   noOrDefaultHandler = handler


ClassMapModeCopp.addCommandClass( MatchIpv6AccessGroupCoppCmd )
ClassMapModeCopp.addCommandClass( QosCliCopp.AbortCmd )


# --------------------------------------------------------------------------------
# show policy-map ( copp | ( interface control-plane ) ) copp-system-policy
# --------------------------------------------------------------------------------
def showPMapInterfaceControlPlane( mode, args ):
   t0( "Handler: show policy-map ( copp \
         | ( interface control-plane ) ) copp-system-policy" )
   mapType = args.get( 'control-plane', 'copp' )
   pmapName = args[ 'copp-system-policy' ]
   policyMapAllModel = PolicyMapAllModel()
   mapType = 'control-plane' if mapType == 'copp' else mapType
   emapType = mapTypeToEnum( mapType )
   emapTypeInSliceHwStatus = False
   for sliceHwStatus in gv.qosSliceHwStatus.values():
      if emapType in sliceHwStatus.pmapType:
         emapTypeInSliceHwStatus = True
         break
   if not emapTypeInSliceHwStatus:
      return policyMapAllModel
   direction = tacDirection.input
   policyMapsContainer = PMapModelContainer( gv.qosConfig, gv.cliQosAclConfig,
                                gv.qosStatus, gv.qosHwStatus, gv.qosSliceHwStatus,
                                gv.qosAclHwStatus, gv.qosAclSliceHwStatus, emapType,
                                direction, gv.hwEpochStatus, None,
                                policyMapAllModel )
   # show policy-map copp <pmap_name> will display counters also
   policyMapsContainer.counterDetail = True
   policyMapsContainer.populatePolicyMapInterface( pmapName )
   return policyMapAllModel

def getIntfsForServicePolicy( mapType, pmapName, direction ):
   servicePolicyKey = None
   for servicePolicy in gv.qosStatus.servicePolicyStatus:
      if servicePolicy.pmapName == pmapName and \
             servicePolicy.type == mapType and \
             servicePolicy.direction == direction:
         servicePolicyKey = servicePolicy
   if servicePolicyKey is None:
      return []
   intfsConfigured = []
   for intf, enabled in gv.qosStatus.servicePolicyStatus[
                servicePolicyKey ].intfIds.items():
      if enabled:
         intfsConfigured.append( intf.intfId )
   intfsConfigured = Arnet.sortIntf( intfsConfigured )
   return intfsConfigured

# --------------------------------------------------------------------------------
# show policy-map ( copp | ( interface control-plane ) ) PMAP
# --------------------------------------------------------------------------------
def showPMapInterfaceIntfCopp( mode, args ):
   pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' )
   mapType = 'control-plane'
   emapType = mapTypeToEnum( mapType )
   intfIdModel = InterfaceIdModel()

   for sliceHwStatus in gv.qosSliceHwStatus.values():
      if emapType not in sliceHwStatus.pmapType:
         return intfIdModel

   intfs = getIntfsForServicePolicy( emapType, pmapName, tacDirection.input )
   intfIdModel.extend( intfs )
   return intfIdModel

# ------------------------------------------------------------------------
# Register top-level CLI commands ( class-map )
# ------------------------------------------------------------------------
# Goto 'class-map' config mode. Create a class map context.
# The context holds all current editing values. The context
# is associated with the session of the mode.

# --------------------------------------------------------------------------------
# class-map type ( copp | control-plane ) match-any CMAP
# --------------------------------------------------------------------------------
def gotoCoppClassMapMode( mode, args ):
   mapType = 'control-plane'
   cmapName = args.get( 'CMAP' )
   matchType = 'match-any'
   t0( f'gotoClassMapMode { cmapName }' )
   emapType = mapTypeToEnum( mapType )
   if emapType == coppMapType:
      cpType = classMapCpType( gv.cliQosAclConfig, cmapName )
      if cpType == tacClassMapCpType.cmapCpStatic:
         mode.addError( CliError[ 'cannotConfigureCMap' ] )
         return
   context = QosCliServicePolicy.ClassMapContext( mode, emapType, cmapName,
                                                  matchType )

   if cmapName in gv.cliQosAclConfig.cmapType[ emapType ].cmap:
      entry = gv.cliQosAclConfig.cmapType[ emapType ].cmap[ cmapName ]
      context.copyEditEntry( entry )
      if entry.dynamic:
         mode.addWarning( DYNAMIC_CMAP_EDIT_WARN )
   else:
      context.newEditEntry()

   mode.cmapContext = context
   childMode = mode.childMode( ClassMapModeCopp, context=context )
   mode.session_.gotoChildMode( childMode )

# --------------------------------------------------------------------------------
# ( no | default ) class-map type ( copp | control-plane ) [ match-any ] CMAP
# --------------------------------------------------------------------------------
def deleteCoppClassMap( mode, args ):
   mapType = 'control-plane'
   cmapName = args.get( 'CMAP' )
   cpType = classMapCpType( gv.cliQosAclConfig, cmapName )
   if cpType == tacClassMapCpType.cmapCpStatic:
      mode.addError( CliError[ 'cannotDeleteCMap' ] )
      return
   emapType = mapTypeToEnum( mapType )
   if cmapName in gv.cliQosAclConfig.cmapType[ emapType ].cmap:
      QosCliServicePolicy.deleteClassMap( mode, cmapName, mapType )

# Goto 'policy-map' config mode. Create a policy map context.
# The context holds all current editing values. The context
# is associated with the session of the mode.
# --------------------------------------------------------------------------------
# [ no | default ] policy-map type ( copp | control-plane )
#       ( copp-system-policy | PMAP )
# --------------------------------------------------------------------------------
def gotoCoppPolicyMapMode( mode, args ):
   pmapName = args.get( 'PMAP', 'copp-system-policy' )
   t0( 'gotoCoppPolicyMapMode of policy', pmapName )
   mapType = 'control-plane'
   if pmapName != tacPMapNm.coppName and \
         not mode.session_.startupConfig():
       # not qosHwStatus.intfCoppSupported and \
      # if pmapName != tacPMapNm.coppName and \
      #    not mode.session_.startupConfig():
      mode.addError( "Not Supported on this hardware platform" )
      return
   cmapName = tacCMapNm.coppDefault
   context = QosCliServicePolicy.gotoPolicyMapModeCommon( mode, pmapName,
                                             mapType, cmapName )
   if context:
      mode.pmapContext = context
      childMode = mode.childMode( PolicyMapModeCopp, context=context )
      mode.session_.gotoChildMode( childMode )

def deleteCoppPolicyMap( mode, args ):
   pmapName = args.get( 'PMAP', 'copp-system-policy' )
   mapType = 'control-plane'
   QosCliServicePolicy.deletePolicyMap( mode, pmapName, mapType )

# -------------------------------------------------------------------------------
# policy-map class mode
# -------------------------------------------------------------------------------

class PolicyMapClassModeCopp( QosCliServicePolicy.PolicyMapClassMode ):
   name = "Policy Map Copp Class Configuration"

   def obtainShapeOrBandwidthValue( self, no, rateUnit, value ):
      cpType = classMapCpType( gv.cliQosAclConfig, self.cmapName )
      if cpType == tacClassMapCpType.cmapCpStatic:
         if no == 'default':
            value = tacActionRateType.invalid
            rateUnit = tacRateUnit.rateUnitInvalid
         elif no:
            value = tacActionRateType.noValue
            rateUnit = tacRateUnit.rateUnitInvalid
      else:
         if no:
            value = tacActionRateType.noValue
            rateUnit = tacRateUnit.rateUnitInvalid
      return ( value, rateUnit )

   def configureShape( self, no, rateUnit, value ):
      context = self.pmapClassContext
      classAction = context.currentEntry()

      if tacActionType.actionSetShape not in classAction.policyAction:
         context.newEditEntry( self.cmapName, bypassCheck=True )
         classAction = context.currentEntry()

      def currentShape():
         rate = classAction.policyAction[ tacActionType.actionSetShape ].rate
         return rate.val

      def defaultShape():
         cpType = classMapCpType( gv.cliQosAclConfig, self.cmapName )
         if cpType == tacClassMapCpType.cmapCpStatic:
            cpStaticType = classMapCpStaticType( gv.cliQosAclConfig, self.cmapName )
            hwCoppStaticClass = coppStaticClassFromHwStatus( gv.qosSliceHwStatus )
            return hwCoppStaticClass[ cpStaticType ].defaultMax
         return None

      ( value, rateUnit ) = self.obtainShapeOrBandwidthValue( no, rateUnit, value )
      rate = classAction.policyAction[ tacActionType.actionSetShape ].rate
      if value is not None:
         rate.val = value
      if rateUnit is not None:
         rate.rateUnit = rateUnit

      # For a static class, if parameters have been restored to default,
      # then we should write invalid values in config. But if it is startup
      # we don't have default min and max and we skip this check.
      if not self.session_.startupConfig():
         if not gv.qosHwStatus.intfCoppSupported:
            value = defaultShape()
            if value is not None and value == currentShape():
               rate = classAction.policyAction[ tacActionType.actionSetShape ].rate
               rate.val = tacActionRateType.invalid
               rate.rateUnit = tacRateUnit.rateUnitInvalid

   def configureBandwidth( self, no, rateUnit, value ):
      context = self.pmapClassContext
      classAction = context.currentEntry()

      if tacActionType.actionSetBandwidth not in classAction.policyAction:
         context.newEditEntry( self.cmapName, bypassCheck=True )
         classAction = context.currentEntry()

      def currentBandwidth():
         rate = classAction.policyAction[ tacActionType.actionSetBandwidth ].rate
         return rate.val

      def defaultBandwidth():
         cpType = classMapCpType( gv.cliQosAclConfig, self.cmapName )
         if cpType == tacClassMapCpType.cmapCpStatic:
            cpStaticType = classMapCpStaticType( gv.cliQosAclConfig, self.cmapName )
            hwCoppStaticClass = coppStaticClassFromHwStatus( gv.qosSliceHwStatus )
            return hwCoppStaticClass[ cpStaticType ].defaultMin
         return None

      ( value, rateUnit ) = self.obtainShapeOrBandwidthValue( no, rateUnit, value )
      rate = classAction.policyAction[ tacActionType.actionSetBandwidth ].rate
      if value is not None:
         rate.val = value
      if rateUnit is not None:
         rate.rateUnit = rateUnit

      if not self.session_.startupConfig():
         if not gv.qosHwStatus.intfCoppSupported:
            value = defaultBandwidth()
            if value is not None and value == currentBandwidth():
               rate = \
                   classAction.policyAction[ tacActionType.actionSetBandwidth ].rate
               rate.val = tacActionRateType.invalid
               rate.rateUnit = tacRateUnit.rateUnitInvalid


PolicyMapClassModeCopp.addCommandClass( QosCliCopp.AbortCmd )

# ------------------------------------------------------------------------
# Set commands ( policy-map )
# ------------------------------------------------------------------------
# --------------------------------------------------------------------------------
# [ no | default ] shape ( kbps | pps ) RATE
# --------------------------------------------------------------------------------
nodeCoppPps = CliCommand.guardedKeyword( 'pps',
      "Specify Rate in packets per second", guard=guardPMapCoppRateUnitPps )

nodeCoppKbps = CliCommand.guardedKeyword( 'kbps',
      "Specify Rate in Kilo-bits per second", guard=guardPMapCoppRateUnitKbps )

class ShapeCmd( CliCommand.CliCommandClass ):
   syntax = 'shape ( kbps | pps ) RATE'
   noOrDefaultSyntax = 'shape ...'
   data = {
      'shape': CliCommand.guardedKeyword( 'shape',
         "Specify maximum rate limit", guard=coppActionShaperSupportedGuard ),
      'kbps': nodeCoppKbps,
      'pps': nodeCoppPps,
      'RATE': CliMatcher.DynamicIntegerMatcher( coppActionRateShapeAndBw,
         helpdesc='Shape value in the rate unit specified' ),
   }

   adapter = rateUnitAdapter
   handler = 'QosCliCoppHandler.doShapeCmdHandler'
   noHandler = 'QosCliCoppHandler.noShapeCmdHandler'
   defaultHandler = 'QosCliCoppHandler.defaultShapeCmdHandler'


PolicyMapClassModeCopp.addCommandClass( ShapeCmd )

# --------------------------------------------------------------------------------
# [ no | default ] class CMAP [ insert-before CLASS ]
# --------------------------------------------------------------------------------
class InsertBeforeClassCmd( CliCommand.CliCommandClass ):
   syntax = 'class CMAP [ insert-before CLASS ]'
   noOrDefaultSyntax = syntax
   data = {
      'insert-before': CliCommand.guardedKeyword( 'insert-before',
         "insert the class with a higher priority than a given class",
         guard=QosCliServicePolicy.guardClassMap ),
      'class': matcherClass,
      'CMAP': matcherClassName,
      'CLASS': matcherClassName,
   }

   handler = 'QosCliCoppHandler.doInsertBeforeClassCmd'
   noHandler = 'QosCliCoppHandler.noInsertBeforeClassCmd'
   defaultHandler = 'QosCliCoppHandler.defaultInsertBeforeClassCmd'


# --------------------------------------------------------------------------------
# [ no | default ] bandwidth ( kbps | pps ) BANDWIDTH
# --------------------------------------------------------------------------------
class BandwidthCmd( CliCommand.CliCommandClass ):
   syntax = 'bandwidth ( kbps | pps ) BANDWIDTH'
   noOrDefaultSyntax = 'bandwidth ...'
   data = {
      'bandwidth': CliCommand.guardedKeyword( 'bandwidth',
         "Specify minimum bandwidth", guard=coppActionShaperSupportedGuard ),
      'kbps': nodeCoppKbps,
      'pps': nodeCoppPps,
      'BANDWIDTH': CliMatcher.DynamicIntegerMatcher( coppActionRateShapeAndBw,
         helpdesc='Bandwidth value in the rate unit specified' ),
   }

   adapter = rateUnitAdapter
   handler = 'QosCliCoppHandler.doBandwidthCmdHandler'
   noHandler = 'QosCliCoppHandler.noBandwidthCmdHandler'
   defaultHandler = 'QosCliCoppHandler.defaultBandwidthCmdHandler'


PolicyMapClassModeCopp.addCommandClass( BandwidthCmd )

# ------------------------------------------------------------------------------
# [ no | default ] police rate RATE_VALUE [ RATE_UNIT ]
#       burst-size BURST_VALUE [ BURST_UNIT ]
# ------------------------------------------------------------------------------
nodeCoppPolice = CliCommand.guardedKeyword( 'police',
      "Configure policer parameters", guard=coppActionPolicerSupportedGuard )

matcherPoliceCirValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify lower rate' )

matcherPoliceCommittedBurstValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify burst size' )

class CoppPoliceRateCmd( CliCommand.CliCommandClass ):
   syntax = ( 'police rate RATE_VALUE [ RATE_UNIT ]'
              'burst-size BURST_VALUE [ BURST_UNIT ]' )
   noOrDefaultSyntax = 'police ...'
   data = {
      'police': nodeCoppPolice,
      'rate': CliCommand.guardedKeyword( 'rate', "Set lower rate",
         guard=QosCliServicePolicy.guardIngressPolicing ),
      'RATE_VALUE': matcherPoliceCirValue,
      'RATE_UNIT': RateExpression,
      'burst-size': CliCommand.guardedKeyword( 'burst-size',
         "Set burst-size for lower rate",
         guard=QosCliServicePolicy.guardIngressPolicing ),
      'BURST_VALUE': matcherPoliceCommittedBurstValue,
      'BURST_UNIT': BurstExpression,
   }

   handler = 'QosCliCoppHandler.doCoppPoliceRateCmd'
   noOrDefaultHandler = 'QosCliCoppHandler.noOrDefaultCoppPoliceRateCmd'


PolicyMapClassModeCopp.addCommandClass( CoppPoliceRateCmd )

# --------------------------------------------------------------------------------
# police cir CIR_VALUE [ RATE_UNIT ] bc BC_VALUE [ BURST_UNIT ]
# --------------------------------------------------------------------------------
class CoppPoliceCirCmd( CliCommand.CliCommandClass ):
   syntax = 'police cir CIR_VALUE [ RATE_UNIT ] bc BC_VALUE [ BURST_UNIT ]'
   data = {
      'police': nodeCoppPolice,
      'cir': CliCommand.guardedKeyword( 'cir', "Set committed information rate",
         guard=QosCliServicePolicy.guardIngressPolicing ),
      'CIR_VALUE': matcherPoliceCirValue,
      'RATE_UNIT': RateExpression,
      'bc': CliCommand.guardedKeyword( 'bc', "Set committed burst rate",
         guard=QosCliServicePolicy.guardIngressPolicing ),
      'BC_VALUE': matcherPoliceCommittedBurstValue,
      'BURST_UNIT': BurstExpression,

   }

   handler = 'QosCliCoppHandler.doCoppPoliceCirCmd'


PolicyMapClassModeCopp.addCommandClass( CoppPoliceCirCmd )


class PolicyMapModeCopp( QosCliServicePolicy.PolicyMapMode ):
   name = "Copp Policy Map Configuration"

   def getChildMode( self, mapType, context ):
      childMode = self.childMode( PolicyMapClassModeCopp,
                                  context=context )
      return childMode

   def onExit( self ):
      t0( 'PolicyMapModeCopp onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )


PolicyMapModeCopp.addShowCommandClass( QosCliCopp.PmapModeShowCmd )
PolicyMapModeCopp.addCommandClass( InsertBeforeClassCmd )
PolicyMapModeCopp.addCommandClass( QosCliCopp.AbortCmd )

# --------------------------------------------------------------------------------
# [ no | default ] service-policy input copp-system-policy
# --------------------------------------------------------------------------------
def _configureCpServicePolicy( mode, args ):
   coppServicePolicyIs( gv.qosInputConfig )

def _configureNoCpServicePolicy( mode, args ):
   coppServicePolicyIs( gv.qosInputConfig, no=True )

def _configureDefaultCpServicePolicy( mode, args ):
   if gv.hwEpochStatus.globalProtectionModeEnabled:
      _configureCpServicePolicy( mode, args )
   else:
      _configureNoCpServicePolicy( mode, args )


AclCli.CpConfigMode.clearHook.addExtension( _configureCpServicePolicy )

# --------------------------------------------------------------------------------
# For all the class in QosCliCopp.py having @staticmethod decorator for handler
# functions are converted as:
# [ do|no|default ] <Classname>
# --------------------------------------------------------------------------------
def doCmapModeShowCmd( mode, args ):
   if 'diff' in args:
      QosCliServicePolicy.showCMapDiff( mode )
   elif 'active' in args or 'current' in args:
      QosCliServicePolicy.showCMapActive( mode )
   else:
      QosCliServicePolicy.showCMapPending( mode )

def doMatchIpAccessGroupCoppCmd( mode, args ):
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( noOrDefault, 'ip', args[ 'GROUP' ] )

def doMatchIpv6AccessGroupCoppCmd( mode, args ):
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( noOrDefault, 'ipv6', args[ 'GROUP' ] )

def doPmapModeShowCmd( mode, args ):
   if 'diff' in args:
      QosCliServicePolicy.showPMapDiff( mode )
   elif 'active' in args or 'current' in args:
      QosCliServicePolicy.showPMapActive( mode )
   else:
      QosCliServicePolicy.showPMapCurrent( mode )

def doInsertBeforeClassCmd( mode, args ):
   mode.setClass( False, args[ 'CMAP' ], args.get( 'CLASS' ) )

def noInsertBeforeClassCmd( mode, args ):
   mode.setClass( True, args[ 'CMAP' ], args.get( 'CLASS' ) )

def defaultInsertBeforeClassCmd( mode, args ):
   mode.setClass( False, args[ 'CMAP' ],
            args.get( 'CLASS' ), default=True )

def doAbortCmd( mode, args ):
   mode.abort()

# ------------------------------------------------------------------------
# Set commands ( policy-map )
# ------------------------------------------------------------------------
# --------------------------------------------------------------------------------
# [ no | default ] shape ( kbps | pps ) RATE
# --------------------------------------------------------------------------------
def doShapeCmdHandler( mode, args ):
   PolicyMapClassModeCopp.configureShape( mode, False, args[ 'rateUnit' ],
         args.get( 'RATE' ) )

def noShapeCmdHandler( mode, args ):
   PolicyMapClassModeCopp.configureShape( mode, True, None, None )

def defaultShapeCmdHandler( mode, args ):
   PolicyMapClassModeCopp.configureShape( mode, 'default', None, None )

# --------------------------------------------------------------------------------
# [ no | default ] bandwidth ( kbps | pps ) BANDWIDTH
# --------------------------------------------------------------------------------
def doBandwidthCmdHandler( mode, args ):
   PolicyMapClassModeCopp.configureBandwidth( mode, False, args[ 'rateUnit' ],
         args.get( 'BANDWIDTH' ) )

def noBandwidthCmdHandler( mode, args ):
   PolicyMapClassModeCopp.configureBandwidth( mode, True, None, None )

def defaultBandwidthCmdHandler( mode, args ):
   PolicyMapClassModeCopp.configureBandwidth( mode, 'default', None, None )

def doCoppPoliceRateCmd( mode, args ):
   mode.configurePolicer( mode, no=False, cir=int( args[ 'RATE_VALUE' ] ),
         cirUnit=args.get( 'RATE_CIR_UNIT' ), bc=int( args[ 'BURST_VALUE' ] ),
         bcUnit=args.get( 'BURST_BC_UNIT' ), cmdVersion=2 )

def noOrDefaultCoppPoliceRateCmd( mode, args ):
   mode.configurePolicer( mode, no=True )

def doCoppPoliceCirCmd( mode, args ):
   mode.confiurePolicer( mode, no=False, cir=int( args.get( 'CIR_VALUE' ) ),
         cirUnit=args.get( 'RATE_CIR_UNIT' ), bc=int( args.get( 'BC_VALUE' ) ),
         bcUnit=args.get( 'BURST_BC_UNIT' ), cmdVersion=1 )

def setCoppServicePolicy( mode, noOrDefaultKw=None, pmapName=None ):
   if noOrDefaultKw is None:
      intfName = mode.intf.name
      if isLagPort( intfName ):
         mode.addError( CliError[ 'lagPolicingNotSupported' ] )
         return
   QosCliServicePolicy.setServicePolicy( mode, noOrDefaultKw=noOrDefaultKw,
                  mapType='control-plane', pmapName=pmapName, direction='input' )

def doServicePolicyTypeControlPlaneCmd( mode, args ):
   pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' )
   setCoppServicePolicy( mode, None, pmapName )

def noServicePolicyTypeControlPlaneCmd( mode, args ):
   pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' )
   setCoppServicePolicy( mode, True, pmapName )

# Clear counters for qos and pdp mapType
def clearPMapIntfCounters( mode, args ):
   mapTypeOptions = [ 'qos', 'pdp' ]
   mapType = None
   for opt in mapTypeOptions:
      if opt in args:
         mapType = args[ opt ]
         break
   pmapName = args.get( 'PMAPNAME' )
   # Signal QosAgent to clear counters from "qos/status"
   if pmapName is None and mapType in [ 'qos', 'pdp' ]:
      # Clear counters for all policy-maps of mapType
      if mapType == 'qos':
         gv.cliCounterConfig.clearPolicyQosCountersRequestTime = Tac.now()
      elif mapType == 'pdp':
         gv.cliCounterConfig.clearPdpCountersRequestTime = Tac.now()
   elif mapType in [ 'qos', 'pdp' ]:
      # Clear counters for a single policy-map pmapName of type mapType.
      direction = tacDirection.input
      emapType = mapTypeToEnum( mapType )
      spConfigKey = Tac.newInstance( "Qos::ServicePolicyKey", emapType,
                                     direction, pmapName )
      clearCountersTime = Tac.now()
      if spConfigKey in gv.qosInputConfig.servicePolicyConfig:
         spConfig = gv.qosInputConfig.servicePolicyConfig[ spConfigKey ]
         spConfig.clearCountersTime = clearCountersTime
      if spConfigKey in gv.qosInputProfileConfig.servicePolicyConfig:
         spConfig = gv.qosInputProfileConfig.servicePolicyConfig[ spConfigKey ]
         spConfig.clearCountersTime = clearCountersTime
   else:
      gv.cliCounterConfig.clearCoppCountersRequestTime = Tac.now()

# ------------------------------------------------------------------------------
# configure convert copp-system-policy
# -----------------------------------------------------------------------------
# This will read copp-system-policy from a non-per-port-copp config
# It then creates a new policy map with the same class maps and attaches
# the new policy map to all ports.

def assignClassActionCoppPolicer( classAction, className, cir, cirUnit, bc, bcUnit ):
   # pir and be are 0 because they are not used for copp policers
   classAction.policer = ( className, cir, bc, 0, 0 )
   classAction.policer.cirUnit = cirUnit
   classAction.policer.bcUnit = bcUnit
   action = tacActionType.actionSetDrop
   classAction.policer.redActions.newMember( action )
   classAction.policer.redActions[ action ].value = 1

def Plugin( entityManager ):

   gv.qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   gv.qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   gv.qosHwStatus = LazyMount.mount( entityManager,
                        "qos/hardware/status/global", "Qos::HwStatus", "r" )
   qosSliceHwStatusDirPath = ( "cell/" + str( Cell.cellId() ) +
                                "/qos/hardware/status/slice" )
   gv.qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   gv.qosAclHwStatus = LazyMount.mount( entityManager,
                        "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   gv.qosAclSliceHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/slice", "Tac::Dir", "ri" )
   gv.hwEpochStatus = LazyMount.mount( entityManager, "hwEpoch/status",
                                    "HwEpoch::Status", "r" )
   gv.qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   gv.cliQosAclConfig = ConfigMount.mount( entityManager, "qos/acl/input/cli",
                                        "Qos::Input::AclConfig", "w" )
   gv.intfConfigEthPhySliceDir = ConfigMount.mount( entityManager,
         "interface/config/eth/phy/slice/", "Tac::Dir", "w" )
   gv.cliCounterConfig = LazyMount.mount( entityManager, "qos/cliCounterConfig",
                                       "Qos::CounterConfig", "w" )
   gv.qosInputProfileConfig = ConfigMount.mount(
      entityManager, "qos/input/config/qosProfile",
      "Qos::Input::Config", "w" )
